﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FsSaveData.cpp,PageSampleFsSaveData}
 *
 * @brief
 *  ユーザーセーブデータ機能のサンプルプログラム
 */

/**
 * @page PageSampleFsSaveData ユーザーセーブデータ機能
 * @tableofcontents
 *
 * @brief
 *  ユーザーセーブデータ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleFsSaveData_SectionBrief 概要
 *  ここでは、ユーザーセーブデータ機能のサンプルプログラムの説明をします。
 *
 *  ファイル・ディレクトリの操作方法については、
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleFsSaveData_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FsSaveData Samples/Sources/Applications/FsSaveData @endlink 以下にあります。
 *
 * @section PageSampleFsSaveData_SectionNecessaryEnvironment 必要な環境
 *  このデモの実行には開発機が必要です。
 *
 * @section PageSampleFsSaveData_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleFsSaveData_SectionPrecaution 注意事項
 *  このデモは アプリケーションプログラムフォーマット = NCA では動作しません。 Raw または NSP で実行してください。
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *  このデモは実行される開発機に事前に 1 つ以上のユーザーアカウントを作成している必要があります。次項の実行手順を参考にしてください。
 *
 * @section PageSampleFsSaveData_SectionHowToExecute 実行手順
 *  開発機を起動し、DevMenu の Accounts メニューからユーザーアカウントを作成して下さい。
 *  その後、サンプルプログラムをビルドし、アプリケーションプログラムフォーマット = Raw または NSP に設定して実行してください。
 *
 * @section PageSampleFsSaveData_SectionDetail 解説
 *
 * @subsection PageSampleFsSaveData_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  FsSaveData.cpp
 *  @includelineno FsSaveData.cpp
 *
 * @subsection PageSampleFsSaveData_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - アカウントライブラリによるユーザー識別子の取得
 *  - EnsureSaveData 関数によるユーザーセーブデータの作成
 *  - MountSaveData 関数によるユーザーセーブデータのマウント
 *  - セーブデータに格納するファイルが無い場合は作成する
 *  - セーブデータに格納するファイルの読み込み・更新
 *  - Commit 関数によるセーブデータ更新内容のコミット
 *
 *  セーブデータの利用には事前に nmeta ファイルの記述による設定が必要です。
 *
 *  このサンプルでは同梱の FsSaveData.*.nmeta ファイルを利用しています。
 *  nmeta ファイルでは Application/UserAccountSaveDataSize に 4 MiB、Application/UserAccountSaveDataJournalSize に 1 MiB を設定しています。
 *  さらに Application/StartupUserAccount を None （規定値）とし起動時本体アカウント選択を利用しない設定をしています。
 *  この場合セーブデータを作成するため、サンプルコードのように EnsureSaveData の実行およびエラーハンドリングを行う必要があります。
 *
 *  nmeta ファイルで Application/StartupUserAccount を Required もしくは RequiredWithNetworkServiceAccountAvailable に設定した場合、
 *  起動時に選択されたアカウントのセーブデータがアプリケーション起動時にシステムによって生成されるため、
 *  EnsureSaveData の実行およびエラーハンドリングは必要ありません。
 *
 *  セーブデータ機能の詳細については、Nintendo SDK ドキュメントのファイルシステム機能のマニュアルにある「セーブデータ」の項目も併せて参照してください。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  FsSaveData_OutputExample.txt
 *
 */

#include <cstdlib>
#include <cstring>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/fs.h>
#include <nn/oe.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>


namespace {
/**
*   @brief セーブデータ用の構造体
*/
    struct SaveDataStructure
    {
        int value;
    };
} // namespace

extern "C" void nnMain()
{
    nn::Result result;

    // oe ライブラリを初期化します。
    nn::oe::Initialize();

    // アカウントライブラリを初期化します。
    nn::account::Initialize();

    // セーブデータをマウントします。
    {
        NN_LOG("Mount save data as \"save\"\n");

        // 例として、リストの先頭のユーザーの識別子を利用します。
        // 実際はユーザーによる選択操作が必要です。 (参照: アカウントガイド)
        nn::account::Uid user = nn::account::InvalidUid;
        int userCount = 0;
        result = nn::account::ListAllUsers(&userCount, &user, 1);
        NN_ASSERT(result.IsSuccess() && userCount > 0);

        // 選択したユーザーのセーブデータを作成します。
        // 既存の場合、何もせず ResultSuccess が返ります。
        result = nn::fs::EnsureSaveData(user);
        if ( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            // EnsureSaveData() について、アプリケーションは容量不足のエラーハンドリングを行う必要があります。
            // セーブデータ作成のための容量が不足している旨のメッセージは、システムが自動的にエラービューアで表示します。
            // アプリケーションは、アカウント選択操作をキャンセルして前のシーンに戻るなどの対応をする必要があります。
            NN_ABORT("Usable space not enough.\n");
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。

        // マウント名 "save" としてセーブデータをマウントします。
        result = nn::fs::MountSaveData("save", user);
        // 失敗した際は必ずアボートしてください。
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    const size_t FileSize = sizeof(SaveDataStructure);

    const char* FilePath = "save:/file";

    // ファイルの存在を確認します。
    {
        nn::fs::DirectoryEntryType directoryEntryType;
        result = nn::fs::GetEntryType(&directoryEntryType, FilePath);
        if( nn::fs::ResultPathNotFound().Includes(result) )
        {
            // 対象ファイルが存在しません。
            // ファイルを作成します。
            {
                result = nn::fs::CreateFile(FilePath, FileSize);
                if( nn::fs::ResultPathNotFound::Includes(result) )
                {
                    // 対象ファイルの親ディレクトリが存在しません。
                    // 親ディレクトリが必ず存在する場合は、このエラーハンドリングは不要です。
                }
                else if( nn::fs::ResultPathAlreadyExists::Includes(result)
                      || nn::fs::ResultTargetLocked::Includes(result) )
                {
                    // 対象ファイルが既に存在しています。
                    // ファイルが既に存在していても構わない場合は、このエラーハンドリングは不要です。
                    // エラーハンドリングしない場合、ファイルのサイズが FileSize である保証が無いことに注意してください。
                    // 必要であれば対象ファイルを削除してから再度作成してください。
                }
                else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
                {
                    // セーブデータのデータ保存領域が不足しています。
                    NN_ABORT("Usable space not enough.\n");
                    return;
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。
            }

            // 作成したファイルに初期データを書き込みます。
            {
                nn::fs::FileHandle fileHandle;

                // ファイルをオープンします。
                result = nn::fs::OpenFile(&fileHandle, FilePath, nn::fs::OpenMode_Write);
                if( nn::fs::ResultPathNotFound::Includes(result) )
                {
                    // 対象ファイルが存在しません。
                    // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
                }
                else if( nn::fs::ResultTargetLocked::Includes(result) )
                {
                    // 対象ファイルが既にオープンされています。
                    // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。

                // ファイルに書き込みます。
                SaveDataStructure data = {0};
                result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
                if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
                {
                    // セーブデータのデータ保存領域が不足しています。
                    // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
                    NN_ABORT("Usable space not enough.\n");
                }
                // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
                // これ以上のエラーハンドリングは不要です。

                nn::fs::CloseFile(fileHandle);
            }

            NN_LOG("Create initial data as %s\n", FilePath);
        }
        // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
        // これ以上のエラーハンドリングは不要です。
    }

    // ファイルのデータを読み込み、更新します。
    {
        SaveDataStructure data;

        nn::fs::FileHandle fileHandle;

        // ファイルをオープンします
        {
            result = nn::fs::OpenFile(&fileHandle, FilePath, nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
            if( nn::fs::ResultPathNotFound::Includes(result) )
            {
                // 対象ファイルが存在しません。
                // 存在するファイルしか開かない場合は、このエラーハンドリングは不要です。
            }
            else if( nn::fs::ResultTargetLocked::Includes(result) )
            {
                // 対象ファイルが既にオープンされています。
                // ファイルが既にオープンされている可能性が無い場合は、このエラーハンドリングは不要です。
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }

        // ファイルを読み込みます。
        {
            size_t readSize;

            // ファイル読み込み失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
            NN_LOG("Read %s\n", FilePath);
            (void)nn::fs::ReadFile(&readSize, fileHandle, 0, &data, sizeof(data));
            NN_ASSERT_EQUAL(readSize, sizeof(data));
        }

        NN_LOG("data.value: 0x%x\n", data.value);

        // 読み込んだデータを更新します。
        NN_LOG("Increment data.value: 0x%x -> 0x%x\n", data.value, data.value + 1);
        data.value++;

        // 更新したデータを書き込みます。
        {
            NN_LOG("Write %s\n", FilePath);
            result = nn::fs::WriteFile(fileHandle, 0, &data, sizeof(data), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
            if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
            {
                // セーブデータのデータ保存領域が不足しています。
                // ファイルをオープンする際に OpenMode_AllowAppend を指定していない場合は、このエラーハンドリングは不要です。
                NN_ABORT("Usable space not enough.\n");
            }
            // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
            // これ以上のエラーハンドリングは不要です。
        }

        NN_LOG("Close %s\n", FilePath);
        nn::fs::CloseFile(fileHandle);

        // セーブデータの更新内容をコミットします。
        // コミット失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
        NN_LOG("Commit \"save\"\n");
        (void)nn::fs::Commit("save");
    }

    // アンマウントします。
    NN_LOG("Unmount \"save\"\n\n");
    nn::fs::Unmount("save");

    return;
}
