﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @examplesource{Input.cpp,PageSampleFriendsPresence}
 *
 * @brief
 *  入力取得プログラム
 */

#include "Input.h"

#include <nn/nn_Assert.h>
#include <nn/hid/hid_DebugPad.h>

namespace input {

namespace
{
    const nn::hid::NpadIdType UseNpadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::No4,
        nn::hid::NpadId::No5,
        nn::hid::NpadId::No6,
        nn::hid::NpadId::No7,
        nn::hid::NpadId::No8,
        nn::hid::NpadId::Handheld
    };

    const int UseNpadIdCountMax = sizeof (UseNpadIds) / sizeof (UseNpadIds[0]);
}

static void ConvertDebugPadButtonsToNpadButtons(nn::hid::NpadButtonSet* out, const nn::hid::DebugPadButtonSet& in) NN_NOEXCEPT
{
    if (in.Test<nn::hid::DebugPadButton::A>())
    {
        out->Set(nn::hid::NpadButton::A::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::B>())
    {
        out->Set(nn::hid::NpadButton::B::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::X>())
    {
        out->Set(nn::hid::NpadButton::X::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Y>())
    {
        out->Set(nn::hid::NpadButton::Y::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Up>())
    {
        out->Set(nn::hid::NpadButton::Up::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Down>())
    {
        out->Set(nn::hid::NpadButton::Down::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Left>())
    {
        out->Set(nn::hid::NpadButton::Left::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Right>())
    {
        out->Set(nn::hid::NpadButton::Right::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::L>())
    {
        out->Set(nn::hid::NpadButton::L::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::R>())
    {
        out->Set(nn::hid::NpadButton::R::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::ZL>())
    {
        out->Set(nn::hid::NpadButton::ZL::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::ZR>())
    {
        out->Set(nn::hid::NpadButton::ZR::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Start>())
    {
        out->Set(nn::hid::NpadButton::Plus::Index);
    }
    if (in.Test<nn::hid::DebugPadButton::Select>())
    {
        out->Set(nn::hid::NpadButton::Minus::Index);
    }
}

void Initialize() NN_NOEXCEPT
{
    nn::hid::InitializeNpad();
    nn::hid::InitializeDebugPad();

    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(UseNpadIds, UseNpadIdCountMax);
}

void GetButtons(nn::hid::NpadButtonSet* buttons, int index) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(buttons);
    NN_ASSERT_RANGE(index, 0, SupportedNpadCount);

    buttons->Reset();

    if (index == 0)
    {
        // No1 or Handheld
        nn::hid::NpadStyleSet style1 = nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1);
        nn::hid::NpadStyleSet style2 = nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld);

        if (style1.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);

            *buttons = state.buttons;
        }
        else if (style2.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);

            *buttons = state.buttons;
        }

        if (buttons->IsAllOff())
        {
            nn::hid::DebugPadState state = {};

            nn::hid::GetDebugPadState(&state);

            ConvertDebugPadButtonsToNpadButtons(buttons, state.buttons);
        }
    }
    else
    {
        // No2 ~ No4
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(UseNpadIds[index]);

        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, UseNpadIds[index]);

            *buttons = state.buttons;
        }
    }
}

}
