﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @examplesource{GuiConsole.cpp,PageSampleFriendsPresence}
 *
 * @brief
 *  コンソールの GUI 表示プログラム
 */

#include "Console.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Color.h>
#include <cstdarg>
#include <cstdlib>
#include <mutex>

#include <nn/gfx/gfx_Enum.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

namespace console {

namespace
{
    // グラフィックシステムに割り当てるメモリのサイズ
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    // レンダリングバッファの数
    const int GraphicsBufferCount = 2;

    // テキストライターで保持できる文字の最大数
    const size_t TextWriterCharCountMax = 32 * 1024;

    // タイムスタンプ文字列の長さ
    const size_t TimestampLength = sizeof ("xxx:xx:xx:xxx: ") - 1;

    // エスケープ引数の最大数
    const int EscapeArgumentCountMax = 8;

    struct Console
    {
        char* buffer;
        char* line;
        int cursorX;
        int cursorY;
        int scrollX;
        int scrollY;
        bool isCreated;
        bool isVisible;
        bool isLooped;
        nn::util::Color4u8 color;
        nn::gfx::ViewportScissorState viewportScissorState;
        ConsoleCreateParam createParam;

        char* GetCursor() const
        {
            return &buffer[createParam.bufferWidth * cursorY + cursorX];
        }
    };

    struct Escape
    {
        char code;
        int args[EscapeArgumentCountMax];
        int num;
    };

    const ConsoleCreateParam DefaultConsoleCreateParam =
    {
        20.0f,                  // lineHeight
        1024,                   // bufferWidth
        1024,                   // bufferHeight
        1024,                   // windowWidth
        36,                     // windowHeight
        0,                      // viewX
        0,                      // viewY
        DefaultScreenWidth,     // viewWidth
        DefaultScreenHeight,    // viewHeight
        false,                  // isTimestampEnabled
        true                    // isTeeEnabled
    };

    const nn::util::Color4u8 SystemTextColor[] =
    {
        nn::util::Color4u8(  0,   0,   0), // 黒
        nn::util::Color4u8(160,   0,   0), // 赤
        nn::util::Color4u8(  0, 160,   0), // 緑
        nn::util::Color4u8(160, 160,   0), // 黄
        nn::util::Color4u8(  0,   0, 160), // 青
        nn::util::Color4u8(160,   0, 160), // マゼンタ
        nn::util::Color4u8(  0, 160, 160), // シアン
        nn::util::Color4u8(255, 255, 255)  // 白
    };
}

namespace
{
    nn::os::Mutex g_MutexForLogging(true);
    nn::os::Mutex g_MutexForColor(true);

    nn::os::Tick g_StartTick = nn::os::Tick();

    int g_ScreenWidth = DefaultScreenWidth;
    int g_ScreenHeight = DefaultScreenHeight;

    nns::gfx::GraphicsFramework g_Framework;
    int g_TextureDescriptorIndex = -1;
    int g_SamplerDescriptorIndex = -1;

    nn::gfx::util::DebugFontTextWriter g_TextWriter;
    nn::Bit8* g_TextWriterHeap = nullptr;

    nn::util::Color4f g_ScreenColor = nn::util::Color4f::Black();
    nn::util::Color4f g_DefaultTextColor = nn::util::Color4f::White();

    Console g_Console[ConsoleCount] = {};

    char g_PrintfBuffer[32 * 1024] = {};
}

static void WriteTimestamp(int index) NN_NOEXCEPT
{
    NN_ASSERT(g_MutexForLogging.IsLockedByCurrentThread());

    Console& con = g_Console[index];

    nn::TimeSpan elapsed = (nn::os::GetSystemTick() - g_StartTick).ToTimeSpan();

    int hours = static_cast<int>(elapsed.GetHours());
    elapsed -= nn::TimeSpan::FromHours(hours);

    int minutes = static_cast<int>(elapsed.GetMinutes());
    elapsed -= nn::TimeSpan::FromMinutes(minutes);

    int seconds = static_cast<int>(elapsed.GetSeconds());
    elapsed -= nn::TimeSpan::FromSeconds(seconds);

    NN_ASSERT(con.createParam.bufferWidth - g_Console[index].cursorX > TimestampLength);

    nn::util::SNPrintf(con.GetCursor(), TimestampLength + 1,
        "%03d:%02d:%02d:%03d: ", hours, minutes, seconds, static_cast<int>(elapsed.GetMilliSeconds()));

    con.cursorX += TimestampLength;
}

static void WriteNewLine(int index) NN_NOEXCEPT
{
    NN_ASSERT(g_MutexForLogging.IsLockedByCurrentThread());

    Console& con = g_Console[index];

    *con.GetCursor() = '\0';

    con.cursorX = 0;
    con.cursorY++;

    if (con.cursorY == con.createParam.bufferHeight)
    {
        con.cursorY = 0;
        con.isLooped = true;
    }

    std::memset(con.GetCursor(), 0, con.createParam.bufferWidth);

    if (con.cursorX == 0 && con.createParam.isTimestampEnabled)
    {
        WriteTimestamp(index);
    }
}

static void WriteBuffer(int index, const char* buffer) NN_NOEXCEPT
{
    NN_ASSERT(g_MutexForLogging.IsLockedByCurrentThread());

    Console& con = g_Console[index];
    const char* p = buffer;

    while (*p)
    {
        if (con.cursorX == 0 && con.createParam.isTimestampEnabled)
        {
            WriteTimestamp(index);
        }

        if (*p == '\n')
        {
            WriteNewLine(index);
        }
        else if (*p == '\r')
        {
            // 行頭復帰
            con.cursorX = con.createParam.isTimestampEnabled ? TimestampLength : 0;
        }
        else
        {
            *con.GetCursor() = *p;

            if (++con.cursorX == con.createParam.bufferWidth)
            {
                WriteNewLine(index);
            }
        }

        p++;
    }
}

static bool IsEscapeCode(char c) NN_NOEXCEPT
{
    return (c == 'A' || c == 'B' || c == 'C' || c == 'D' || c == 'E' || c == 'F' || c == 'G' || c == 'H' || c == 'J' ||
        c == 'K' || c == 'S' || c == 'T' || c == 'f' || c == 'm');
}

static int ParseEscape(Escape* escape, const char* line) NN_NOEXCEPT
{
    NN_ASSERT(*line == '\033');

    escape->code = '\0';
    escape->num = 0;

    const char* e = line;
    e++;

    if (*e++ != '[')
    {
        return -1;
    }
    if (IsEscapeCode(*e))
    {
        escape->code = *e;
        return 2;
    }

    for (int i = 0; i < EscapeArgumentCountMax; i++)
    {
        char* end = nullptr;
        long n = std::strtol(e, &end, 10);

        escape->args[escape->num++] = static_cast<int>(n);
        e = end;

        if (*e != ';')
        {
            break;
        }
        e++;
    }

    if (!IsEscapeCode(*e))
    {
        return -1;
    }

    escape->code = *e;

    return static_cast<int>(e - line);
}

static void DrawLine(int index, int curosorY, int bufferY) NN_NOEXCEPT
{
    NN_ASSERT(g_MutexForLogging.IsLockedByCurrentThread());

    Console& con = g_Console[index];

    g_TextWriter.SetCursor(0, curosorY * con.createParam.lineHeight);
    g_TextWriter.SetScale(1.0f, 1.0f);

#if 0

    g_TextWriter.SetTextColor(con.color);
    g_TextWriter.Print("%s", &con.buffer[con.createParam.bufferWidth * bufferY]);

#else

    const char* p = &con.buffer[con.createParam.bufferWidth * bufferY];

    int n = 0;

    for (int i = 0; i < con.createParam.bufferWidth; i++)
    {
        char c = p[i];

        if (c == '\0')
        {
            break;
        }
        if (c == '\033')
        {
            if (n > 0)
            {
                con.line[n] = '\0';

                g_TextWriter.SetTextColor(con.color);
                g_TextWriter.Print("%s", con.line);

                n = 0;
            }

            Escape esc = {};
            int length = ParseEscape(&esc, &p[i]);

            if (length > 0)
            {
                if (esc.code == 'm')
                {
                    if (esc.num == 0 || esc.args[0] == 0)
                    {
                        con.color = nn::util::Color4u8(g_DefaultTextColor);
                    }
                    else if (esc.num > 0 && esc.args[0] >= 30 && esc.args[0] <= 37)
                    {
                        con.color = SystemTextColor[esc.args[0] - 30];
                    }
                }

                i += length;
            }
        }
        else
        {
            con.line[n++] = c;
        }
    }

    if (n > 0)
    {
        con.line[n] = '\0';

        g_TextWriter.SetTextColor(con.color);
        g_TextWriter.Print("%s", con.line);
    }

#endif
}

static void DrawLog(int index, nn::gfx::CommandBuffer* commandBuffer) NN_NOEXCEPT
{
    NN_ASSERT(g_MutexForLogging.IsLockedByCurrentThread());

    Console& con = g_Console[index];

    int y1 = con.cursorY - con.scrollY;
    int y0 = y1 - con.createParam.windowHeight + 1;

    if (y0 < 0)
    {
        if (con.isLooped)
        {
            y0 = con.createParam.bufferHeight + y0;
        }
        else
        {
            y0 = 0;
        }
    }
    if (y1 < 0)
    {
        NN_ASSERT(con.isLooped);

        y1 = con.createParam.bufferHeight + y1;
    }

    if (y0 <= y1)
    {
        for (int y = y0; y <= y1; y++)
        {
            DrawLine(index, y - y0, y);
        }
    }
    else
    {
        for (int y = y0; y < con.createParam.bufferHeight; y++)
        {
            DrawLine(index, y - y0, y);
        }

        int written = con.createParam.bufferHeight - y0;

        for (int y = 0; y <= y1; y++)
        {
            DrawLine(index, written + y, y);
        }
    }

    g_TextWriter.Draw(commandBuffer);
}

static void MakeCommand(nns::gfx::GraphicsFramework* framework, int bufferIndex, void*) NN_NOEXCEPT
{
    framework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = framework->GetRootCommandBuffer(bufferIndex);

        nn::gfx::ColorTargetView* target = framework->GetColorTargetView();

        {
            std::lock_guard<decltype (g_MutexForColor)> lock(g_MutexForColor);

            rootCommandBuffer->ClearColor(target,
                g_ScreenColor.GetR(), g_ScreenColor.GetG(), g_ScreenColor.GetB(), g_ScreenColor.GetA(), nullptr);
        }

        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);

        std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

        for (int i = 0; i < ConsoleCount; i++)
        {
            Console& con = g_Console[i];

            if (con.isVisible)
            {
                rootCommandBuffer->SetViewportScissorState(&con.viewportScissorState);
                DrawLog(i, rootCommandBuffer);
            }
        }
    }
    framework->EndFrame(bufferIndex, true);
}

void Initialize(int screenWidth, int screenHeight) NN_NOEXCEPT
{
    g_ScreenWidth = screenWidth;
    g_ScreenHeight = screenHeight;

    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    // フレームワークの初期化
    {
        nns::gfx::GraphicsFramework::FrameworkInfo info;

        info.SetDefault();
        info.SetDisplayWidth(g_ScreenWidth);
        info.SetDisplayHeight(g_ScreenHeight);
        info.SetBufferCount(GraphicsBufferCount);
        info.SetSwapChainBufferCount(GraphicsBufferCount);

        g_Framework.Initialize(info);
    }
    // フォントの初期化
    {
        nn::gfx::util::DebugFontTextWriterInfo info;

        info.SetDefault();
        info.SetCharCountMax(TextWriterCharCountMax);
        info.SetBufferCount(GraphicsBufferCount);
        info.SetUserMemoryPoolEnabled(false);

        size_t heapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(g_Framework.GetDevice(), info);

        g_TextWriterHeap = new nn::Bit8[heapSize];

        NN_ASSERT_NOT_NULL(g_TextWriterHeap);

        g_TextWriter.Initialize(g_Framework.GetDevice(), info, g_TextWriterHeap, heapSize, nullptr, 0, 0);
    }

    g_TextureDescriptorIndex = g_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    g_SamplerDescriptorIndex = g_Framework.AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    g_TextWriter.SetDisplayWidth(g_Framework.GetDisplayWidth());
    g_TextWriter.SetDisplayHeight(g_Framework.GetDisplayHeight());
    g_TextWriter.SetTextureDescriptor(g_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), g_TextureDescriptorIndex);
    g_TextWriter.SetSamplerDescriptor(g_Framework.GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), g_SamplerDescriptorIndex);

    g_Framework.SetMakeCommandCallback(MakeCommand, nullptr);

    g_StartTick = nn::os::GetSystemTick();
}

void Finalize() NN_NOEXCEPT
{
    g_Framework.QueueFinish();

    g_TextWriter.Finalize();

    if (g_TextureDescriptorIndex != -1)
    {
        g_Framework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, g_TextureDescriptorIndex);
    }
    if (g_SamplerDescriptorIndex != -1)
    {
        g_Framework.FreeDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, g_SamplerDescriptorIndex);
    }

    if (g_TextWriterHeap)
    {
        delete[] g_TextWriterHeap;
    }

    {
        std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

        for (int i = 0; i < ConsoleCount; i++)
        {
            Console& con = g_Console[i];

            if (con.isCreated)
            {
                delete[] con.buffer;
                delete[] con.line;

                con.viewportScissorState.Finalize(g_Framework.GetDevice());
            }
        }

        std::memset(g_Console, 0, sizeof (g_Console));
    }

    g_Framework.Finalize();
}

void SetScreenColor(float r, float g, float b) NN_NOEXCEPT
{
    std::lock_guard<decltype (g_MutexForColor)> lock(g_MutexForColor);

    g_ScreenColor.SetR(r);
    g_ScreenColor.SetG(g);
    g_ScreenColor.SetB(b);
}

const ConsoleCreateParam& GetDefaultConsoleCreateParam() NN_NOEXCEPT
{
    return DefaultConsoleCreateParam;
}

void CreateConsole(int index, const ConsoleCreateParam& createParam) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    NN_ASSERT(createParam.bufferWidth > 0);
    NN_ASSERT(createParam.bufferHeight > 0);
    NN_ASSERT(createParam.windowWidth > 0);
    NN_ASSERT(createParam.windowHeight > 0);
    NN_ASSERT(createParam.windowHeight < createParam.bufferHeight);

    NN_ASSERT(createParam.viewWidth > 0);
    NN_ASSERT(createParam.viewHeight > 0);
    NN_ASSERT(!createParam.isTimestampEnabled || (createParam.isTimestampEnabled && createParam.bufferWidth > TimestampLength + 1));

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    Console& con = g_Console[index];

    NN_ASSERT(!con.isCreated);

    size_t bufferSize = static_cast<size_t>(createParam.bufferHeight * createParam.bufferWidth);

    con.buffer = new char[bufferSize];
    std::memset(con.buffer, 0, bufferSize);

    con.line = new char[createParam.bufferWidth];
    std::memset(con.line, 0, createParam.bufferWidth);

    con.color = nn::util::Color4u8(g_DefaultTextColor);

    // ビューポートとシザー矩形の初期化
    {
        nn::gfx::ViewportStateInfo viewportInfo;

        // ビューポートの原点は左上（向き：↑→）
        float viewportX = static_cast<float>(createParam.viewX);
        float viewportY = static_cast<float>(-createParam.viewY);
        float viewportWidth = static_cast<float>(g_ScreenWidth);
        float viewportHeight = static_cast<float>(g_ScreenHeight);

        viewportInfo.SetDefault();
        viewportInfo.SetOriginX(viewportX);
        viewportInfo.SetOriginY(viewportY);
        viewportInfo.SetWidth(viewportWidth);
        viewportInfo.SetHeight(viewportHeight);

        nn::gfx::ScissorStateInfo scissorInfo;

        // シザー矩形の原点は左下（向き：↑→）
        int scissorX = createParam.viewX;
        int scissorY = g_ScreenHeight - (createParam.viewY + createParam.viewHeight);
        int scissorWidth = createParam.viewWidth;
        int scissorHeight = createParam.viewHeight;

        if (scissorX < 0)
        {
            scissorWidth += scissorX;
            scissorX = 0;

            if (scissorWidth < 0)
            {
                scissorWidth = 0;
            }
        }
        if (scissorY < 0)
        {
            scissorY = 0;
        }

        scissorInfo.SetDefault();
        scissorInfo.SetOriginX(scissorX);
        scissorInfo.SetOriginY(scissorY);
        scissorInfo.SetWidth(scissorWidth);
        scissorInfo.SetHeight(scissorHeight);

        nn::gfx::ViewportScissorState::InfoType viewportScissorInfo;

        viewportScissorInfo.SetDefault();
        viewportScissorInfo.SetScissorEnabled(true);
        viewportScissorInfo.SetViewportStateInfoArray(&viewportInfo, 1);
        viewportScissorInfo.SetScissorStateInfoArray(&scissorInfo, 1);

        con.viewportScissorState.Initialize(g_Framework.GetDevice(), viewportScissorInfo);
    }

    con.createParam = createParam;

    con.isVisible = true;
    con.isCreated = true;
}

void SetVisible(int index, bool isVisible) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    g_Console[index].isVisible = isVisible;
}

bool GetVisible(int index) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    return g_Console[index].isVisible;
}

void ProcessFrame() NN_NOEXCEPT
{
    g_Framework.ProcessFrame();
}

void Clear(int index) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    Console& con = g_Console[index];

    if (!con.isCreated)
    {
        return;
    }

    std::memset(con.buffer, 0, static_cast<size_t>(con.createParam.bufferHeight * con.createParam.bufferWidth));

    con.cursorX = 0;
    con.cursorY = 0;
    con.scrollX = 0;
    con.scrollY = 0;
    con.isLooped = false;
}

void Printf(int index, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    Console& con = g_Console[index];

    std::va_list args;

    va_start(args, format);
    {
        nn::util::VSNPrintf(g_PrintfBuffer, sizeof (g_PrintfBuffer), format, args);

        if (con.isCreated)
        {
            WriteBuffer(index, g_PrintfBuffer);
        }
        if (con.createParam.isTeeEnabled)
        {
            NN_VLOG(format, args);
        }
    }
    va_end(args);
}

void Scroll(int index, int dy) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    return SetScrollPosition(index, g_Console[index].scrollY + dy);
}

void SetScrollPosition(int index, int y) NN_NOEXCEPT
{
    NN_ASSERT_RANGE(index, 0, ConsoleCount);

    std::lock_guard<decltype (g_MutexForLogging)> lock(g_MutexForLogging);

    Console& con = g_Console[index];

    if (!con.isCreated)
    {
        return;
    }

    int scrollMax = (con.isLooped ? con.createParam.bufferHeight : con.cursorY) - con.createParam.windowHeight;

    if (scrollMax < 0)
    {
        scrollMax = 0;
    }

    con.scrollY = y;

    if (con.scrollY < 0)
    {
        con.scrollY = 0;
    }
    else if (con.scrollY > scrollMax)
    {
        con.scrollY = scrollMax;
    }
}

}
