﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{FontllSaveToJpeg.cpp,PageSampleFontllSaveToJpeg}
 *
 * @brief
 *  Fontll ライブラリを用いてグリフ画像を生成するサンプルプログラム
 *
 */

/**
 * @page PageSampleFontllSaveToJpeg FontllSaveToJpeg
 * @tableofcontents
 *
 * @image html Applications\FontllSaveToJpeg\fontllsavetojpeg.png
 *
 * @brief
 *  Fontll ライブラリを用いてグリフ(文字)画像を生成するサンプルプログラムです。
 *
 * @section PageSampleFontllSaveToJpeg_SectionBrief 概要
 *  ここでは、FontllSaveToJpeg の説明をします。
 *
 * @section PageSampleFontllSaveToJpeg_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/FontllSaveToJpeg Samples/Sources/Applications/FontllSaveToJpeg @endlink 以下にあります。
 *
 * @section PageSampleFontllSaveToJpeg_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは、ホストPCファイルシステム(nn::fs::MountHost)を利用して Jpegファイル を保存します。
 *  そのため、開発機にインストールし単体起動するなど、ホストPCファイルシステムが利用できない環境では実行ができません。
 *
 * @section PageSampleFontllSaveToJpeg_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleFontllSaveToJpeg_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleFontllSaveToJpeg_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *  実行が完了すると、サンプルの実行ファイルと同じフォルダ（Samples/Sources/Applications/FontllSaveToJpeg/Binaries 以下）に、グリフ画像が出力されています。
 *
 * @section PageSampleFontllSaveToJpeg_SectionDetail 解説
 *  フォントデータを読み込んでグリフ画像を生成し、Jpeg 形式に変換して保存する流れになっています。
 */

#include <malloc.h>
#include <cstring>
#include <memory>
#include <algorithm>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>
#include <nn/image/image_JpegEncoder.h>

#include "HostExecutableFilepath.h"

namespace {

    static const int TextureHeight = 256;
    static const int TextureWidth  = 256;
    static const int TextureSize   = TextureHeight * TextureWidth;
    static const int FontSize      = 192;

    static const uint8_t BackgroundColor = 128;
    static const uint8_t BaselingColor = 255;
    static const uint8_t ToplineColor = 255;

    // スケーラブルフォントエンジンに読み込む、ttf ファイル
    static const bool       IsInputFontTTF    = false;
    static const char*const InputFontFileName = "nintendo_NTLG-DB_002.bfttf";

    static float g_AscentRatio = 1.0f;
    static char  g_MyFontNameBuffer[nn::fontll::FontNameLengthMax];
}

//------------------------------------------------------------------------------
// グリフ画像を作成します
static void CreateGlyphImage(uint8_t* pDstBuff, size_t dstBuffSize, nn::fontll::ScalableFontEngine* pFontEngine, uint32_t charCode)
{
    NN_ASSERT(dstBuffSize >= TextureSize);
    NN_UNUSED(dstBuffSize);

    // グリフビットマップを取得します。
    // エンジン内部ワークにビットマップを生成しています。
    // FormatGrayMap8 だけが指定可能です。
    nn::fontll::GlyphMap* pMap = pFontEngine->AcquireGlyphmap(charCode, nn::fontll::FormatGrayMap8);
    NN_ASSERT(pMap != NULL);

    // 表示確認用に グリフビットマップ を テクスチャ にコピーします。
    {
        // 全体をクリア
        memset(pDstBuff, BackgroundColor, TextureSize);

        //const uint32_t cache_flush_size = pMap->width + 32;
        const int font_ascent     = static_cast<int>(g_AscentRatio * FontSize + 0.5f);
        const int offsetFromTop   = std::max(font_ascent - pMap->hiY, 0);
        const int render_origin_y = offsetFromTop;
        const int render_origin_x = static_cast<int>((TextureWidth - pMap->width) * 0.5 + pMap->loX);

        // ベースラインと、文字のトップにラインを描画する
        memset(&pDstBuff[(render_origin_y) * TextureWidth], BaselingColor, TextureWidth);
        memset(&pDstBuff[(font_ascent)     * TextureWidth], ToplineColor,  TextureWidth);

        // エンジンが生成したグリフビットマップをテクスチャにコピーする
        for (int32_t l = 0; l < pMap->height; l++)
        {
            uint8_t* line_start = &pDstBuff[(l + render_origin_y) * TextureWidth + render_origin_x];
            memcpy(line_start, &pMap->bits[l * pMap->width], pMap->width);
        }
    }

    // エンジンのグリフ情報を破棄します。
    pFontEngine->ReleasesGlyph(pMap);
}

//------------------------------------------------------------------------------
// Jpeg ファイルを保存します
static void SaveAsJpeg(const char* pFilePath, int cx, int cy, void* pImage, size_t imageSize)
{
    NN_UNUSED(imageSize);
    static const size_t JpegBufSize = 1024 * 1024;
    void* jpegBuf = malloc(JpegBufSize);
    size_t jpegSize = 0;


    // Jpeg エンコードして jpegBuf に格納
    {
        nn::image::JpegStatus result;
        nn::image::JpegEncoder encoder;
        nn::image::Dimension dimension;
        dimension.width  = cx;
        dimension.height = cy;

        encoder.SetPixelData(pImage, dimension, cx);
        encoder.SetQuality(100);
        encoder.SetSamplingRatio(nn::image::JpegSamplingRatio_444);

        result = encoder.Analyze();
        NN_ASSERT(result == nn::image::JpegStatus_Ok);

        size_t workSize = encoder.GetAnalyzedWorkBufferSize();
        void* workBuf = malloc(workSize);

        result = encoder.Encode(&jpegSize, jpegBuf, JpegBufSize, workBuf, workSize);
        NN_ASSERT(result == nn::image::JpegStatus_Ok);

        free(workBuf);
    }

    // Jpeg エンコードしたものをファイルに書き込む
    {
        nn::Result result;
        nn::fs::FileHandle file;

        result = nn::fs::DeleteFile(pFilePath);
        NN_ASSERT(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result));

        result = nn::fs::CreateFile(pFilePath, static_cast<int64_t>(jpegSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::OpenFile(&file, pFilePath, nn::fs::OpenMode::OpenMode_Write);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::WriteFile(file, 0, jpegBuf, jpegSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::fs::CloseFile(file);
    }

    free(jpegBuf);
}

//------------------------------------------------------------------------------
// グリフ画像をJpegファイルとして保存します。
static void SaveGlyphImageAsJpeg(uint8_t* pGlyphBuff, const char* pFilePath)
{
    // レンダリング結果をビットマップにファイル出力します。
    const size_t sizeBuff = TextureWidth * TextureHeight * sizeof(uint32_t);
    uint32_t* pBmpBuff = static_cast<uint32_t*>(malloc(sizeBuff));

    for (int y = 0; y < TextureHeight; y++)
    {
        for (int x = 0; x < TextureWidth; x++)
        {
            uint8_t c = pGlyphBuff[y * TextureWidth + x];
            uint32_t color = c << 24 | c << 16 | c << 8 | c;
            pBmpBuff[y * TextureWidth + x] = color;
        }
    }

    SaveAsJpeg(pFilePath, TextureWidth, TextureHeight, pBmpBuff, sizeBuff);

    free(pBmpBuff);
}
//------------------------------------------------------------------------------
static void* Allocate(size_t size)
{
    return malloc(size);
}
static void Free(void* ptr, size_t size)
{
    NN_UNUSED(size);
    free(ptr);
}

extern "C" void nnMain()
{
    static const size_t MaxPathLength = 512;
    // ファイルシステムを初期化
    {
        nn::Result result;
        nn::fs::SetAllocator(Allocate, Free);

        // 注意：正式なファイルアクセス方法が整備されるまでのワークアラウンドです。
        // 第一引数から、実行ファイルの host 上での位置を取得します。
        char contentPath[260];
        char* pExeFilePath = nn::os::GetHostArgv()[ 0 ];
        char* pEnd = strrchr( pExeFilePath, '\\' );
        NN_ASSERT_NOT_NULL( pEnd );

        // 実行ファイルからの相対パスで、リソースフォルダを定義します。
        size_t length = pEnd - pExeFilePath + 1;
        strncpy( contentPath, pExeFilePath, length );
        *(contentPath + length) = '\0';
        result = nn::fs::MountHost("out", contentPath);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_UNUSED(result);

        strcpy( contentPath + length, "..\\Contents" );
        result = nn::fs::MountHost("contents", contentPath);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        NN_UNUSED(result);
    }

    nn::fontll::ScalableFontEngine fontEngine;

    // エンジンにワークメモリを指定して初期化します。
    const size_t workSize = 10 * 1024 * 1024;
    void* pWorkBuf = malloc(workSize);
    fontEngine.Initialize(pWorkBuf, workSize);

    // フォントを読み込んでエンジンに設定します。
    void* pFontData = NULL;
    {
        char fontFilepath[MaxPathLength] = {};
        strcat(fontFilepath, "contents:/");
        strcat(fontFilepath, InputFontFileName);

        nn::fs::FileHandle file;
        nn::Result openResult = nn::fs::OpenFile(&file, fontFilepath, nn::fs::OpenMode::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(openResult);

        int64_t fileSize = 0;
        nn::Result sizeResult = nn::fs::GetFileSize(&fileSize, file);
        NN_ABORT_UNLESS_RESULT_SUCCESS(sizeResult);

        pFontData = malloc(static_cast<size_t>(fileSize));
        NN_ASSERT(pFontData != NULL);

        nn::Result readResult = nn::fs::ReadFile(file, 0, pFontData, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(readResult);

        nn::fs::CloseFile(file);

        // ttf形式ではない場合( bfttf形式 の場合)は、デコード処理を行います。
        void* pTTFFontData = (!IsInputFontTTF) ? nn::fontll::ScalableFontEngineHelper::Decode(pFontData, static_cast<uint32_t>(fileSize)) : pFontData;

        // エンジンにフォントを指定します。
        // LoadFontに渡したデータはエンジンから参照し続けられるので、使用中は解放してはいけません。
        fontEngine.LoadFont(g_MyFontNameBuffer, pTTFFontData, 0, nn::fontll::FontNameLengthMax);
        fontEngine.SetFont(g_MyFontNameBuffer);
    }

    // フォントサイズを設定します。
    fontEngine.SetScale(FontSize << 16, 0, 0, FontSize << 16);

    // 縁取り設定(縁取り描画を行わない場合はコメントアウトします)
    fontEngine.SetOutlineWidth(5);
    fontEngine.SetFlags(nn::fontll::ScalableFontEngine::Flags_OutlinedUnFilled);

    // グリフの配置用にフォント情報を取得して必要な情報を記憶しておきます。
    {
        nn::fontll::Metrics metrics;
        fontEngine.GetFontMetrics(&metrics);

        g_AscentRatio = static_cast<float>(metrics.os2WinAscent) / metrics.metricsResolution;
    }

    // グリフ画像を生成し、ビットマップファイルに保存します。
    for(uint32_t charIdx = 0; charIdx <= 'z' - 'a'; charIdx++)
    {
        uint8_t* pRenderingResultBuff = static_cast<uint8_t*>(malloc(TextureSize));
        uint32_t charCode = 'a' + charIdx;
        char charName[] = "a";
        charName[0] += static_cast<char>(charIdx);
        NN_LOG("Printing '%s'\n", charName);

        CreateGlyphImage(pRenderingResultBuff, TextureSize, &fontEngine, charCode);

        char resultBmpPath[MaxPathLength] = {};
        strcat(resultBmpPath, "out:/rendering_result_lower_");
        strcat(resultBmpPath, charName);
        strcat(resultBmpPath, ".jpg");
        SaveGlyphImageAsJpeg(pRenderingResultBuff, resultBmpPath);

        free(pRenderingResultBuff);
    }
    for(uint32_t charIdx = 0; charIdx <= 'Z' - 'A'; charIdx++)
    {
        uint8_t* pRenderingResultBuff = static_cast<uint8_t*>(malloc(TextureSize));
        uint32_t charCode = 'A' + charIdx;
        char charName[] = "A";
        charName[0] += static_cast<char>(charIdx);
        NN_LOG("Printing '%s'\n", charName);

        CreateGlyphImage(pRenderingResultBuff, TextureSize, &fontEngine, charCode);

        char resultBmpPath[MaxPathLength] = {};
        strcat(resultBmpPath, "out:/rendering_result_upper_");
        strcat(resultBmpPath, charName);
        strcat(resultBmpPath, ".jpg");
        SaveGlyphImageAsJpeg(pRenderingResultBuff, resultBmpPath);

        free(pRenderingResultBuff);
    }

    free(pFontData);
    free(pWorkBuf);

    nn::fs::Unmount("out");
    nn::fs::Unmount("contents");
}

