﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/font.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include "FontCommon.h"

//------------------------------------------------------------------------------
//
// ワードラッピング関連処理です。
//
// nn::font::WordWrapping クラスを用いて、行の表示幅に応じて文字列の適切な
// 位置に改行コードを挿入します。
//
//------------------------------------------------------------------------------

const nn::util::Unorm8x4 Red = {{255, 64, 64, 255}};

enum Tag
{
    Tag_Begin = 0x0001, // 強調表示の開始
    Tag_End   = 0x0002  // 強調表示の終了
};

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeWordWrappingText();
void InitializeWordWrapping(int screenHeight, int fontHeight);

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeWordWrapping();

//---------------------------------------------------------------------------
// 更新
//---------------------------------------------------------------------------
void CalcWordWrapping(nn::font::DispStringBuffer& dispStringBuffer, nn::font::Font& font, int screenWidth);

//---------------------------------------------------------------------------
// タグで囲まれた文章を強調表示するタグプロセッサです。
//---------------------------------------------------------------------------
class MyTagProcessor : public nn::font::WideTagProcessor
{
public:
    // コンストラクタです。
    MyTagProcessor()
        : m_DefaultSetting()
    {
    }

    // デストラクタです。
    virtual ~MyTagProcessor()
    {
    }

    // 範囲計算時に呼ばれるメソッドです。
    virtual Operation CalcRect(nn::font::Rectangle* pRect, uint32_t code, ContextType* pContext)
    {
        ProcessTags(pContext->writer, code);
        return nn::font::WideTagProcessor::CalculateRect(pRect, pContext, code);
    }

    // 描画時に呼ばれるメソッドです。
    virtual Operation Process(uint32_t code, ContextType* pContext)
    {
        ProcessTags(pContext->writer, code);
        return nn::font::WideTagProcessor::Process(code, pContext);
    }

    // TextWriter の初期状態を保存します。
    void SaveDefaultSetting(const nn::font::WideTextWriter* pWriter)
    {
        m_DefaultSetting = *pWriter;
    }

private:
    // タグを処理します。
    void ProcessTags(nn::font::WideTextWriter* pWriter, uint32_t code)
    {
        switch(code)
        {
        case Tag_Begin:
            // タグの開始時に文字の大きさと色を変更します。
            pWriter->SetScale(2.0f, 2.0f);
            pWriter->SetTextColor(Red);
            break;
        case Tag_End:
            // タグの終了時に元の設定に戻します。
            pWriter->SetScale(m_DefaultSetting.GetScaleX(), m_DefaultSetting.GetScaleY());
            pWriter->SetTextColor(m_DefaultSetting.GetTextColor());
            break;
        default:
            break;
        }
    }

    nn::font::WideTextWriter m_DefaultSetting; // 初期状態を保存します。
};
