﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/font.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/font/font_ScalableFont.h>
#include <nn/fontll/fontll_ScalableFontEngine.h>

#include "FontScalableFont.h"

//------------------------------------------------------------------------------
//   定数定義
//------------------------------------------------------------------------------

// スケーラブルフォント
static const int ScalableFontCount = 2; // スケーラブルフォントの数
static const int ScalableFontFileCount = 3; // フォントファイルの数

// UseExternalMemoryPoolForTextureCache を true にすると、外部の MemoryPool にまとめて内部テクスチャを生成する動作を確認できます。
// テクスチャキャシュの数が多い場合、生成する MemoryPool の数が少なく抑えられるため初期化処理コストが軽減されます。
static const bool                  UseExternalMemoryPoolForTextureCache = false;
static nn::gfx::MemoryPool         g_ExternalMemoryPool;
static void*                       g_pExternalMemoryPoolBuffer = nullptr;

static const char* const DemoScalableFontFileName[ScalableFontFileCount] =
{
    "Contents:/IconFont-Regular.ttf",
    "Contents:/nintendo_NTLG-DB_002.bfttf",
    "Contents:/NotoSerif-Regular.ttf",
};

//------------------------------------------------------------------------------
//   変数定義
//------------------------------------------------------------------------------

// スケーラブルフォント用テクスチャキャッシュ
static nn::font::ScalableFont g_ScFont[ScalableFontCount];
static nn::font::TextureCache* g_pTextureCache;
static void* g_pScalableFontBinary[ScalableFontFileCount];
static void* g_pScalableFontRawResource[ScalableFontFileCount];

//------------------------------------------------------------------------------
// スケーラブルフォントに渡すアロケータ
//------------------------------------------------------------------------------
static void* AllocateForScalableFont(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

//------------------------------------------------------------------------------
// スケーラブルフォントに渡すアロケータ
//------------------------------------------------------------------------------
static void DeallocateForScalableFont(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
//   テクスチャキャッシュの初期化
//------------------------------------------------------------------------------
void InitializeTextureCashe()
{
    size_t fontBinarySize[ScalableFontFileCount];
    for (int i = 0; i < ScalableFontFileCount; i++)
    {
        nn::Result result;
        nn::fs::FileHandle fileHandle;
        int64_t fileSize;

        result = nn::fs::OpenFile(&fileHandle, DemoScalableFontFileName[i], nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::GetFileSize(&fileSize, fileHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        void* pFontFileImage = AllocateFromApplicationHeap(
            static_cast<size_t>(fileSize),
            nn::font::ResourceAlignment);
        g_pScalableFontRawResource[i] = pFontFileImage;

        size_t readSize;
        result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        fontBinarySize[i] = static_cast<size_t>(fileSize);
        g_pScalableFontBinary[i] = pFontFileImage;

        nn::fs::CloseFile(fileHandle);
    }
    NN_ASSERT_NOT_NULL(g_pScalableFontBinary);

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    //-----------------------------------------
    // テクスチャキャッシュの初期化
    {
        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(AllocateForScalableFont(sizeof(nn::font::TextureCache::InitializeArg), 4, NULL));
        pArg->SetDefault();
        pArg->allocateFunction = AllocateForScalableFont;
        pArg->textureCacheWidth = 1024;
        pArg->textureCacheHeight = 1024;
        // g_pScalableFontBinary[0]～[1]は複数のフォントを1つのFontFaceで扱える方法で登録しています。
        // g_pScalableFontBinary[2]は別FontFaceで登録しています。
        pArg->pFontDatas[0][0] = g_pScalableFontBinary[0]; // FontFace 0 番の中の、1 番目に優先されるフォント
        pArg->pFontDatas[0][1] = g_pScalableFontBinary[1]; // FontFace 0 番の中の、2 番目に優先されるフォント
        pArg->pFontDatas[1][0] = g_pScalableFontBinary[2]; // FontFace 1 番の中の、1 番目に優先されるフォント
        pArg->pFontDataSizes[0][0] = fontBinarySize[0];
        pArg->pFontDataSizes[0][1] = fontBinarySize[1];
        pArg->pFontDataSizes[1][0] = fontBinarySize[2];
        pArg->pFontDataTypes[0][0] = nn::font::TextureCache::FontDataType_Ttf;
        pArg->pFontDataTypes[0][1] = nn::font::TextureCache::FontDataType_Bfttf;
        pArg->pFontDataTypes[1][0] = nn::font::TextureCache::FontDataType_Ttf;
        pArg->fontCount = 2; // FontFace の数
        pArg->fontListCount[0] = 2; // FontFace 0 番に連なるフォント数
        pArg->fontListCount[1] = 1; // FontFace 1 番に連なるフォント数

        // FontFace 0 番の中の、1 番目に優先されるフォントが扱う文字コードの範囲を、
        // 英字(a～z および A～Z)だけに設定します。
        pArg->charCodeRangeCount[0][0] = 2;
        pArg->charCodeRangeFirst[0][0][0] = 'a';
        pArg->charCodeRangeLast[0][0][0] = 'z';
        pArg->charCodeRangeFirst[0][0][1] = 'A';
        pArg->charCodeRangeLast[0][0][1] = 'Z';

        // フォントのフチ描画を有効化します
        pArg->pBorderWidths[0][0] = 4;
        pArg->pBorderWidths[0][1] = 4;
        pArg->pBorderWidths[1][0] = 4;

        // テクスチャキャッシュの初期化
        {
            g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();
            NN_ASSERT_NOT_NULL(g_pTextureCache);

            if (NN_STATIC_CONDITION(UseExternalMemoryPoolForTextureCache))
            {
                // 外部メモリプールを作って初期化する
                size_t ExternalMemoryPoolSize = 0;
                {
                    // メモリプール属性を設定します。
                    nn::gfx::MemoryPool::InfoType info;
                    nn::font::TextureCache::SetMemoryPoolInfo(&info);

                    // 内部テクスチャを確保するのに（最小限必要となる）メモリプールサイズを計算します。
                    const size_t MemPoolSizeForTexture = nn::font::TextureCache::CalculateMemoryPoolSize(pDevice, 1024, 1024);

                    const size_t ExtreBufferSize = 1024 * 64;
                    ExternalMemoryPoolSize = nn::util::align_up(
                        MemPoolSizeForTexture + ExtreBufferSize, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, info));

                    g_pExternalMemoryPoolBuffer = AllocateForScalableFont(
                        ExternalMemoryPoolSize, nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, info), NULL);
                    NN_ASSERT_NOT_NULL(g_pExternalMemoryPoolBuffer);
                    info.SetPoolMemory(g_pExternalMemoryPoolBuffer, ExternalMemoryPoolSize);

                    g_ExternalMemoryPool.Initialize(pDevice, info);
                }

                // 中途半端なオフセット値の場合テクスチャを配置するのに必要なアライメントで調整する必要があります。
                ptrdiff_t externalMemoryPoolOffset = 3;
                externalMemoryPoolOffset = nn::util::align_up(
                    externalMemoryPoolOffset, nn::font::TextureCache::CalculateMemoryPoolAlignment(pDevice, 1024, 1024));

                g_pTextureCache->Initialize(pDevice, *pArg, &g_ExternalMemoryPool, externalMemoryPoolOffset, ExternalMemoryPoolSize);
            }
            else
            {
                g_pTextureCache->Initialize(pDevice, *pArg);
            }

            g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
        }

        DeallocateForScalableFont(pArg, NULL);
    }
}

//------------------------------------------------------------------------------
//   フォントの初期化
//------------------------------------------------------------------------------
void InitializeFont()
{
    for (int i = 0; i < ScalableFontCount; i++)
    {
        nn::font::ScalableFont::InitializeArg arg;
        arg.SetDefault();
        arg.pTextureCache = g_pTextureCache;
        arg.fontSize = 30;
        arg.fontFace = static_cast<uint16_t>(i);
        // 複数のフォントをそれぞれ初期化する
        // デフォルトでは、プロットされていない文字は常に代替文字(デフォルトでは'?')
        // に置き換えられますが、この状態ではプロットされていない文字のGetCharWidth
        // を呼び出したときに常に代替文字の幅を返すため、「プロットされていない文字は、
        // それがフォント内になければ代替文字にし、フォントにあれば元々表示するはず
        // だった文字と同じ幅の空白として扱う」設定にするのがお勧めです。
        arg.isAlternateCharSpaceWithOriginalWidthForNotReadyChar = true;
        arg.alternateChar = '?';

        g_ScFont[i].Initialize(arg);

        // 使用する文字をフォントエンジンに登録します。
        // ここでは ASCII 文字をすべて登録しています。
        for (uint16_t j = 0; j < 255; j++)
        {
            g_ScFont[i].RegisterGlyph(j);
        }

        // 日本語を登録します。
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("漢字「本日は晴天なり」"));
        // 特殊な文字をいくつか登録します。
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("ABCDE")); // g_pScalableFontBinary[0] に含まれる文字
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("あいうえお")); // g_pScalableFontBinary[0] には含まれず、[1] に含まれる文字
        g_ScFont[i].RegisterGlyph(0x2c65); // g_pScalableFontBinary[0]、[1] には含まれず、[2] に含まれる文字
    }
}

//------------------------------------------------------------------------------
//   初期化
//------------------------------------------------------------------------------
void InitializeScalableFont()
{
    //-----------------------------------------
    // テクスチャキャッシュの初期化
    InitializeTextureCashe();

    //-----------------------------------------
    // フォントの初期化
    InitializeFont();

    // CPU 側のメモリへグリフのレンダリングを行います。
    g_pTextureCache->UpdateTextureCache();

    // CPU 側のメモリへレンダリングしたグリフイメージをテクスチャとして使用できるようにします。
    g_pTextureCache->CompleteTextureCache();
}

//------------------------------------------------------------------------------
//   解放
//------------------------------------------------------------------------------
void FinalizeScalableFont()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    for (int i = 0; i < ScalableFontCount; i++)
    {
        g_ScFont[i].Finalize(pDevice);
    }
    g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    g_pTextureCache->Finalize(pDevice, DeallocateForScalableFont, NULL);
    DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
    g_pTextureCache = NULL;

    for (int i = 0; i < ScalableFontFileCount; i++)
    {
        DeallocateApplicationHeap(g_pScalableFontRawResource[i]);
    }

    if (NN_STATIC_CONDITION(UseExternalMemoryPoolForTextureCache))
    {
        g_ExternalMemoryPool.Finalize(pDevice);
        DeallocateForScalableFont(g_pExternalMemoryPoolBuffer, NULL);
    }
}

//------------------------------------------------------------------------------
//   フォントの設定
//------------------------------------------------------------------------------
void SetScalableFont(nn::font::WideTextWriter* pTextWriter)
{
    pTextWriter->SetFont(&g_ScFont[0]);
}

//------------------------------------------------------------------------------
//   バッファ生成
//------------------------------------------------------------------------------
void PrintScalableFont(nn::font::WideTextWriter* pTextWriter)
{
    // 各FontFaceで描画する
    pTextWriter->SetCursorY(350.0f);
    for (int i = 0; i < ScalableFontCount; i++)
    {
        pTextWriter->SetCursorX(50.0f);
        pTextWriter->SetFont(&g_ScFont[1]);
        pTextWriter->Printf(FONT_SAMPLE_LITERAL("Font face %d: "), i);
        pTextWriter->SetFont(&g_ScFont[i]);
        pTextWriter->Print(FONT_SAMPLE_LITERAL("ABCDEあいうえお\x2c65\n"));
    }
}

//------------------------------------------------------------------------------
//   フォントリストの優先順位の変更
//------------------------------------------------------------------------------
void ChangeFontOrder()
{
    // フォントフェース 0 に登録された 2 つのフォントの優先順位を変更します。
    // 例えば言語設定に応じて本体内蔵フォントの構築順を変更したい場合などには
    // この処理を参考にしてください。
    {
        uint32_t newOrder[2] = { 1, 0 }; // {0,1}から{1,0}に変更
        g_pTextureCache->ChangeFontListOrder(0, newOrder);
    }

    // グリフを再登録する
    for (int i = 0; i < ScalableFontCount; i++)
    {
        // 使用する文字をフォントエンジンに登録します。
        // ここでは ASCII 文字をすべて登録しています。
        for (uint16_t j = 0; j < 255; j++)
        {
            g_ScFont[i].RegisterGlyph(j);
        }

        // 日本語を登録します。
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("漢字「本日は晴天なり」"));
        // 特殊な文字をいくつか登録します。
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("ABCDE")); // g_pScalableFontBinary[0] に含まれる文字
        g_ScFont[i].RegisterGlyphs(FONT_SAMPLE_LITERAL("あいうえお")); // g_pScalableFontBinary[0] には含まれず、[1] に含まれる文字
        g_ScFont[i].RegisterGlyph(0x2c65); // g_pScalableFontBinary[0]、[1] には含まれず、[2] に含まれる文字
    }

    // テクスチャキャッシュを更新する
    g_pTextureCache->UpdateTextureCache();
    g_pTextureCache->CompleteTextureCache();
}
