﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{ErrServiceStatus/ErrServiceStatus.cpp,PageSampleErrServiceStatusInfo}
 *
 * @brief
 *  ERR ライブラリを使用しサービスステータス情報を取得するサンプルプログラム
 */

/**
 * @page PageSampleErrServiceStatusInfo ERR ライブラリの基本操作
 * @tableofcontents
 *
 * @brief
 *  ERR ライブラリを使用したサービスステータス情報取得サンプルプログラムの解説です。
 *
 * @section PageSampleErrServiceStatusInfo_SectionBrief 概要
 *  ERR ライブラリを使用して、サービスステータス情報の取得を行うサンプルです。
 *
 * @section PageSampleErrServiceStatusInfo_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/ErrServiceStatus
 *  Samples/Sources/Applications/ErrServiceStatus @endlink 以下にあります。
 *
 * @section PageSampleErrServiceStatusInfo_SectionNecessaryEnvironment 必要な環境
 *  ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleErrServiceStatusInfo_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleErrServiceStatusInfo_SectionPrecaution 注意事項
 *  Windows 環境で実行する場合は、下記のように仮想デバイス証明書を配置しておく必要があります。
 *    - 実行ファイル.exe
 *    - sdcard/
 *        - device_cert.p12
 *        - device_cert.key
 *
 * @section PageSampleErrServiceStatusInfo_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleErrServiceStatusInfo_SectionDetail 解説
 *
 * このサンプルプログラムの処理の流れは以下の通りです。
 *
 * - ネットワーク環境の初期化
 *
 * - ダウンロード時のキャンセル動作の確認
 * - service-status.json ファイルのダウンロード
 * - ダウンロードした service-status.json ファイルの情報取得
 *     - 任意の ApplicationId と ServerCode を指定し、該当する最新のサービスステータスを抽出する。
 *     - 抽出されたサービスステータス情報を表示する。
 *
 */


#include <curl/curl.h>
#include <nn/nn_Log.h>
#include <nn/nifm.h>
#include <nn/socket.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Uuid.h>
#include <nn/util/util_UuidApi.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_ResEndian.h>

#include <../Include/nn/err/err_ServiceStatusAccessorApi.h>
#include <../Include/nn/err/err_ServiceStatusDownloaderApi.h>

namespace
{
    const size_t ThreadStackSize = 1024 * 8;  // スレッド操作スレッドのスタックサイズ
    const nn::TimeSpanType Timespan = nn::TimeSpanType::FromMilliSeconds(10);
    const int MaxFileSize = 921600;

    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
    NN_ALIGNAS(4096) uint8_t g_Stack[ThreadStackSize];

    nn::os::ThreadType g_ThreadStack;

    nn::err::ServiceStatusDownloader g_ServiceStatusDownloader;

    const nn::ApplicationId ApplicationId1 = { 0x01000000000020c5 };
    const nn::ApplicationId ApplicationId2 = { 0x01000000000020c6 };
    const char* ServerCode1 = "S0002";
    const char* ServerCode2 = "S0003";
    const char* ServerCode3 = "S0006";
}


bool InitializeNetwork()
{
    nn::nifm::Initialize();

    NN_LOG("Waiting for network interface availability...\n");
    nn::nifm::SubmitNetworkRequestAndWait();

    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_LOG("Network is not available.\n");
        return false;
    }

    nn::Result ret = nn::socket::Initialize(g_SocketConfigWithMemory);

    curl_global_init(CURL_GLOBAL_ALL);

    NN_LOG("Connecting to network...\n");

    if (ret.IsFailure())
    {
        NN_LOG("Connection could not be established.\n");
        return false;
    }

    NN_LOG("Connection was established.\n");

    return true;
}

void FinalizeNetwork()
{
    curl_global_cleanup();

    nn::socket::Finalize();
}

void CancelThreadFunction(void *pArg)
{
    nn::os::SleepThread(Timespan);

    NN_LOG("* nn::err::Cancel().\n");
    g_ServiceStatusDownloader.Cancel();

    return;
}

void ShowTime(const char* pName, nn::time::CalendarTime utcTime)
{
    NN_LOG("%s:%04d/%02d/%02d %02d:%02d\n", pName,
           utcTime.year, utcTime.month, utcTime.day,
           utcTime.hour, utcTime.minute);
}

void GetErrStatus(void* pData, const size_t size, const nn::ApplicationId& applicationId, const char* pServerCode)
{
    char* pCopyData = (char *)malloc(size + 1);
    std::memset(pCopyData, 0x00, size + 1);
    std::memcpy(pCopyData, pData, size);

    nn::err::ServiceStatusAccessor serviceStatusAccessor;
    serviceStatusAccessor.Initialize(pCopyData, size);

    nn::err::ServiceStatus serviceStatus;
    nn::Result result = serviceStatusAccessor.GetServiceStatus(&serviceStatus, applicationId, pServerCode);
    if (result.IsSuccess())
    {
        NN_LOG  ("  application_id = %016llx,server_code = %s\n", applicationId.value,pServerCode);
        NN_LOG  ("  hasMaintenance       :%s\n", (serviceStatus.hasMaintenance) ? ("true") : ("false"));
        ShowTime("  startMaintenanceDate ", serviceStatus.startMaintenanceDate);
        ShowTime("  endMaintenanceDate   ", serviceStatus.endMaintenanceDate);

        NN_LOG  ("  hasIncident          :%s\n", (serviceStatus.hasIncident) ? ("true") : ("false"));
        ShowTime("  startIncidentDate    ", serviceStatus.startIncidentDate);
        ShowTime("  endIncidentDate      ", serviceStatus.endIncidentDate);
    }
    else
    {
        NN_LOG("  Failed to get service status.\n");
    }

    free(pCopyData);

    return;
}

void DownloadJsonFile()
{
    NN_LOG("****************************************************************************************************\n");

    NN_ALIGNAS(4096) char jsonFile[MaxFileSize] = {};

    NN_LOG("** CASE-1 (Cancel)\n");
    {
        NN_LOG("* Call DownloadJsonFile().\n");

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_ThreadStack, CancelThreadFunction,  NULL, &g_Stack, sizeof(g_Stack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&g_ThreadStack);

        // キャンセルが行われるため、失敗します。
        nn::Result result = g_ServiceStatusDownloader.Download(jsonFile, MaxFileSize);
        nn::os::WaitThread(&g_ThreadStack);
        if (!result.IsSuccess()
            && result.GetModule() == g_ServiceStatusDownloader.CheckDownloadResult().GetModule()
            && result.GetDescription() == g_ServiceStatusDownloader.CheckDownloadResult().GetDescription())
        {
            NN_LOG("  Failed (module=%d,Description=%d). (TEST OK)\n",result.GetModule(),result.GetDescription());
        }
        else
        {
            NN_LOG("  Succeeded. (TEST NG)\n");
            return;
        }
    }

    NN_LOG("** CASE-2(Download)\n");
    {
        std::memset(jsonFile,0x00,sizeof(jsonFile));

        NN_LOG("* Call DownloadJsonFile().\n");
        nn::Result result = g_ServiceStatusDownloader.Download(jsonFile, MaxFileSize);
        if (result.IsSuccess() && g_ServiceStatusDownloader.CheckDownloadResult().IsSuccess())
        {
            NN_LOG("  Succeeded!\n");
            NN_LOG("  Downloaded json data=%s\n",jsonFile);

            NN_LOG("* Call GetErrStatus().\n");

            GetErrStatus(jsonFile, MaxFileSize, ApplicationId1, ServerCode1);

            GetErrStatus(jsonFile, MaxFileSize, ApplicationId1, ServerCode2);

            GetErrStatus(jsonFile, MaxFileSize, ApplicationId1, ServerCode3);

            GetErrStatus(jsonFile, MaxFileSize, ApplicationId2, ServerCode3);

            NN_LOG("  Succeeded!\n");
        }
        else
        {
            NN_LOG("  Failed. (module=%d,Description=%d)\n",result.GetModule(),result.GetDescription());
            return;
        }
    }

    NN_LOG("** CASE-3(CreateDownloadContext)\n");
    {
        nn::err::DownloadContext context;
        std::memset(jsonFile,0x00,sizeof(jsonFile));

        NN_LOG("* Call CreateDownloadContext().\n");
        nn::Result result = g_ServiceStatusDownloader.CreateDownloadContext(&context, jsonFile, MaxFileSize);
        if (result.IsSuccess() && g_ServiceStatusDownloader.CheckDownloadResult().IsSuccess())
        {
            NN_LOG("* Call curl_easy_perform().\n");

            CURLcode errorCode = curl_easy_perform(context.pCurl);

            if (errorCode == CURLE_OK)
            {
                NN_LOG("  Succeeded!\n");
                NN_LOG("* FileSize=%u\n", context.readSize);
            }
            else
            {
                NN_LOG("  Failed (errorCode = %d).\n", errorCode);
                g_ServiceStatusDownloader.DestroyDownloadContext(&context);
                return;
            }

            NN_LOG("* Call CheckDownloadResult().\n");
            result = g_ServiceStatusDownloader.CheckDownloadResult();
            if (result.IsSuccess() && g_ServiceStatusDownloader.CheckDownloadResult().IsSuccess())
            {
                NN_LOG("  Succeeded!\n");
                NN_LOG("  Downloaded json data=%s\n",context.pBuffer);

                NN_LOG("* Call GetErrStatus().\n");

                GetErrStatus(context.pBuffer, context.readSize, ApplicationId1, ServerCode1);

                GetErrStatus(context.pBuffer, context.readSize, ApplicationId1, ServerCode2);

                GetErrStatus(context.pBuffer, context.readSize, ApplicationId1, ServerCode3);

                GetErrStatus(context.pBuffer, context.readSize, ApplicationId2, ServerCode3);

                NN_LOG("  Succeeded!\n");
            }
            else
            {
                NN_LOG("  Failed (module=%d,Description=%d).\n",result.GetModule(),result.GetDescription());
                g_ServiceStatusDownloader.DestroyDownloadContext(&context);
                return;
            }
        }
        else
        {
            NN_LOG("  Failed (module=%d,Description=%d).\n",result.GetModule(),result.GetDescription());
            return;
        }
        g_ServiceStatusDownloader.DestroyDownloadContext(&context);
    }
    NN_LOG("  All Test Succeeded!\n");
    return;
}

extern "C" void nnMain()
{
    if (InitializeNetwork() == false)
    {
        return;
    }

    g_ServiceStatusDownloader.Initialize();

    DownloadJsonFile();

    g_ServiceStatusDownloader.Finalize();

    FinalizeNetwork();

    return;
}
