﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <sstream>

#include <nn/nn_Assert.h>

#include "Demo1Color.h"
#include "Demo1MomentModeState.h"

class Preprocess : public ReadWriteBase
{
public:
    Preprocess(
        nn::irsensor::MomentProcessorConfig* pMomentConfig)
        : m_pMomentConfig(pMomentConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << (m_pMomentConfig->preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize ?
            "Binarize" :
            "Cutoff");
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(delta);
        m_pMomentConfig->preprocess =
            (m_pMomentConfig->preprocess == nn::irsensor::MomentProcessorPreprocess_Binarize ?
                nn::irsensor::MomentProcessorPreprocess_Cutoff :
                nn::irsensor::MomentProcessorPreprocess_Binarize);
    }

private:
    nn::irsensor::MomentProcessorConfig* m_pMomentConfig;
};

class IntensityThreshold : public ReadWriteBase
{
public:
    IntensityThreshold(
        nn::irsensor::MomentProcessorConfig* pMomentConfig)
        : m_pMomentConfig(pMomentConfig)
    {
    }
    virtual void operator()(std::stringstream& sstr) NN_NOEXCEPT NN_OVERRIDE
    {
        sstr << m_pMomentConfig->preprocessIntensityThreshold;
    }
    virtual void operator()(int8_t delta) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pMomentConfig->preprocessIntensityThreshold += delta * 5;
        m_pMomentConfig->preprocessIntensityThreshold =
            std::min(
                std::max(
                    m_pMomentConfig->preprocessIntensityThreshold,
                    0),
                nn::irsensor::IrCameraIntensityMax
            );
    }

private:
    nn::irsensor::MomentProcessorConfig* m_pMomentConfig;
};

MomentModeState::MomentModeState(
    IrSensorMode* pNextProcessor,
    int* pMenuSelection,
    nn::irsensor::IrCameraHandle irCameraHandle) NN_NOEXCEPT
    : IrSensorModeState(pNextProcessor, pMenuSelection, irCameraHandle)
    , m_MomentProcessorState()
    , m_MomentProcessorConfig()
    , m_pPreprocess(NULL)
    , m_pIntensityThreshold(NULL)
{
    nn::irsensor::GetMomentProcessorDefaultConfig(&m_MomentProcessorConfig);

    nn::irsensor::MomentProcessorConfig* pMomentConfig = &m_MomentProcessorConfig;

    AddCommonReadWriteMenu(&m_ReadWriteMenu,
        &pMomentConfig->irCameraConfig,
        nn::irsensor::MomentProcessorExposureTimeMin,
        nn::irsensor::MomentProcessorExposureTimeMax
    );

    m_pPreprocess = new Preprocess(pMomentConfig);
    MenuItem preprocessMenuItem("Preprocess",
        m_pPreprocess
    );
    m_ReadWriteMenu.push_back(preprocessMenuItem);

    m_pIntensityThreshold = new IntensityThreshold(pMomentConfig);
    MenuItem intensityThresholdMenuItem("Intensity Threshold",
        m_pIntensityThreshold
    );
    m_ReadWriteMenu.push_back(intensityThresholdMenuItem);

    AddCommonReadOnlyMenu(
        &m_ReadOnlyMenu,
        &m_MomentProcessorState.samplingNumber,
        &m_MomentProcessorState.ambientNoiseLevel
    );
}

MomentModeState::~MomentModeState() NN_NOEXCEPT
{
    delete m_pPreprocess;
    delete m_pIntensityThreshold;
}

void MomentModeState::Start() NN_NOEXCEPT
{
    nn::irsensor::RunMomentProcessor(m_IrCameraHandle, m_MomentProcessorConfig);
}

void MomentModeState::Update() NN_NOEXCEPT
{
    int count;
    nn::irsensor::GetMomentProcessorStates(&m_MomentProcessorState, &count, 1, m_IrCameraHandle);
}

namespace
{

void WriteMomentProcessorState(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::irsensor::MomentProcessorState* pMomentState,
    const int screenIndex) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pPrimitiveRenderer);
    NN_ASSERT_NOT_NULL(pCommandBuffer);
    NN_ASSERT_NOT_NULL(pMomentState);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);
    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    // DepthStencilTypeをDepthStencilType_DepthNoWriteTestへ設定
    pPrimitiveRenderer->SetDepthStencilState(
        pCommandBuffer,
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest
    );

    float xStart;
    float xEnd;
    float yStart;
    float yEnd;
    if (screenIndex == 0)
    {
        xStart = 0.45f, xEnd = 0.0f;
        yStart = 0.2f, yEnd = 0.8f;
    }
    else
    {
        xStart = 0.95f, xEnd = 0.5f;
        yStart = 0.2f, yEnd = 0.8f;
    }

    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;

    size_t idx = 0;
    for (size_t y = 0; y < nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        for (size_t x = 0; x < nn::irsensor::MomentProcessorBlockColumnCount; ++x)
        {
            nn::util::Vector3fType corner;
            uint8_t averageIntensity = static_cast<uint8_t>(
                pMomentState->blocks[idx].averageIntensity
            );

            // 各面を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * x /
                    static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount),
                yStart + (yEnd - yStart) * y /
                    static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount), 0.0f);

            nn::util::Vector3fType size;
            nn::util::VectorSet(&size,
                (xEnd - xStart) / static_cast<float>(
                    nn::irsensor::MomentProcessorBlockColumnCount
                ),
                (yEnd - yStart) / static_cast<float>(
                    nn::irsensor::MomentProcessorBlockRowCount), 0.0f
                );

            nn::util::Uint8x4 gridColor ={ {
                    averageIntensity,
                    averageIntensity,
                    averageIntensity,
                    averageIntensity }
            };
            pPrimitiveRenderer->SetColor(gridColor);
            pPrimitiveRenderer->DrawQuad(
                pCommandBuffer,
                corner,
                size);

            // 重心点を描画
            nn::util::VectorSet(&corner,
                xStart + (xEnd - xStart) * pMomentState->blocks[idx].centroid.x /
                    static_cast<float>(nn::irsensor::IrCameraImageWidth),
                yStart + (yEnd - yStart) * pMomentState->blocks[idx].centroid.y /
                    static_cast<float>(nn::irsensor::IrCameraImageHeight), 0.0f);
            pPrimitiveRenderer->SetColor(Color::Red);
            pPrimitiveRenderer->DrawSphere(
                pCommandBuffer,
                nns::gfx::PrimitiveRenderer::Surface_Solid,
                nns::gfx::PrimitiveRenderer::Subdiv_Coarse,
                corner,
                4.0f / static_cast<float>(nn::irsensor::IrCameraImageWidth));

            idx++;
        }
    }

    // グリッドを描画
    pPrimitiveRenderer->SetLineWidth(0.0f);

    for (size_t x = 0; x <= nn::irsensor::MomentProcessorBlockColumnCount; ++x)
    {
        float xValue = xStart + (xEnd - xStart) * x /
            static_cast<float>(nn::irsensor::MomentProcessorBlockColumnCount);
        nn::util::VectorSet(&begin, xValue, yStart, 0.0f);
        nn::util::VectorSet(&end, xValue, yEnd, 0.0f);
        pPrimitiveRenderer->SetColor(Color::White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
    for (size_t y = 0; y <= nn::irsensor::MomentProcessorBlockRowCount; ++y)
    {
        float yValue = yStart + (yEnd - yStart) * y /
            static_cast<float>(nn::irsensor::MomentProcessorBlockRowCount);
        nn::util::VectorSet(&begin, xStart, yValue, 0.0f);
        nn::util::VectorSet(&end, xEnd, yValue, 0.0f);
        pPrimitiveRenderer->SetColor(Color::White);
        pPrimitiveRenderer->DrawLine(pCommandBuffer, begin, end);
    }
}

}

void MomentModeState::Render(
    nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer,
    nn::gfx::CommandBuffer* pCommandBuffer,
    int index) NN_NOEXCEPT
{
    WriteMomentProcessorState(pPrimitiveRenderer, pCommandBuffer, &m_MomentProcessorState, index);
}
