﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/fs.h>
#include <nn/gfx.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>
#include <nn/vi.h>

#include <nn/util/util_Color.h>

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN )
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

#include "Demo1.h"
#include "Demo1Color.h"
#include "Demo1Gfx.h"

void* GfxDemo::ReadResource( const char* pFilename ) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile( &hFile, pFilename, nn::fs::OpenMode_Read );
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    result = nn::fs::GetFileSize( &fileSize, hFile );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile( &readSize, hFile, 0, &fileHeader,
                                sizeof( nn::util::BinaryFileHeader ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT_EQUAL( readSize, sizeof( nn::util::BinaryFileHeader ) );
    size_t alignment = fileHeader.GetAlignment();

    m_pMemory.AlignUp( alignment );
    void* pBuffer = m_pMemory.Get();
    result = nn::fs::ReadFile( &readSize, hFile, 0, pBuffer, static_cast< size_t >( fileSize ) );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_ASSERT( readSize == static_cast< size_t >( fileSize ) );
    m_pMemory.Advance( static_cast< ptrdiff_t >( fileSize ) );

    nn::fs::CloseFile( hFile );

    return pBuffer;
}

void GfxDemo::InitializeResTextureFile() NN_NOEXCEPT
{
    void *pTextureResource = ReadResource( "Contents:/Check.bntx" );
    m_pResTextureFile = nn::gfx::ResTextureFile::ResCast( pTextureResource );
    m_pResTextureFile->Initialize( &m_pGraphicsSystem->GetDevice() );
    for( int idxTexture = 0, textureCount = m_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        m_pResTextureFile->GetResTexture( idxTexture )->Initialize(
            &m_pGraphicsSystem->GetDevice()
        );

    }
    m_IndexBg = m_pResTextureFile->GetTextureDic()->FindIndex("Check_BG_01");
    NN_ASSERT(m_IndexBg != -1);
    m_IndexText = m_pResTextureFile->GetTextureDic()->FindIndex("Check_Text_01");
    NN_ASSERT(m_IndexText != -1);
}

void GfxDemo::FinalizeResTextureFile() const NN_NOEXCEPT
{
    for( int idxTexture = 0, textureCount = m_pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        m_pResTextureFile->GetResTexture( idxTexture )->Finalize( &m_pGraphicsSystem->GetDevice() );
    }
    m_pResTextureFile->Finalize( &m_pGraphicsSystem->GetDevice() );
}


void GfxDemo::InitializeResources() NN_NOEXCEPT
{
    // メモリ
    m_pMemoryHeap.Reset(new unsigned char [32 * 1024 * 1024] );
    m_pMemory = m_pMemoryHeap;

    InitializeResTextureFile();
}

void GfxDemo::FinalizeResources() const NN_NOEXCEPT
{
    // 各オブジェクトを破棄
    FinalizeResTextureFile();

    delete[] reinterpret_cast<unsigned char*>(m_pMemoryHeap.Get());
}

// --------------------------------------------------------------------------------------
//  メモリ確保関数
// --------------------------------------------------------------------------------------
void* AllocateFunction(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    return pAllocator->Allocate(size, alignment);
}

// --------------------------------------------------------------------------------------
//  メモリ開放関数
// --------------------------------------------------------------------------------------
void DeallocateFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

bool GfxDemo::InitializeMesh(
    nns::gfx::PrimitiveRenderer::PrimitiveMesh& meshBuffer,
    bool bUseVertexColor) const NN_NOEXCEPT
{
    const int NumIndices = 72;
    const int NumVertices = 48;

    // 頂点カラーを利用する場合
    if (bUseVertexColor)
    {
        if (!meshBuffer.Initialize(GetPrimitiveRenderer().GetGpuBuffer(),
            NumVertices, NumIndices))
        {
            return false;
        }
    }
    else
    {
        if (!meshBuffer.Initialize(GetPrimitiveRenderer().GetGpuBuffer(), NumVertices, NumIndices,
            nns::gfx::PrimitiveRenderer::VertexFormat_Pos))
        {
            return false;
        }
    }
    return true;
}

void GfxDemo::InitializeIndexBuffer(
    nns::gfx::PrimitiveRenderer::PrimitiveMesh& meshBuffer) const NN_NOEXCEPT
{
    uint32_t idxFace = 0;
    size_t idx = 0;
    uint32_t* pIndexData = meshBuffer.GetIndexBufferCpuAddress();
    for (idxFace = 0; idxFace < 12; idxFace++)
    {
        pIndexData[idx++] = idxFace * 4 + 0;
        pIndexData[idx++] = idxFace * 4 + 1;
        pIndexData[idx++] = idxFace * 4 + 2;

        pIndexData[idx++] = idxFace * 4 + 0;
        pIndexData[idx++] = idxFace * 4 + 2;
        pIndexData[idx++] = idxFace * 4 + 3;
    }
}

namespace {
const nn::util::Vector3fType Center = { 0.f, 0.f, 0.f };
const nn::util::Vector3fType SizeVecor = { 2.f, 2.f, 2.f };

const float Width = 1.f;
const float Height = 1.f;
const float DepthVertex = 1.f;

// インデックスバッファを設定します。
static const uint32_t FaceTable[12][4] =
{
    { 0, 1, 2, 3 }, // Front
    { 3, 2, 6, 7 }, // Right
    { 7, 6, 5, 4 }, // Back
    { 4, 5, 1, 0 }, // Left
    { 1, 5, 6, 2 }, // Top
    { 4, 0, 3, 7 }, // Bottom

    { 8, 9,10,11 }, // Front
    { 11,10,14,15 }, // Right
    { 15,14,13,12 }, // Back
    { 12,13, 9, 8 }, // Left
    { 9,14,15,10 }, // Top
    { 12, 8,11,15 }, // Bottom
};

// 頂点バッファを設定します。
const nn::util::Float3 VertexPos[16] =
{
    NN_UTIL_FLOAT_3_INITIALIZER(Width / 2.f,  Height, -DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width, -Height,       -DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width, -Height,       -DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width / 2.f,  Height, -DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width / 2.f,  Height,  DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width, -Height,        DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width, -Height,        DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width / 2.f,  Height,  DepthVertex / 2.f),

    NN_UTIL_FLOAT_3_INITIALIZER(Width / 2.f,  Height + 4.f, -DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width,       -Height + 4.f,       -DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width,      -Height + 4.f,       -DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width / 2.f, Height + 4.f, -DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width / 2.f,  Height + 4.f,  DepthVertex / 2.f),
    NN_UTIL_FLOAT_3_INITIALIZER(Width,       -Height + 4.f,        DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width,      -Height + 4.f,        DepthVertex),
    NN_UTIL_FLOAT_3_INITIALIZER(-Width / 2.f, Height + 4.f,  DepthVertex / 2.f)
};

static const nn::util::Float2 VertexUv[8] =
{
    NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  0.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  1.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  1.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  0.0f),

    NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  0.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(0.0f,  1.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  1.0f),
    NN_UTIL_FLOAT_2_INITIALIZER(1.0f,  0.0f)
};

static const nn::util::Float4 VertexColor[16] =
{
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),

    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(1.0f, 1.0f, 1.0f, 1.0f),
    NN_UTIL_FLOAT_4_INITIALIZER(0.2f, 0.2f, 0.2f, 1.0f),
};
}

void GfxDemo::InitializeVertexBuffer(
    nns::gfx::PrimitiveRenderer::PrimitiveMesh& meshBuffer,
    bool bUseVertexColor) const NN_NOEXCEPT
{
    const float ScaleX = 0.8f;
    const float ScaleY = 0.8f;
    const float ScaleZ = 0.8f;
    uint32_t idxFace = 0;
    idxFace = 0;
    size_t idxVertex = 0;

    nn::util::Float3* pPos = static_cast<nn::util::Float3*>(
        meshBuffer.GetVertexBufferCpuAddress(nns::gfx::PrimitiveRenderer::VertexAttribute_Pos)
        );

    // 頂点座標
    int idxCorner = 0;
    for (idxFace = 0; idxFace < 12; idxFace++)
    {
        for (idxCorner = 0; idxCorner < 4; idxCorner++)
        {
            const size_t Index = FaceTable[idxFace][idxCorner];

            // 位置
            pPos[idxVertex].x = VertexPos[Index].x * ScaleX;
            pPos[idxVertex].y = VertexPos[Index].y * ScaleY;
            pPos[idxVertex].z = VertexPos[Index].z * ScaleZ;
            idxVertex++;
        }
    }

    // 頂点カラーを利用する場合
    if (bUseVertexColor)
    {
        nn::util::Float2* pUv = static_cast<nn::util::Float2*>(
            meshBuffer.GetVertexBufferCpuAddress(
                nns::gfx::PrimitiveRenderer::VertexAttribute_Uv
            )
            );
        nn::util::Float4* pColor = static_cast<nn::util::Float4*>(
            meshBuffer.GetVertexBufferCpuAddress(
                nns::gfx::PrimitiveRenderer::VertexAttribute_Color
            )
            );
        idxCorner = 0;
        idxVertex = 0;
        for (idxFace = 0; idxFace < 12; idxFace++)
        {
            for (idxCorner = 0; idxCorner < 4; idxCorner++)
            {
                const size_t Index = FaceTable[idxFace][idxCorner];

                // カラー
                pColor[idxVertex].x = VertexColor[Index].x;
                pColor[idxVertex].y = VertexColor[Index].y;
                pColor[idxVertex].z = VertexColor[Index].z;
                pColor[idxVertex].w = VertexColor[Index].w;
                // UV
                pUv[idxVertex].x = VertexUv[idxCorner].x;
                pUv[idxVertex].y = VertexUv[idxCorner].y;
                idxVertex++;
            }
        }
    }
}


// ---------------------------------------------------------------
// 独自頂点バッファを使用した描画
// ---------------------------------------------------------------
bool GfxDemo::DrawUserMesh(
    nn::util::Matrix4x3f& modelMatrix,
    bool bUseVertexColor,
    int64_t frame ) const NN_NOEXCEPT
{
    nns::gfx::PrimitiveRenderer::PrimitiveMesh meshBuffer;

    // meshBuffer を初期化する
    if (!InitializeMesh(meshBuffer, bUseVertexColor))
    {
        return false;
    }

    // インデックスバッファを設定します。
    InitializeIndexBuffer(meshBuffer);

    // Vertex Buffer を初期化する
    InitializeVertexBuffer(meshBuffer, bUseVertexColor);

    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);

    // 頂点カラーを利用する場合
    if ( bUseVertexColor )
    {
        GetPrimitiveRenderer().SetColor(Color::Red);
    }
    else
    {
        GetPrimitiveRenderer().SetColor(Color::Green);
    }

    // 1メッシュを2回に分けて描画
    GetPrimitiveRenderer().DrawUserMesh( &GetCommandBuffer(),
                                        nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
                                        &meshBuffer, 0, 36, 0 );

    int idexCount = ( frame % 360 ) / 10;
    if ( idexCount < 3 )
    {
        idexCount = 36;
    }
    GetPrimitiveRenderer().DrawUserMesh( &GetCommandBuffer(),
                                        nn::gfx::PrimitiveTopology::PrimitiveTopology_TriangleList,
                                        &meshBuffer, 36, idexCount, 0 );

    return true;
}


// 画面全体にテクスチャを張る
void GfxDemo::DrawTexture(
    nn::util::Matrix4x3f& modelMatrix,
    nn::gfx::DescriptorSlot& textureDescriptor,
    nn::gfx::DescriptorSlot& samplerDescriptor) const NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);

    GetPrimitiveRenderer().SetViewMatrix(&viewMatrix);
    GetPrimitiveRenderer().SetProjectionMatrix(&projectionMatrix);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawScreenQuad(&GetCommandBuffer(), textureDescriptor,
        samplerDescriptor);
}

// ビューとプロジェクションを設定
void GfxDemo::SetViewMatrix(int64_t frame) const NN_NOEXCEPT
{
    const float Radius = 20.f;
    const float ComponentX = Radius * sin(frame / 500.f);
    const float ComponentZ = Radius * cos(frame / 500.f);

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::Vector3fType camPos = { ComponentX, 10.f, ComponentZ };
    nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
    nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);
    GetPrimitiveRenderer().SetViewMatrix(&viewMatrix);
}

// プロジェクションを初期化
void GfxDemo::InitProjection() const NN_NOEXCEPT
{
    nn::util::Matrix4x4fType projectionMatrix;
    const float Fovy = nn::util::FloatPi / 4.0f;
    const float Aspect = static_cast< float >(m_pGraphicsSystem->GetDisplayWidth()) /
        static_cast< float >(m_pGraphicsSystem->GetDisplayHeight());
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, Fovy,
        Aspect, 0.1f, 1000.f);
    GetPrimitiveRenderer().SetProjectionMatrix(&projectionMatrix);

    // Depth Enable
    GetPrimitiveRenderer().SetDepthStencilState(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest
    );
}

// 軸を描画
void GfxDemo::DrawAxis() const NN_NOEXCEPT
{
    float interval = -10.f;
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    GetPrimitiveRenderer().SetLineWidth(1.f);
    for (int i = 0; i < 21; i++)
    {

        nn::util::VectorSet(&begin, -10.f, 0.f, interval);
        nn::util::VectorSet(&end, 10.f, 0.f, interval);
        GetPrimitiveRenderer().SetColor(Color::White);
        GetPrimitiveRenderer().DrawLine(&GetCommandBuffer(), begin, end);
        nn::util::VectorSet(&begin, interval, 0.f, -10.f);
        nn::util::VectorSet(&end, interval, 0.f, 10.f);
        GetPrimitiveRenderer().DrawLine(&GetCommandBuffer(), begin, end);
        interval += 1.0f;
    }

    GetPrimitiveRenderer().SetLineWidth(5.f);
    nn::util::Vector3fType zeroVector;
    nn::util::VectorSet(&zeroVector, 0.f, 0.f, 0.f);
    nn::util::Vector3fType axisPos;
    nn::util::VectorSet(&axisPos, 5.f, 0.f, 0.f);
    GetPrimitiveRenderer().SetColor(Color::Red);
    GetPrimitiveRenderer().DrawLine(&GetCommandBuffer(), zeroVector, axisPos);
    nn::util::VectorSet(&axisPos, 0.f, 0.f, 5.f);
    GetPrimitiveRenderer().SetColor(Color::Blue);
    GetPrimitiveRenderer().DrawLine(&GetCommandBuffer(), zeroVector, axisPos);
    nn::util::VectorSet(&axisPos, 0.f, 5.f, 0.f);
    GetPrimitiveRenderer().SetColor(Color::Green);
    GetPrimitiveRenderer().DrawLine(&GetCommandBuffer(), zeroVector, axisPos);
}

// CIRCLEを描画
void GfxDemo::DrawCircle(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, -6.f, 0.f, -7.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetColor(Color::Green);
    GetPrimitiveRenderer().DrawCircle(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
        Center, 1.f);

    nn::util::VectorSet(&translate, -2.f, 0.f, -7.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetLineWidth(3.f);
    GetPrimitiveRenderer().DrawCircle(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
        Center, 1.f);
}

// TRIANGLEを描画
void GfxDemo::DrawTriangle(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, 2.f, 0.f, -7.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetLineWidth(3.f);
    GetPrimitiveRenderer().SetColor(Color::Blue);
    GetPrimitiveRenderer().DrawTriangle(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        Center, SizeVecor);

    nn::util::VectorSet(&translate, 6.f, 0.f, -7.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawTriangle(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        Center, SizeVecor);
}

// CUBEを描画する
void GfxDemo::DrawCube(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, -6.f, 0.f, -2.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetLineWidth(3.f);
    GetPrimitiveRenderer().SetColor(Color::Red);
    GetPrimitiveRenderer().DrawCube(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        Center, SizeVecor);

    nn::util::VectorSet(&translate, -2.f, 0.f, -2.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawCube(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        Center, SizeVecor);
}

// QUADSを描画する
void GfxDemo::DrawQuad(
    nn::util::Matrix4x3f& modelMatrix,
    nn::gfx::DescriptorSlot& textureDescriptor,
    nn::gfx::DescriptorSlot& samplerDescriptor) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, 2.f, 0.f, -2.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    // 乗算するカラーをwhiteにしておく
    GetPrimitiveRenderer().SetColor(Color::White);
    GetPrimitiveRenderer().DrawQuad(&GetCommandBuffer(),
        Center, SizeVecor, textureDescriptor, samplerDescriptor);

    nn::util::VectorSet(&translate, 6.f, 0.f, -2.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetColor(Color::Red);
    GetPrimitiveRenderer().DrawQuad(&GetCommandBuffer(), Center, SizeVecor);
}

// SPHEREを描画
void GfxDemo::DrawShere(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, 2.f, 0.f, 3.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetLineWidth(3.f);
    GetPrimitiveRenderer().SetColor(Color::Green);
    GetPrimitiveRenderer().DrawSphere(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
        Center, 3.f);

    nn::util::VectorSet(&translate, 6.f, 0.f, 3.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawSphere(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        nns::gfx::PrimitiveRenderer::Subdiv::Subdiv_Normal,
        Center, 3.f);
}

// CONEを描画
void GfxDemo::DrawCone(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, -6.f, 0.f, 3.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetColor(Color::Blue);
    GetPrimitiveRenderer().DrawCone(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        Center, 1.f, 1.f);

    nn::util::VectorSet(&translate, -2.f, 0.f, 3.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawCone(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        Center, 1.f, 1.f);
}

// CAPSULEを描画
void GfxDemo::DrawCapsule(nn::util::Matrix4x3f& modelMatrix) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;

    nn::util::VectorSet(&translate, 2.f, 0.f, 8.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().SetColor(Color::Red);
    GetPrimitiveRenderer().DrawCapsule(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Wired,
        Center, 1.f, 2.f);

    nn::util::VectorSet(&translate, 6.f, 0.f, 8.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    GetPrimitiveRenderer().SetModelMatrix(&modelMatrix);
    GetPrimitiveRenderer().DrawCapsule(&GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::Surface::Surface_Solid,
        Center, 1.f, 2.f);
}

// 図形を描画する
void GfxDemo::DrawObjects(
    int64_t frame,
    nn::gfx::DescriptorSlot& textureDescriptor,
    nn::gfx::DescriptorSlot& samplerDescriptor) const NN_NOEXCEPT
{
    nn::util::Vector3f translate;
    nn::util::Matrix4x3f modelMatrix;
    nn::util::Vector3f vecZero;
    nn::util::VectorZero(&vecZero);

    float rotY = frame * 0.05f;
    nn::util::Vector3f rotValue;
    nn::util::VectorSet(&rotValue, 0.2f, rotY, 0.2f);
    nn::util::MatrixIdentity(&modelMatrix);
    nn::util::MatrixSetRotateXyz(&modelMatrix, rotValue);
    nn::util::MatrixSetAxisW(&modelMatrix, vecZero);

    // CIRCLEを描画
    DrawCircle(modelMatrix);

    // TRIANGLEを描画
    DrawTriangle(modelMatrix);

    // CUBEを描画する
    DrawCube(modelMatrix);

    // QUADSを描画する
    DrawQuad(modelMatrix, textureDescriptor, samplerDescriptor);

    // SPHEREを描画
    DrawShere(modelMatrix);

    // CONEを描画
    DrawCone(modelMatrix);

    // CAPSULEを描画
    DrawCapsule(modelMatrix);

    // 独自頂点バッファを使用した描画
    nn::util::VectorSet(&translate, -2.f, 0.f, 8.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    // 頂点カラーを利用しない場合
    DrawUserMesh(modelMatrix, false, frame);

    nn::util::VectorSet(&translate, -6.f, 0.f, 8.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    // 頂点カラーを利用する場合
    DrawUserMesh(modelMatrix, true, frame);
}

// ---------------------------------------------------------------
// コマンド生成
// ---------------------------------------------------------------
void GfxDemo::MakeCommand( int64_t frame, const char* pName ) NN_NOEXCEPT
{
    NN_UNUSED(pName);

    nn::gfx::DescriptorSlot constantBufferDescriptor;
    nn::gfx::DescriptorSlot textureDescriptor;
    nn::gfx::DescriptorSlot samplerDescriptor;
    nn::gfx::DescriptorPool* pBufferDescriptorPool = m_pGraphicsSystem->GetBufferDescriptorPool();
    pBufferDescriptorPool->GetDescriptorSlot(
        &constantBufferDescriptor,
        m_pGraphicsSystem->GetBufferDescriptorIndex());

    nn::gfx::DescriptorPool* pTextureDescriptorPool = m_pGraphicsSystem->GetTexureDescriptorPool();
    pTextureDescriptorPool->GetDescriptorSlot(
        &textureDescriptor,
        m_TextureDescriptorIndex0);
    nn::gfx::DescriptorPool* pSamplerDescriptorPool = m_pGraphicsSystem->GetSamplerDescriptorPool();
    pSamplerDescriptorPool->GetDescriptorSlot(
        &samplerDescriptor,
        m_pGraphicsSystem->GetSamplerDescriptorIndex());

    // Depth Enable
    GetPrimitiveRenderer().SetDepthStencilState( &GetCommandBuffer(),
        nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthNoWriteTest );

    nn::util::Vector3f translate;

    nn::util::Matrix4x3f modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);

    // 画面全体にテクスチャを張る
    DrawTexture(modelMatrix, textureDescriptor, samplerDescriptor);

    // ビューとプロジェクションを設定
    SetViewMatrix(frame);

    // プロジェクションを初期化
    InitProjection();

    // 軸を描画
    DrawAxis();

    // 図形を描画する
    DrawObjects(frame, textureDescriptor, samplerDescriptor);

    // model, view, projection行列をデフォルトに
    GetPrimitiveRenderer().SetDefaultParameters();

    // PrimitiveRendererを利用してスキャンバッファにコピー
    nn::gfx::ColorTargetView* pTarget = NULL;
    nn::gfx::SwapChain* pSwapChain = m_pGraphicsSystem->GetSwapChain();
    pTarget = pSwapChain->AcquireNextScanBufferView();
    GetCommandBuffer().SetRenderTargets(1, &pTarget, NULL);

    // カラーバッファをテクスチャとして利用するのでキャッシュをフラッシュ
    GetCommandBuffer().FlushMemory(nn::gfx::GpuAccess_ColorBuffer);
    // 上書き設定
    GetCommandBuffer().SetBlendState(
        GetPrimitiveRenderer().GetBlendState(
            nns::gfx::PrimitiveRenderer::BlendType::BlendType_Opacity
        )
    );

    // 画面を黒で塗りつぶす
    GetPrimitiveRenderer().SetColor(Color::Black);
    GetPrimitiveRenderer().Draw2DRect(&GetCommandBuffer(),
        0.0f,
        0.0f,
        static_cast<float>(m_pGraphicsSystem->GetDisplayWidth()),
        static_cast<float>(m_pGraphicsSystem->GetDisplayHeight())
    );

    // 左下にグラフィックを描画
    GetPrimitiveRenderer().SetColor(Color::White);
    m_pGraphicsSystem->DrawScreenQuarter(GraphicsSystem::DrawPosition::DrawPosition_BottomLeft);
}

void GfxDemo::Initialize() NN_NOEXCEPT
{
    InitializeResources();

    nn::gfx::TextureView* textView0 = m_pResTextureFile->GetResTexture(m_IndexBg)
        ->GetTextureView();
    nn::gfx::TextureView* textView1 = m_pResTextureFile->GetResTexture(m_IndexText)
        ->GetTextureView();
    nn::gfx::DescriptorSlot descriptorSlot0;
    nn::gfx::DescriptorSlot descriptorSlot1;

    m_TextureDescriptorIndex0 = m_pGraphicsSystem->RegisterTextureViewSlot(
        &descriptorSlot0,
        *textView0
    );
    m_TextureDescriptorIndex1 = m_pGraphicsSystem->RegisterTextureViewSlot(
        &descriptorSlot1,
        *textView1
    );

    NN_ASSERT(m_pMemoryHeap.Distance(m_pMemory.Get()) < 32 * 1024 * 1024);

    // 現在のスレッド(メインスレッド)を 0 番コアに割り当て
    nn::os::SetThreadCoreMask(nn::os::GetCurrentThread(), 0, 1);

    m_pThreadStack = reinterpret_cast<char*>(
        GetStandardAllocator().Allocate(
            ThreadStackSize,
            nn::os::ThreadStackAlignment
        )
        );

    // スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &m_GfxThread, GfxThreadFunc, (void*)this, m_pThreadStack,
        ThreadStackSize, nn::os::DefaultThreadPriority));
}

GfxDemo::~GfxDemo() NN_NOEXCEPT
{

}

void GfxDemo::Start() NN_NOEXCEPT
{
    // スレッドをStart
    nn::os::StartThread(&m_GfxThread);
}

void GfxDemo::Wait() NN_NOEXCEPT
{
    // スレッドの破棄の前にWait
    nn::os::WaitThread(&m_GfxThread);
}

void GfxDemo::End() NN_NOEXCEPT
{
}

void GfxDemo::Finalize() NN_NOEXCEPT
{

    FinalizeResources();

    // サウンドスレッドの破棄
    nn::os::DestroyThread(&m_GfxThread);

    GetStandardAllocator().Free(m_pThreadStack);
}

void GfxDemo::Draw() NN_NOEXCEPT
{
    static int64_t s_Frame = 0;
    MakeCommand(s_Frame, m_Name.c_str());
    s_Frame++;
}

void GfxDemo::GfxThreadFuncImpl() const NN_NOEXCEPT
{
}

void GfxDemo::GfxThreadFunc(void* pArg) NN_NOEXCEPT
{
    GfxDemo* pGfxDemo = reinterpret_cast<GfxDemo*>(pArg);
    pGfxDemo->GfxThreadFuncImpl();
}
