﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{CodecOpusEncoder.cpp,PageSampleCodecOpusEncoder}
 *
 * @brief
 * Opus エンコーダのサンプルプログラム
 */

/**
 * @page PageSampleCodecOpusEncoder Opus エンコーダ
 * @tableofcontents
 *
 * @brief
 * Opus エンコーダのサンプルプログラムの解説です。
 *
 * @section PageSampleCodecOpusEncoder_SectionBrief 概要
 * Opus エンコーダで Pcm データをエンコードし、エンコードされたデータを
 * Opus デコーダでデコードしながらオーディオ再生を行うサンプルです。
 * オーディオ再生にはオーディオレンダラを利用します。
 *
 * @section PageSampleCodecOpusEncoder_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/CodecOpusEncoder
 * Samples/Sources/Applications/CodecOpusEncoder @endlink 以下にあります。
 *
 * @section PageSampleCodecOpusEncoder_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleCodecOpusEncoder_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、1 分ほど BGM を再生し、自動的に終了します。
 *
 * @section PageSampleCodecOpusEncoder_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleCodecOpusEncoder_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCodecOpusEncoder_SectionDetail 解説
 *
 * @subsection PageSampleCodecOpusEncoder_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CodecOpusEncoder.cpp
 *  @includelineno CodecOpusEncoder.cpp
 *
 * @subsection PageSampleCodecOpusEncoder_SectionSampleDetail サンプルプログラムの解説
 *
 * このサンプルプログラムでは、Pcm をエンコードして Opus データを作成し、
 * その Opus データをデコードして得られた Pcm データを再生します。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - ファイルシステムの初期化処理
 * - Pcm データの読み込み
 * - Opus エンコーダの初期化
 * - Opus デコーダの初期化
 * - オーディオレンダラの初期化、および開始
 * - Opus エンコーダの実行、Opus デコーダの実行、デコード結果の再生
 * - オーディオレンダラの終了
 * - Opus エンコーダの終了
 * - Opus デコーダの終了
 * - ファイルシステムの終了処理
 *
 * 最初にファイルシステムの初期化を行います。
 * リソースファイルは Wav フォーマットの音源データを使用します。
 * リソースディレクトリの場所は適宜変更してください。
 *
 * ファイルから、音源の情報と Pcm データを読み込みます。
 *
 * ヘッダ情報をもとに Opus エンコーダを初期化します。
 *
 * 初期化関数 nn::codec::OpusEncoder::Initialize() にはサンプルレートとチャンネル数、さらにそれらによってサイズが決まるワークバッファを与える必要があります。
 *
 * ワークバッファのサイズは nn::codec::OpusEncoder::GetWorkBufferSize() により取得可能です。
 *
 * Opus デコーダを初期化します。
 * nn::codec::OpusDecoder に関しては Samples/Sources/Applications/OpusDecoder サンプルおよび関連する API リファレンスを参照してください。
 *
 * オーディオレンダラを準備します。
 * オーディオレンダラの詳細に関しては Samples/Sources/Applications/AudioRenderer サンプルおよび関連する API リファレンスを参照してください。
 *
 * メインループでは、リアルタイムストリーム再生のような処理を行います。
 * Opus エンコーダにより Pcm データをエンコードした後 (例えばオーディオデータ配信側の処理)、
 * Opus デコーダによりエンコードデータを Pcm データにデコードし (例えば配信されたオーディオデータ
 * の受信者側の処理)、
 * 再生が終わったバッファに Pcm データを追加してnn::audio::VoiceType 経由で順次再生します。
 *
 * Pcm データをすべてエンコードし、再生を終えると、オーディオレンダラを停止し、
 * Opus エンコーダ、Opus デコーダ、オーディオレンダラを終了し、メモリを破棄します。
 */

#include <cstdlib>  // std::malloc, std::free
#include <nn/nn_Abort.h>
#include <nn/audio.h>
#include <nn/codec.h>
#include <nn/fs.h>
#include <nn/mem.h>
#include <nn/os.h>

#include <nns/audio/audio_WavFormat.h>
#include <nns/nns_Log.h>

//==============================================================================
// USE_SILK_CODING を有効にすると、Silk 符号化モードでエンコードします。
//#define USE_SILK_CODING
//==============================================================================

namespace {

char g_HeapBuffer[8 * 1024 * 1024];

char* g_MountRomCacheBuffer = nullptr;

const int BufferCount = 4;
const int OpusChannelCount = 2;
const int OpusSampleCount = 960;

NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN int16_t g_PcmBuffer[1024 * 1024];
NN_STATIC_ASSERT(BufferCount * OpusSampleCount * OpusChannelCount * sizeof(int16_t) <= sizeof(g_PcmBuffer));

void InitializeFileSystem()
{
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    g_MountRomCacheBuffer = new char[mountRomCacheBufferSize];

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("asset", g_MountRomCacheBuffer, mountRomCacheBufferSize));
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = nullptr;
}

}

extern "C" void nnMain()
{
#ifdef USE_SILK_CODING
    NNS_LOG("CodecOpusEncoder sample (using SILK coding)\n");
#else  // USE_SILK_CODING
    NNS_LOG("CodecOpusEncoder sample (using CELT coding)\n");
#endif  // USE_SILK_CODING

    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));
    nn::mem::StandardAllocator pcmBufferAllocator(g_PcmBuffer, sizeof(g_PcmBuffer));

    // ファイルシステムを初期化します
    InitializeFileSystem();

    // 音源データを読み込みます
    const char Filename[] = "asset:/SampleBgm0-1ch.wav";
    nn::fs::FileHandle handle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&handle, Filename, nn::fs::OpenMode_Read));
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, handle));
    auto const pWaveDataHead = new uint8_t[static_cast<size_t>(size)];
    NN_ABORT_UNLESS_NOT_NULL(pWaveDataHead);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(handle, 0, pWaveDataHead, static_cast<std::size_t>(size)));
    nn::fs::CloseFile(handle);

    const int8_t* pWaveData = reinterpret_cast<const int8_t*>(pWaveDataHead);

    // ヘッダより、サンプルレートとチャンネル数を取得します
    nns::audio::WavFormat wavFormat;
    auto parseResult = ParseWavFormat(&wavFormat, pWaveDataHead, static_cast<size_t>(size));
    NN_ABORT_UNLESS(parseResult == nns::audio::WavResult_Success);
    const int channelCount = wavFormat.channelCount;
    const int encoderSampleRate = wavFormat.sampleRate;
    const int64_t sampleCountPerChannel = wavFormat.dataSize / (wavFormat.bitsPerSample >> 3) / channelCount;

    // Opus エンコーダを初期化します
    nn::codec::OpusEncoder encoder;
    void* encoderWorkBuffer = std::malloc(encoder.GetWorkBufferSize(encoderSampleRate, channelCount));
    NN_ABORT_UNLESS_NOT_NULL(encoderWorkBuffer);
    nn::codec::OpusResult opusResult = encoder.Initialize(encoderSampleRate, channelCount, encoderWorkBuffer, encoder.GetWorkBufferSize(encoderSampleRate, channelCount));
    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);

    // エンコーダのビットレートを設定する。
    const int HigherBitRate = 60000;
    const int LowerBitRate = 12000;
    encoder.SetBitRate(HigherBitRate);

    // Opus デコーダを初期化します
    const int decoderSampleRate = 48000;
    nn::codec::OpusDecoder decoder;
    void* decoderWorkBuffer = std::malloc(decoder.GetWorkBufferSize(decoderSampleRate, channelCount));
    NN_ABORT_UNLESS_NOT_NULL(decoderWorkBuffer);
    opusResult = decoder.Initialize(decoderSampleRate, channelCount, decoderWorkBuffer, decoder.GetWorkBufferSize(decoderSampleRate, channelCount));
    NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);

    //オーディオレンダラを構築します
    nn::audio::AudioRendererParameter audioRendererParameter;
    nn::audio::InitializeAudioRendererParameter(&audioRendererParameter);
    audioRendererParameter.sampleRate = decoderSampleRate;
    audioRendererParameter.sampleCount = 240;
    audioRendererParameter.mixBufferCount = 2;
    audioRendererParameter.voiceCount = 1;
    audioRendererParameter.sinkCount = 1;
    NN_ABORT_UNLESS(nn::audio::IsValidAudioRendererParameter(audioRendererParameter));

    std::size_t audioRendererWorkBufferSize = nn::audio::GetAudioRendererWorkBufferSize(audioRendererParameter);
    void* audioRendererWorkBuffer = allocator.Allocate(audioRendererWorkBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(audioRendererWorkBuffer);

    nn::audio::AudioRendererHandle audioRendererHandle;
    nn::os::SystemEvent systemEvent;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::OpenAudioRenderer(&audioRendererHandle, &systemEvent, audioRendererParameter, audioRendererWorkBuffer, audioRendererWorkBufferSize));

    std::size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(audioRendererParameter);
    void* configBuffer = allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, audioRendererParameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 2);

    nn::audio::DeviceSinkType sink;
    int8_t bus[2] = { 0, 1 };
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &sink, &finalMix, bus, sizeof(bus) / sizeof(bus[0]), "MainAudioOut"));

    // WaveBuffer に追加するオーディオデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType pcmBufferMemoryPool;
    NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &pcmBufferMemoryPool, g_PcmBuffer, sizeof(g_PcmBuffer)));
    NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&pcmBufferMemoryPool));

    nn::audio::VoiceType voice;
    nn::audio::AcquireVoiceSlot(&config, &voice, decoderSampleRate, channelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
    nn::audio::SetVoiceDestination(&config, &voice, &finalMix);
    nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 0);
    nn::audio::SetVoiceMixVolume(&voice, &finalMix, 1.0f, 0, 1);

    // ストリーム再生用のバッファを用意します
    // 最初は 0 で初期化したものを渡し、この後のメインループで Opus データのデコード結果を与えています
    nn::audio::WaveBuffer waveBuffer[BufferCount];
    void* pcmBuffer[BufferCount];
    for (int i = 0; i < BufferCount; ++i)
    {
        const std::size_t pcmBufferSize = OpusSampleCount * OpusChannelCount * sizeof(int16_t);
        pcmBuffer[i] = pcmBufferAllocator.Allocate(pcmBufferSize, nn::audio::BufferAlignSize);

        std::memset(pcmBuffer[i], 0, pcmBufferSize);

        waveBuffer[i].buffer = pcmBuffer[i];
        waveBuffer[i].size = sizeof(g_PcmBuffer[i]);
        waveBuffer[i].startSampleOffset = 0;
        waveBuffer[i].endSampleOffset = static_cast<int32_t>(sizeof(g_PcmBuffer[i]) / sizeof(int16_t));
        waveBuffer[i].loop = false;
        waveBuffer[i].isEndOfStream = false;
        waveBuffer[i].pContext = nullptr;
        waveBuffer[i].contextSize = 0;

        nn::audio::AppendWaveBuffer(&voice, &waveBuffer[i]);
    }

    // オーディオレンダラを開始します
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(audioRendererHandle));

    // Pcm データ本体を取得します
    auto p = reinterpret_cast<const int16_t*>(pWaveData);
    int bufferCount = BufferCount;
    const size_t encodedDataSizeMaximum = nn::codec::OpusPacketSizeMaximum;

    // フレームはエンコード単位で変更することができます。
    // 5m -> 10ms -> 2.5ms -> 20ms を1 周期として切り替えるエンコードを実演しています。
    const int encodeFrameDurationCount = 4;
#if !defined(USE_SILK_CODING)
    const auto codingMode = nn::codec::OpusCodingMode_Celt;
    const int encodeFrameDurations[encodeFrameDurationCount] =
    {
//         // encodeFrameDuration を 5000 よりも小さい値とする場合、環境によっては
//         // 処理がリアルタイムに回らない可能性があるので、注意が必要です。
//          2500,
//          5000,
        10000,
        20000,
        10000,
        20000
    };
#else  // !defined(USE_SILK_CODING)
    const auto codingMode = nn::codec::OpusCodingMode_Silk;
    const int encodeFrameDurations[encodeFrameDurationCount] =
    {
        10000,
        20000,
        10000,
        20000
    };
#endif // !defined(USE_SILK_CODING)
    // 符号化モードは、EncodeInterleaved() 呼び出し前に指定することができます。
    encoder.BindCodingMode(codingMode);

    const int encodeSampleCountMaximum = encoder.CalculateFrameSampleCount(20000);
    auto encodedDataBuffer = new uint8_t[encodedDataSizeMaximum];
    auto decodedDataBuffer = new int16_t[encodeSampleCountMaximum * channelCount];
    auto inputDataBuffer = new int16_t[encodeSampleCountMaximum * channelCount];
    int sampleCountTotal = 0;
    int encodeFrameDurationIndex = 0;

    for (auto remaining = sampleCountPerChannel; ; )
    {
        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));

        // 再生が完了したバッファがあれば、Opus デコードを実行します
        if (auto pReleasedWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voice))
        {
            nn::audio::WaveBuffer* pWaveBuffer = nullptr;
            int16_t* pPcmBuffer = nullptr;
            for (int i = 0; i < BufferCount; ++i)
            {
                if (pReleasedWaveBuffer == &waveBuffer[i])
                {
                    pWaveBuffer = &waveBuffer[i];
                    pPcmBuffer = static_cast<int16_t*>(pcmBuffer[i]);
                    break;
                }
            }
            if (remaining > 0) // < sampleCountPerChannel / encodeSampleCount)
            {
                // ビットレートはエンコード単位で切り替えることができます。
                // 以下の処理では、10000 サンプルを処理する毎に
                // 低ビットレート/高ビットレートを切り替える実演をしています。
                if (sampleCountTotal >= 10000)
                {
                    int bitRate = encoder.GetBitRate();
                    if (HigherBitRate == bitRate)
                    {
                        encoder.SetBitRate(LowerBitRate);
                    }
                    else
                    {
                        encoder.SetBitRate(HigherBitRate);
                    }
                    sampleCountTotal -= 10000;
                }

                // エンコード
                const int encodeSampleCount = encoder.CalculateFrameSampleCount(encodeFrameDurations[encodeFrameDurationIndex++]);
                encodeFrameDurationIndex = encodeFrameDurationIndex % encodeFrameDurationCount;
                size_t copySampleCount = encodeSampleCount;
                // 端数の処理
                if (remaining < encodeSampleCount)
                {
                    copySampleCount = static_cast<int>(remaining);
                    std::memset(inputDataBuffer, 0x0, sizeof(int16_t) * channelCount * encodeSampleCount);
                }
                std::memcpy(inputDataBuffer, p, sizeof(int16_t) * channelCount * copySampleCount);
                size_t encodedDataSize = 0;
                nn::codec::OpusResult encodeStatus = encoder.EncodeInterleaved(
                    &encodedDataSize,
                    encodedDataBuffer,
                    encodedDataSizeMaximum,
                    inputDataBuffer,
                    encodeSampleCount);
                NN_ABORT_UNLESS(encodeStatus == nn::codec::OpusResult_Success);
                p += encodeSampleCount * channelCount;

                // デコード
                std::size_t consumed = 0;
                int sampleCount = 0;
                const int decodeBufferSize = encodeSampleCount * channelCount * sizeof(int16_t) * decoderSampleRate / encoderSampleRate;
                opusResult = decoder.DecodeInterleaved(
                    &consumed,
                    &sampleCount,
                    pPcmBuffer,
                    decodeBufferSize,
                    encodedDataBuffer,
                    encodedDataSize);
                NN_ABORT_UNLESS(opusResult == nn::codec::OpusResult_Success);
                NN_ABORT_UNLESS(consumed == encodedDataSize);

                // デコード結果を Voice の再々バッファに登録する。
                pWaveBuffer->size = sampleCount * sizeof(int16_t);
                pWaveBuffer->endSampleOffset = sampleCount;
                pWaveBuffer->loop = false;
                pWaveBuffer->isEndOfStream = false;
                nn::audio::AppendWaveBuffer(&voice, pWaveBuffer);

                remaining -= encodeSampleCount;
                sampleCountTotal += encodeSampleCount;

            }
            // データが残っていなければ、全てのバッファの再生が完了するまで待ちます
            else if (--bufferCount == 0)
            {
                break;
            }
        }
    }

    // バッファを解放する。
    for (int i = 0; i < BufferCount; ++i)
    {
        pcmBufferAllocator.Free(pcmBuffer[i]);
    }
    // 利用していたメモリプールにデタッチ要求をします。
    nn::audio::RequestDetachMemoryPool(&pcmBufferMemoryPool);
    // メモリプールがデタッチされるのを待ちます。
    while (nn::audio::IsMemoryPoolAttached(&pcmBufferMemoryPool))
    {
        nn::audio::GetReleasedWaveBuffer(&voice);

        systemEvent.Wait();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(audioRendererHandle, &config));
    }
    // MemoryPool を解放する。
    nn::audio::ReleaseMemoryPool(&config, &pcmBufferMemoryPool);

    // オーディオレンダラを終了し、メモリを破棄します
    nn::audio::StopAudioRenderer(audioRendererHandle);
    nn::audio::CloseAudioRenderer(audioRendererHandle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());
    allocator.Free(audioRendererWorkBuffer);
    allocator.Free(configBuffer);

    delete[] inputDataBuffer;
    delete[] decodedDataBuffer;
    delete[] encodedDataBuffer;
    delete[] pWaveDataHead;

    // Opus デコーダ/エンコーダを終了し、メモリを破棄します
    encoder.Finalize();
    decoder.Finalize();
    std::free(encoderWorkBuffer);
    std::free(decoderWorkBuffer);

    // ファイルシステムの終了処理を行います
    FinalizeFileSystem();

}  // NOLINT(readability/fn_size)
