﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{CdrSimple.cpp,PageSampleCdrSimple}
 *
 * @brief
 *  CDR ライブラリサンプルプログラム
 */

/**
 * @page PageSampleCdrSimple CdrSimple
 * @tableofcontents
 *
 * @brief
 *  CDR ライブラリの使用方法を例示したサンプルです。
 *
 * @section PageSampleCdrSimple_SectionBrief 概要
 *  ここでは、CDR ライブラリを使用した、ホスト側ツールの作成方法を説明します。
 *
 *  CDR ライブラリを使用したプロジェクトの作成方法については、
 *  @ref nn::cdr "CDR ライブラリ の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleCdrSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/CdrSimple Samples/Sources/Applications/CdrSimple @endlink 以下にあります。
 *
 * @section PageSampleCdrSimple_SectionNecessaryEnvironment 必要な環境
 *  あらかじめ nxdmp を作成しておく必要があります。
 *
 * @section PageSampleCdrSimple_SectionHowToOperate 操作方法
 *  1. 解析する nxdmp の入力待ち状態になるので、nxdmp ファイルのフルパスを指定してください
 *  2. nxdmp に対応する nss ファイルが存在するディレクトリの入力待ち状態になるので、パスを指定してください
 *
 * @section PageSampleCdrSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleCdrSimple_SectionDetail 解説
 *
 * @subsection PageSampleCdrSimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  CdrSimple.cpp
 *  @includelineno CdrSimple.cpp
 *
 * @subsection PageSampleCdrSimple_SectionSampleDetail サンプルプログラムの解説
 *  このサンプルプログラムの全体像は以下の通りです。
 *
 *  - nxdmp ファイルの入力を待ち、そのファイルをロードする
 *  - nss ファイルの存在するディレクトリの入力を待ち、nss 検索対象ディレクトリに指定する
 *  - スレッド情報を取得し、表示する
 *  - モジュール情報を取得し、表示する
 *  - nxdmp ファイルをクローズする
 *
 */

#include <memory>
#include "cdr.h"

#define RETURN_IF_FAILURE(operation) \
do { \
    nn::cdr::Error MACRO_ERROR = operation; \
    if (MACRO_ERROR != nn::cdr::Error_Ok) \
    { \
        printf("%s failed because of %d.\n", #operation, MACRO_ERROR); \
        return 1; \
    } \
} while (0)

int LoadNxdmp()
{
    char nxdmpPath[1024];
    printf("Put full path of nxdmp file to load.\n");
    scanf_s("%1023s", nxdmpPath, static_cast<unsigned int>(_countof(nxdmpPath)));
    getchar();
    RETURN_IF_FAILURE(nn::cdr::Open(nxdmpPath));
    return 0;
}

int LoadNss()
{
    char nssPath[1024];
    printf("Put full path of directory which has related nss files.\n");
    scanf_s("%1023s", nssPath, static_cast<unsigned int>(_countof(nssPath)));
    getchar();
    RETURN_IF_FAILURE(nn::cdr::AddSymbolPath(nssPath));
    return 0;
}

int ShowThreadInformation()
{
    int nThreads = 0;
    RETURN_IF_FAILURE(nn::cdr::GetThreadCount(&nThreads));
    printf("  Number of threads: %d\n", nThreads);

    std::unique_ptr<nn::cdr::Thread[]> pThreads(new nn::cdr::Thread[nThreads]);
    RETURN_IF_FAILURE(nn::cdr::GetThreads(pThreads.get(), &nThreads, nThreads));

    for (int iThread = 0; iThread < nThreads; ++iThread)
    {
        printf("  Thread[%d]: ThreadId: %lld, ThreadName: \"%s\", Core: %hd\n", iThread, pThreads[iThread].id, pThreads[iThread].name, pThreads[iThread].core);

        int nCallStackEntries = 0;
        RETURN_IF_FAILURE(nn::cdr::GetCallStackEntryCount(&nCallStackEntries, pThreads[iThread].id));

        std::unique_ptr<nn::cdr::CallStackEntry[]> pEntries(new nn::cdr::CallStackEntry[nCallStackEntries]);
        RETURN_IF_FAILURE(nn::cdr::GetCallStackEntries(pEntries.get(), &nCallStackEntries, nCallStackEntries, pThreads[iThread].id));

        for (int iEntry = 0; iEntry < nCallStackEntries; ++iEntry)
        {
            printf("   Call Stack[%d]:\n", iEntry);
            printf("    Function:    \"%s\"\n", pEntries[iEntry].function);
            printf("    IP:          0x%llx\n", pEntries[iEntry].instructionAddress);
            printf("    Line Number: %d\n", pEntries[iEntry].lineNumber);
            printf("    Module:      %d\n", pEntries[iEntry].moduleIndex);
            printf("    Source:      \"%s\"\n", pEntries[iEntry].source);

            nn::cdr::FilePath filePath;
            int lineNumber = 0;
            RETURN_IF_FAILURE(nn::cdr::GetSource(&filePath, &lineNumber, pEntries[iEntry].instructionAddress));
            printf("   GetSource() for the address 0x%llx: \"%s\" Line %d\n\n", pEntries[iEntry].instructionAddress, filePath.filePath, lineNumber);
        }
    }
    printf("\n");

    return 0;
}

int ShowModuleInformation()
{
    int nModules = 0;
    RETURN_IF_FAILURE(nn::cdr::GetModuleCount(&nModules));
    printf("  Number of modules: %d\n", nModules);

    std::unique_ptr<nn::cdr::Module[]> pModules(new nn::cdr::Module[nModules]);
    RETURN_IF_FAILURE(nn::cdr::GetModules(pModules.get(), &nModules, nModules));

    for (int iMod = 0; iMod < nModules; ++iMod)
    {
        printf("  Module[%d]:\n", iMod);

        // Custom printing logic to handle the build id format.
        printf("   ID:          ");
        for (int iDigit = 0; iDigit < sizeof(pModules[iMod].id); ++iDigit)
        {
            printf("%hhx", pModules[iMod].id[iDigit]);
        }
        printf("\n");

        printf("   Name:        \"%s\"\n", pModules[iMod].filePath);
        printf("   LoadAddress: 0x%llx\n", pModules[iMod].loadAddress);
        printf("   Size:        %lld\n\n", pModules[iMod].size);
    }
    printf("\n");

    return 0;
}

int main()
{
    if (LoadNxdmp() != 0)
    {
        return 1;
    }
    if (LoadNss() != 0)
    {
        return 1;
    }

    ShowThreadInformation();
    ShowModuleInformation();

    RETURN_IF_FAILURE(nn::cdr::Close());
    getchar();
    return 0;
}
