﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bluetooth_Client.h"
#include <cstdlib>

class myBtClient : public bluetoothClient
{
    public:

    nn::os::EventType setTsiEvent;
    nn::os::MutexType setTsiMutex;

    nn::bluetooth::BluetoothAddress tsiAddress;

    void setTsi(const nn::bluetooth::BluetoothAddress &bdAddr, int tsi)
    {
        //NN_LOG("  [ExtSetTsi] %d\n", tsi);
        nn::os::LockMutex(&setTsiMutex);
        auto result = nn::bluetooth::ExtSetTsi(&bdAddr, tsi);
        if (result.IsSuccess())
        {
            setControllerTsiState(bdAddr, tsi);
            for(;;)
            {
                auto timeout = nn::os::TimedWaitEvent(&setTsiEvent, nn::TimeSpan::FromMilliSeconds(1000));
                if (timeout == false)
                {
                    NN_LOG("SetTsi failed to receive callback after 1 second!\n");
                }
                else
                {
                    if (memcmp(tsiAddress.address, bdAddr.address, 6) == 0)
                    {
                        setControllerTsiState(bdAddr, tsi, 1);
                        break;
                    }
                }
            }
        }
        else
        {
            NN_LOG("SetTsi returned a failure:%d\n", result);
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
        }
        nn::os::UnlockMutex(&setTsiMutex);
    }

    private:

    void EventFromExtensionCallbacks(const nn::bluetooth::InfoFromExtensionCallbacks* pInfo)
    {
        const char* extStrP;
        switch (pInfo->eventType)
        {
            case nn::bluetooth::EventFromTsiSetCallback:    extStrP = "EventFromTsiSetCallback"; break;
            case nn::bluetooth::EventFromTsiExitCallback:   extStrP = "EventFromTsiExitCallback"; break;
            case nn::bluetooth::EventFromBurstSetCallback:  extStrP = "EventFromBurstSetCallback"; break;
            case nn::bluetooth::EventFromBurstExitCallback: extStrP = "EventFromBurstExitCallback"; break;
            case nn::bluetooth::EventFromSetZeroRetranCallback: extStrP = "EventFromSetZeroRetranCallback"; break;
            case nn::bluetooth::EventFromGetPendingConnectionsCallback: extStrP = "EventFromGetPendingConnectionsCallback"; break;
            default:
                NN_LOG("Invalid Extension Event: 0x%X\n", pInfo->eventType);
                extStrP = "UNKNOWN";
        }
        //NN_LOG("Extension(HID): %s\n", extStrP);

        if (pInfo->eventType == nn::bluetooth::EventFromTsiSetCallback ||
            pInfo->eventType == nn::bluetooth::EventFromTsiExitCallback)
        {
            tsiAddress = pInfo->bluetoothAddress;
            nn::os::SignalEvent(&setTsiEvent);
        }
        else if (pInfo->eventType == nn::bluetooth::EventFromSetZeroRetranCallback)
        {
            NN_LOG("Zero re-tx mode: %s\n", pInfo->zeroRetran.enable ? "ENABLED" : "DISABLED");
        }
        else if (pInfo->eventType == nn::bluetooth::EventFromGetPendingConnectionsCallback)
        {
            NN_LOG("Pending Connections: %d\n", pInfo->connections.numPending);
        }
    }
};

int randomInRange(int min, int max)
{
    return ( (rand()>>4) % (max + 1 - min) ) + min;
}

static myBtClient client;

//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG("NBM_CLIENT NOW RUNNING \n");

    while (1) {
    client.startBluetooth();

    nn::os::InitializeEvent(&client.setTsiEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeMutex(&client.setTsiMutex, false, 0);

    nn::bluetooth::ExtSetVisibility(false,true);

    client.initControllerInfo();

    int loopTimeMs=0;
    int goalTsi = 10;
    int interation = 0;

    for (int i=0; i< client.devicesPaired; i++)
    {
        nn::bluetooth::BluetoothAddress bdAddr;
        memcpy(bdAddr.address, client.devicesSettingsArray[i].bd_addr, 6);
        nn::bluetooth::HidWakeController(&bdAddr);
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(5000));
        client.setTsi(bdAddr, goalTsi);
    }
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(10000));

    for(;;)
    {
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));

        loopTimeMs += 100;

        if ((loopTimeMs % 1000) == 0)
        {
            // Set sniff-mode connected controllers to active mode.
            int connectedControllers = 0;
            for (int i=0; i<bluetoothClient::MAX_CTRL; i++)
            {
                if (client.controllerInfo[i].connectState == nn::bluetooth::BTHH_CONN_STATE_CONNECTED
                        && client.controllerInfo[i].valid
                        && client.controllerInfo[i].tsi != 0xFF
                        && client.controllerInfo[i].tsiChangeAcked)
                {
                    client.setTsi(client.controllerInfo[i].bdAddr, 0xFF);
                    connectedControllers++;
                }
            }

            // Set new target TSI
            if      (connectedControllers > 6) goalTsi = randomInRange(9,10);
            else if (connectedControllers > 4) goalTsi = randomInRange(7,10);
            else if (connectedControllers > 2) goalTsi = randomInRange(1,10);
            else                               goalTsi = randomInRange(0,10);

            NN_LOG("New TSI: %2d, Controllers Connected: %d, Test Iteration:%d\n", goalTsi, connectedControllers, ++interation);

        }

        // Set active-mode connected controllers to sniff mode.
        for (int i=0; i<bluetoothClient::MAX_CTRL; i++)
        {
            if (client.controllerInfo[i].connectState == nn::bluetooth::BTHH_CONN_STATE_CONNECTED
                    && client.controllerInfo[i].valid
                    && client.controllerInfo[i].tsi == 0xFF
                    && client.controllerInfo[i].tsiChangeAcked)
            {
                client.setTsi(client.controllerInfo[i].bdAddr, goalTsi);
            }
        }

        if (loopTimeMs == 30000)
        {
           loopTimeMs=0;
        }
    }

    client.finishBluetooth();

    nn::os::FinalizeMutex(&client.setTsiMutex);
    nn::os::FinalizeEvent(&client.setTsiEvent);
    }
}


