﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "bluetooth_Client.h"
#include <cstdlib>

struct BleGattAttribute
{
    nn::bluetooth::GattId               id;
    uint16_t                            handle;
    uint16_t                            endGroupHandle;     // used only for Gatt Service
    nn::bluetooth::GattAttributeType    type;
    uint8_t                             property;
    bool                                isPrimaryService;   // used only for Gatt Service
};

//static const nn::bluetooth::GattId gapServiceUuid       = { 0,{ 2,{ 0x1800 } } };
//static const nn::bluetooth::GattId deviceNameCharUuid   = { 0,{ 2,{ 0x2A00 } } };
//static const nn::bluetooth::GattId appearanceCharUuid   = { 0,{ 2,{ 0x2A01 } } };

static const uint8_t smallServiceUuidRawByte[]          = { 0x00, 0x00, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallReadCharUuidRawByte[]         = { 0x01, 0x01, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallWriteReqCharUuidRawByte[]     = { 0x02, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallWriteCmdCharUuidRawByte[]     = { 0x03, 0x03, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };
static const uint8_t smallNotifCharUuidRawByte[]        = { 0x04, 0x04, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x00 };

static const uint8_t largeServiceUuidRawByte[]          = { 0x10, 0x10, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeReadCharUuidRawByte[]         = { 0x11, 0x11, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeWriteReqCharUuidRawByte[]     = { 0x12, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeWriteCmdCharUuidRawByte[]     = { 0x13, 0x13, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };
static const uint8_t largeNotifCharUuidRawByte[]        = { 0x14, 0x14, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x10 };

class myBtClient : public bluetoothClient
{
    public:

    nn::bluetooth::GattAttributeUuid serverUuid;
    uint8_t serverIf = 0xff;
    uint16_t connectionId = 0xff;
    bool clientConnected = false;
    bool isSmallNotifEnabled = false;
    bool isLargeNotifEnabled = false;

    enum {
        SMALL_ATTRIBUTE_SIZE = 20,
        LARGE_ATTRIBUTE_SIZE = 100,
    };

    //-----------------------------------------------------------------------------
    // startBluetoothLowEnergyServerService
    //
    void startBluetoothLowEnergyServerService(nn::bluetooth::GattAttributeUuid uuid)
    {
        bleServerState = REGISTER_SERVER;

        while(bleServerState != DONE)
        {
            switch(bleServerState)
            {
                case REGISTER_SERVER:
                {
                    nn::bluetooth::RegisterLeServer(uuid);
                    bleServerState = WAIT;
                    break;
                }
                case CREATE_SMALL_SERVICE:
                {
                    smallService.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                    memcpy(smallService.id.uuid.uu.uuid128, smallServiceUuidRawByte, smallService.id.uuid.length);
                    smallService.isPrimaryService = true;
                    smallService.type = nn::bluetooth::GattAttributeType_Service;
                    smallService.id.instanceId = 0;

                    nn::bluetooth::CreateLeService(serverIf, smallService.id.uuid, 12, smallService.isPrimaryService);
                    bleServerState = WAIT;
                    break;
                }
                case ADD_SMALL_CHARS:
                {
                    if (addedSmallChars == 0)
                    {
                        smallReadChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(smallReadChar.id.uuid.uu.uuid128, smallReadCharUuidRawByte, smallReadChar.id.uuid.length);
                        smallReadChar.isPrimaryService = false;
                        smallReadChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        smallReadChar.property = nn::bluetooth::GattAttributeProperty_Read;
                        smallReadChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, smallService.id.uuid, smallReadChar.id.uuid, nn::bluetooth::GattAttributePermission_Read, smallReadChar.property);

                        for (int i = 0; i < SMALL_ATTRIBUTE_SIZE; ++i)
                        {
                            smallReadData[i] = i;
                        }

                        nn::bluetooth::user::SetLeResponse(serverIf, smallService.id.uuid, smallReadChar.id.uuid, smallReadData, sizeof(smallReadData));
                    }
                    else if (addedSmallChars == 1)
                    {
                        smallWriteReqChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(smallWriteReqChar.id.uuid.uu.uuid128, smallWriteReqCharUuidRawByte, smallWriteReqChar.id.uuid.length);
                        smallWriteReqChar.isPrimaryService = false;
                        smallWriteReqChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        smallWriteReqChar.property = nn::bluetooth::GattAttributeProperty_Write;
                        smallWriteReqChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, smallService.id.uuid, smallWriteReqChar.id.uuid, nn::bluetooth::GattAttributePermission_Write, smallWriteReqChar.property);

                        memset(smallWriteReqData, 0, sizeof(smallWriteReqData));
                    }
                    if (addedSmallChars == 2)
                    {
                        smallWriteCmdChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(smallWriteCmdChar.id.uuid.uu.uuid128, smallWriteCmdCharUuidRawByte, smallWriteCmdChar.id.uuid.length);
                        smallWriteCmdChar.isPrimaryService = false;
                        smallWriteCmdChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        smallWriteCmdChar.property = nn::bluetooth::GattAttributeProperty_WriteWithoutResponse;
                        smallWriteCmdChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, smallService.id.uuid, smallWriteCmdChar.id.uuid, nn::bluetooth::GattAttributePermission_Write, smallWriteCmdChar.property);

                        memset(smallWriteCmdData, 0, sizeof(smallWriteCmdData));
                    }
                    if (addedSmallChars == 3)
                    {
                        smallNotifChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(smallNotifChar.id.uuid.uu.uuid128, smallNotifCharUuidRawByte, smallNotifChar.id.uuid.length);
                        smallNotifChar.isPrimaryService = false;
                        smallNotifChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        smallNotifChar.property = nn::bluetooth::GattAttributeProperty_Notify | nn::bluetooth::GattAttributeProperty_Read;
                        smallNotifChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, smallService.id.uuid, smallNotifChar.id.uuid, nn::bluetooth::GattAttributePermission_Read, smallNotifChar.property);

                        memset(largeNotifData, 0x00, sizeof(largeNotifData));
                    }
                    else if (addedSmallChars == 4)
                    {
                        smallNotifCharDesc.id.uuid = nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid;
                        smallNotifChar.isPrimaryService = false;
                        smallNotifChar.type = nn::bluetooth::GattAttributeType_Descriptor;

                        nn::bluetooth::AddLeDescriptor(serverIf, smallService.id.uuid, smallNotifCharDesc.id.uuid, nn::bluetooth::GattAttributePermission_Read | nn::bluetooth::GattAttributePermission_Write);
                    }

                    bleServerState = WAIT;
                    break;
                }
                case CREATE_LARGE_SERVICE:
                {
                    largeService.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                    memcpy(largeService.id.uuid.uu.uuid128, largeServiceUuidRawByte, largeService.id.uuid.length);
                    largeService.isPrimaryService = true;
                    largeService.type = nn::bluetooth::GattAttributeType_Service;
                    largeService.id.instanceId = 0;

                    nn::bluetooth::CreateLeService(serverIf, largeService.id.uuid, 12, largeService.isPrimaryService);

                    bleServerState = WAIT;
                    break;
                }
                case ADD_LARGE_CHARS:
                {
                    if (addedLargeChars == 0)
                    {
                        largeReadChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(largeReadChar.id.uuid.uu.uuid128, largeReadCharUuidRawByte, largeReadChar.id.uuid.length);
                        largeReadChar.isPrimaryService = false;
                        largeReadChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        largeReadChar.property = nn::bluetooth::GattAttributeProperty_Read;
                        largeReadChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, largeService.id.uuid, largeReadChar.id.uuid, nn::bluetooth::GattAttributePermission_Read, largeReadChar.property);

                        for (int i = 0; i < LARGE_ATTRIBUTE_SIZE; ++i)
                        {
                            largeReadData[i] = i;
                        }

                        nn::bluetooth::user::SetLeResponse(serverIf, largeService.id.uuid, largeReadChar.id.uuid, largeReadData, sizeof(largeReadData));
                    }
                    else if (addedLargeChars == 1)
                    {
                        largeWriteReqChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(largeWriteReqChar.id.uuid.uu.uuid128, largeWriteReqCharUuidRawByte, largeWriteReqChar.id.uuid.length);
                        largeWriteReqChar.isPrimaryService = false;
                        largeWriteReqChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        largeWriteReqChar.property = nn::bluetooth::GattAttributeProperty_Write;
                        largeWriteReqChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, largeService.id.uuid, largeWriteReqChar.id.uuid, nn::bluetooth::GattAttributePermission_Write, largeWriteReqChar.property);

                        memset(largeWriteReqData, 0, sizeof(largeWriteReqData));
                    }
                    if (addedLargeChars == 2)
                    {
                        largeWriteCmdChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(largeWriteCmdChar.id.uuid.uu.uuid128, largeWriteCmdCharUuidRawByte, largeWriteCmdChar.id.uuid.length);
                        largeWriteCmdChar.isPrimaryService = false;
                        largeWriteCmdChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        largeWriteCmdChar.property = nn::bluetooth::GattAttributeProperty_WriteWithoutResponse;
                        largeWriteCmdChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, largeService.id.uuid, largeWriteCmdChar.id.uuid, nn::bluetooth::GattAttributePermission_Write, largeWriteCmdChar.property);

                        memset(largeWriteCmdData, 0, sizeof(largeWriteCmdData));
                    }
                    if (addedLargeChars == 3)
                    {
                        largeNotifChar.id.uuid.length = nn::bluetooth::GattAttributeUuidLength_128;
                        memcpy(largeNotifChar.id.uuid.uu.uuid128, largeNotifCharUuidRawByte, largeNotifChar.id.uuid.length);
                        largeNotifChar.isPrimaryService = false;
                        largeNotifChar.type = nn::bluetooth::GattAttributeType_Characteristic;
                        largeNotifChar.property = nn::bluetooth::GattAttributeProperty_Notify | nn::bluetooth::GattAttributeProperty_Read;
                        largeNotifChar.id.instanceId = 0;

                        nn::bluetooth::AddLeCharacteristic(serverIf, largeService.id.uuid, largeNotifChar.id.uuid, nn::bluetooth::GattAttributePermission_Read, largeNotifChar.property);

                        memset(largeNotifData, 0x00, sizeof(largeNotifData));
                    }
                    else if (addedLargeChars == 4)
                    {
                        largeNotifCharDesc.id.uuid = nn::bluetooth::ClientCharacteristicConfigurationDescriptorUuid;
                        largeNotifChar.isPrimaryService = false;
                        largeNotifChar.type = nn::bluetooth::GattAttributeType_Descriptor;

                        nn::bluetooth::AddLeDescriptor(serverIf, largeService.id.uuid, largeNotifCharDesc.id.uuid, nn::bluetooth::GattAttributePermission_Read | nn::bluetooth::GattAttributePermission_Write);
                    }

                    bleServerState = WAIT;
                    break;
                }
                case START_SERVICE:
                {
                    nn::bluetooth::LeAdvertiseData advData;

                    advData.dataMask = (0x00000001 << 2); // MANU

                    // Data length of manufacturer specific data?
                    advData.len = 20;
                    // Manufacturer ID, little endian
                    advData.data[0] = 0x53;
                    advData.data[1] = 0x05;
                    // Advertise packet format version

                    advData.data[2] = 0x00;
                    // Client ID
                    advData.data[3] = 0x00;
                    advData.data[4] = 0x01;
                    advData.data[5] = 0x00;
                    // Service ID
                    advData.data[6] = 0x00;
                    advData.data[7] = 0x01;
                    advData.data[8] = 0x00;

                    // Wake On BLE
                    advData.data[9] = 0x01;
                    // Paried davice's BD Address
                    advData.data[10] = 0x7C;
                    advData.data[11] = 0xBB;
                    advData.data[12] = 0x8A;
                    advData.data[13] = 0x79;
                    advData.data[14] = 0x50;
                    advData.data[15] = 0x68;
                    // Arbitrary Data
                    for (int i = 0; i < 4; ++i)
                    {
                        advData.data[16 + i] = i;
                    }

                    nn::bluetooth::SetLeAdvertiseData(advData);

                    nn::bluetooth::StartLeService(serverIf, smallService.id.uuid);
                    nn::bluetooth::StartLeService(serverIf, largeService.id.uuid);

                    bleServerState = SET_VISIBILITY;

                    break;
                }
                case SET_VISIBILITY:
                {
                    nn::bluetooth::SetLeVisibility(true, true);
                    bleServerState = DONE;
                    break;
                }
                case WAIT:
                {
                  nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1000));
                  break;
                }
                default:
                  break;
            }
        }
    }  // NOLINT(impl/function_size)

    void sendSmallNotification()
    {
        if (!isSmallNotifEnabled)
        {
            return;
        }

        for (int i = 0; i < sizeof(smallNotifData); ++i)
        {
            if (smallNotifData[i] != 0xFF)
            {
                smallNotifData[i]++;
                break;
            }
        }

        nn::bluetooth::user::LeSendIndication(serverIf, smallService.id.uuid, smallNotifChar.id.uuid, smallNotifData, sizeof(smallNotifData), false);
    }

    void sendLargeNotification()
    {
        if (!isLargeNotifEnabled)
        {
            return;
        }

        for (int i = 0; i < sizeof(largeNotifData); ++i)
        {
            if (largeNotifData[i] != 0xFF)
            {
                largeNotifData[i]++;
                break;
            }
        }

        nn::bluetooth::user::LeSendIndication(serverIf, largeService.id.uuid, largeNotifChar.id.uuid, largeNotifData, sizeof(largeNotifData), false);
    }

    private:

    nn::bluetooth::BluetoothAddress connectedDevice;

    BleGattAttribute smallService;
    BleGattAttribute smallReadChar;
    BleGattAttribute smallWriteReqChar;
    BleGattAttribute smallWriteCmdChar;
    BleGattAttribute smallNotifChar;
    BleGattAttribute smallNotifCharDesc;
    uint8_t addedSmallChars = 0;

    BleGattAttribute largeService;
    BleGattAttribute largeReadChar;
    BleGattAttribute largeWriteReqChar;
    BleGattAttribute largeWriteCmdChar;
    BleGattAttribute largeNotifChar;
    BleGattAttribute largeNotifCharDesc;
    uint8_t addedLargeChars = 0;

    uint8_t smallReadData[SMALL_ATTRIBUTE_SIZE];
    uint8_t smallWriteReqData[SMALL_ATTRIBUTE_SIZE];
    uint8_t smallWriteCmdData[SMALL_ATTRIBUTE_SIZE];
    uint8_t smallNotifData[SMALL_ATTRIBUTE_SIZE];
    uint8_t smallNotifDescData[2];
    uint8_t largeReadData[LARGE_ATTRIBUTE_SIZE];
    uint8_t largeWriteReqData[LARGE_ATTRIBUTE_SIZE];
    uint8_t largeWriteCmdData[LARGE_ATTRIBUTE_SIZE];
    uint8_t largeNotifData[LARGE_ATTRIBUTE_SIZE];
    uint8_t largeNotifDescData[2];

    enum
    {
        WAIT,
        REGISTER_SERVER,
        CREATE_SMALL_SERVICE,
        CREATE_LARGE_SERVICE,
        ADD_SMALL_CHARS,
        ADD_LARGE_CHARS,
        START_SERVICE,
        SET_VISIBILITY,
        DONE
    } bleServerState;

    void EventFromLeConnParamUpdateCallback(const nn::bluetooth::InfoFromLeConnParamUpdateCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);

        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("Error. Failed to update connection parameters.\n");
            return;
        }

        NN_LOG("  Connection ID: %d\n", pInfo->connId);
        NN_LOG("  Connection Interval: %d (* 1.25msec)\n", pInfo->interval);
        NN_LOG("  Slave Latency: %d\n", pInfo->slaveLatency);
        NN_LOG("  Supervision Timeout: %d\n", pInfo->supervisionTimeout);

        return;
    }

    void EventFromLeServerStateChangedCallback(const nn::bluetooth::InfoFromLeAppStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        NN_LOG("  Server %s\n", pInfo->isRegistered ? "Registered" : "Unregistered");

        if (pInfo->isRegistered)
        {
            serverIf = pInfo->serverIf;
            bleServerState = CREATE_SMALL_SERVICE;
        }
        else
        {
            serverIf = 0xFF;
            addedSmallChars = 0;
            addedLargeChars = 0;
            isSmallNotifEnabled = false;
            isLargeNotifEnabled = false;

            memset(smallReadData, 0, sizeof(smallReadData));
            memset(smallWriteReqData, 0, sizeof(smallWriteReqData));
            memset(smallWriteCmdData, 0, sizeof(smallWriteCmdData));
            memset(smallNotifDescData, 0, sizeof(smallNotifDescData));
            memset(largeReadData, 0, sizeof(largeReadData));
            memset(largeWriteReqData, 0, sizeof(largeWriteReqData));
            memset(largeWriteCmdData, 0, sizeof(largeWriteCmdData));
            memset(largeNotifDescData, 0, sizeof(largeNotifDescData));
        }
    }

    void EventFromLeServerConnStateChangedCallback(const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        switch (pInfo->connState)
        {
        case nn::bluetooth::BLE_CONN_STATE_CONNECTED:
            NN_LOG("  Connected to BLE GATT Client\n");
            NN_LOG("  Server: %d    Conn Id: %d\n", pInfo->serverIf, pInfo->connId);
            NN_LOG("  Client Address: %s\n", toHexString(pInfo->address));

            connectedDevice = pInfo->address;
            connectionId = pInfo->connId;
            clientConnected = true;

            break;
        case nn::bluetooth::BLE_CONN_STATE_DISCONNECTED:
            NN_LOG("  Disconnected from BLE GATT Client\n");
            NN_LOG("  Server: %d    Conn Id: %d\n", pInfo->serverIf, pInfo->connId);
            NN_LOG("  Client Address: %s\n", toHexString(pInfo->address));
            NN_LOG("  Reason: %d\n", pInfo->reason);

            clientConnected = false;
            memset(connectedDevice.address, 0, 6);

            isSmallNotifEnabled = false;
            isLargeNotifEnabled = false;

            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    void EventFromLeServerProfileChangedCallback(const nn::bluetooth::InfoFromLeServerProfileChangedCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d\n", pInfo->status);
            return;
        }

        switch (pInfo->type)
        {
        case nn::bluetooth::GattAttributeType_IncludedService:
            NN_LOG("  Created include service.\n");
            NN_LOG("    Server If: %d\n", pInfo->serverIf);
            NN_LOG("    Service Handle: %d\n", pInfo->serviceHandle);
            NN_LOG("    Is Primary Service: %s\n", pInfo->isPrimaryService ? "TRUE" : "FALSE");
            break;
        case nn::bluetooth::GattAttributeType_Characteristic:
            NN_LOG("Added characteristics.\n");
            NN_LOG("    Server If: %d\n", pInfo->serverIf);
            NN_LOG("    Service Handle: %d\n", pInfo->serviceHandle);
            NN_LOG("    Characteristic Handle: %d\n", pInfo->attributeHandle);
            NN_LOG("    Property: %d\n", pInfo->property);

            break;
        case nn::bluetooth::GattAttributeType_Descriptor:
            NN_LOG("Added descriptor.\n");
            NN_LOG("    Server If: %d\n", pInfo->serverIf);
            NN_LOG("    Service Handle: %d\n", pInfo->serviceHandle);
            NN_LOG("    Descriptor Handle: %d\n", pInfo->attributeHandle);
            break;
        case nn::bluetooth::GattAttributeType_Service:
            NN_LOG("Created service.\n");
            NN_LOG("    Server If: %d\n", pInfo->serverIf);
            NN_LOG("    Service Handle: %d\n", pInfo->serviceHandle);
            NN_LOG("    Is Primary Service: %s\n", pInfo->isPrimaryService ? "TRUE" : "FALSE");

            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }

        NN_LOG("    UUID: ");
        if(pInfo->uuid.length == 2)        NN_LOG("0x%04X\n", pInfo->uuid.uu.uuid16);
        else if (pInfo->uuid.length == 4)  NN_LOG("0x%08X\n", pInfo->uuid.uu.uuid32);
        else if (pInfo->uuid.length == 16)
        {
            for (int i = 0; i < pInfo->uuid.length; ++i)
            {
                NN_LOG("%02X ", pInfo->uuid.uu.uuid128[i]);
            }
            NN_LOG("\n");
        }

        if (memcmp(&pInfo->uuid, &smallService.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            smallService.handle = pInfo->attributeHandle;

            bleServerState = ADD_SMALL_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &smallReadChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            smallReadChar.handle = pInfo->attributeHandle;
            addedSmallChars++;

            bleServerState = ADD_SMALL_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &smallWriteReqChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            smallWriteReqChar.handle = pInfo->attributeHandle;
            addedSmallChars++;

            bleServerState = ADD_SMALL_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &smallWriteCmdChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            smallWriteCmdChar.handle = pInfo->attributeHandle;
            addedSmallChars++;

            bleServerState = ADD_SMALL_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &smallNotifChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            smallNotifChar.handle = pInfo->attributeHandle;
            addedSmallChars++;

            bleServerState = ADD_SMALL_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &smallNotifCharDesc.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0 && pInfo->attributeHandle - 1 == smallNotifChar.handle)
        {
            smallNotifCharDesc.handle = pInfo->attributeHandle;
            addedSmallChars++;

            bleServerState = CREATE_LARGE_SERVICE;
        }

        if (memcmp(&pInfo->uuid, &largeService.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            largeService.handle = pInfo->attributeHandle;

            bleServerState = ADD_LARGE_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &largeReadChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            largeReadChar.handle = pInfo->attributeHandle;
            addedLargeChars++;

            bleServerState = ADD_LARGE_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &largeWriteReqChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            largeWriteReqChar.handle = pInfo->attributeHandle;
            addedLargeChars++;

            bleServerState = ADD_LARGE_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &largeWriteCmdChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            largeWriteCmdChar.handle = pInfo->attributeHandle;
            addedLargeChars++;

            bleServerState = ADD_LARGE_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &largeNotifChar.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0)
        {
            largeNotifChar.handle = pInfo->attributeHandle;
            addedLargeChars++;

            bleServerState = ADD_LARGE_CHARS;
        }
        else if (memcmp(&pInfo->uuid, &largeNotifCharDesc.id.uuid, sizeof(nn::bluetooth::GattAttributeUuid)) == 0 && pInfo->attributeHandle - 1 == largeNotifChar.handle)
        {
            largeNotifCharDesc.handle = pInfo->attributeHandle;
            addedLargeChars++;

            bleServerState = START_SERVICE;
        }

        if (pInfo->serviceHandle == smallService.handle && pInfo->attributeHandle > smallService.endGroupHandle)
        {
            smallService.endGroupHandle = pInfo->attributeHandle;
        }
        else if (pInfo->serviceHandle == largeService.handle && pInfo->attributeHandle > largeService.endGroupHandle)
        {
            largeService.endGroupHandle = pInfo->attributeHandle;
        }

    } // NOLINT(impl/function_size)

    void EventFromLeServerGattReqCallback(const nn::bluetooth::InfoFromLeServerGattReqCallback* pInfo)
    {
        NN_LOG("*****[%s]*****\n", __func__);
        if (pInfo->status != nn::bluetooth::BTHH_OK || pInfo->responseStatus != nn::bluetooth::BTHH_OK)
        {
            NN_LOG("ERROR. status: %d    response status: %d\n", pInfo->status, pInfo->responseStatus);
            return;
        }

        NN_LOG("    ConnID: %d\n", pInfo->connId);
        NN_LOG("    Request Type: %d\n", pInfo->requestType);
        NN_LOG("    Service Handle: %d    Attribute Handle: %d\n", pInfo->serviceHandle, pInfo->attributeHandle);

        uint8_t* pBuffer;
        int bufferSize = 0;

        if(pInfo->requestType == nn::bluetooth::BleGattReqType::BLE_GATT_REQ_TYPE_READ)
        {
            int i = 0;
            NN_LOG("    Read Data:\n");
            for (i = 0; i < pInfo->length; ++i)
            {
                NN_LOG("%02X ", pInfo->data[i]);

                if (i % 16 == 15)    NN_LOG("\n");
            }
            if (i % 16 != 15)   NN_LOG("\n");
            NN_LOG("    Offset: %d\n", pInfo->offset);
        }
        else if (pInfo->requestType == nn::bluetooth::BleGattReqType::BLE_GATT_REQ_TYPE_WRITE)
        {
            if (pInfo->attributeHandle == smallWriteReqChar.handle)
            {
                pBuffer = smallWriteReqData;
                bufferSize = sizeof(smallWriteReqData);
            }
            else if (pInfo->attributeHandle == smallWriteCmdChar.handle)
            {
                pBuffer = smallWriteCmdData;
                bufferSize = sizeof(smallWriteCmdData);
            }
            else if (pInfo->attributeHandle == smallNotifCharDesc.handle)
            {
                pBuffer = smallNotifDescData;
                bufferSize = sizeof(smallNotifDescData);
            }
            else if (pInfo->attributeHandle == largeWriteReqChar.handle)
            {
                pBuffer = largeWriteReqData;
                bufferSize = sizeof(largeWriteReqData);
            }
            else if (pInfo->attributeHandle == largeWriteCmdChar.handle)
            {
                pBuffer = largeWriteCmdData;
                bufferSize = sizeof(largeWriteCmdData);
            }
            else if (pInfo->attributeHandle == largeNotifCharDesc.handle)
            {
                pBuffer = largeNotifDescData;
                bufferSize = sizeof(largeNotifDescData);
            }

            int i = 0;
            NN_LOG("    Written Data:\n");
            for (i = 0; i < pInfo->length; ++i)
            {
                NN_LOG("%02X ", pInfo->data[i]);

                if (i % 16 == 15)    NN_LOG("\n");
            }
            if (i % 16 != 15)   NN_LOG("\n");
            NN_LOG("    Offset: %d\n", pInfo->offset);

            if (pInfo->length + pInfo->offset <= bufferSize)
            {
                memcpy(pBuffer, pInfo->data + pInfo->offset, pInfo->length);

                NN_LOG("    Buffer Data:\n");
                for (i = 0; i < bufferSize; ++i)
                {
                    NN_LOG("%02X ", pBuffer[i]);

                    if (i % 16 == 15)    NN_LOG("\n");
                }
                if (i % 16 != 15)   NN_LOG("\n");

                if (pInfo->attributeHandle == smallNotifCharDesc.handle && pBuffer[0] == 0x01 && pBuffer[1] == 0x00)
                {
                    NN_LOG("    Small Notification Enabled.\n");
                    isSmallNotifEnabled = true;
                }
                else if (pInfo->attributeHandle == smallNotifCharDesc.handle && pBuffer[0] == 0x00 && pBuffer[1] == 0x00)
                {
                    NN_LOG("    Small Notification Disabled.\n");
                    isSmallNotifEnabled = false;
                }
                else if (pInfo->attributeHandle == largeNotifCharDesc.handle && pBuffer[0] == 0x01 && pBuffer[1] == 0x00)
                {
                    NN_LOG("    Large Notification Enabled.\n");
                    isLargeNotifEnabled = true;
                }
                else if (pInfo->attributeHandle == largeNotifCharDesc.handle && pBuffer[0] == 0x00 && pBuffer[1] == 0x00)
                {
                    NN_LOG("    Large Notification Disabled.\n");
                    isLargeNotifEnabled = false;
                }
            }
            else
            {
                NN_LOG("Error: data too large. Buffer Size %d, Written Data Size: %d, Offset: %d\n", bufferSize, pInfo->length, pInfo->offset);
            }
        }

    } // NOLINT(impl/function_size)
};


static myBtClient client;

//-----------------------------------------------------------------------------
extern "C" void nnMain()
{
    NN_LOG("NBM_CLIENT NOW RUNNING \n");

    client.serverUuid.length = nn::bluetooth::GattAttributeUuidLength_16;
    client.serverUuid.uu.uuid16 = 0xFEED;

    client.startBluetooth();
    client.startBluetoothLowEnergy();
    client.startBluetoothLowEnergyServerService(client.serverUuid);

    int loopTimeMs=0;



    for(;;)
    {
        if (loopTimeMs % 3000 == 0)
        {
            client.sendSmallNotification();
            client.sendLargeNotification();
        }

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
        loopTimeMs += 100;
        if(loopTimeMs == 60000)
        {
           loopTimeMs=0;
        }
    }

    client.finishBluetooth();
}


