﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/bluetooth/bluetooth_Driver.h>
#include <nn/bluetooth/bluetooth_Api.h>
#include <nn/bluetooth/bluetooth_UserApi.h>
#include <nn/bluetooth/bluetooth_IBluetoothDriver.sfdl.h>
#include <nn/bluetooth/bluetooth_ServiceName.h>

class bluetoothClient
{
    private:
        static const size_t     ThreadStackSize = 12288;

        NN_ALIGNAS(4096) char   threadStack[ ThreadStackSize ];
        nn::os::ThreadType      threadForWaitEventSignals;

        nn::os::SystemEventType     systemEventForExitCore;
        nn::os::MultiWaitType       multiWaitForCore;
        nn::os::MultiWaitHolderType holderForCore;
        nn::os::MultiWaitHolderType holderForExitCore;
        static void WaitEventSignals(void* arg);

        NN_ALIGNAS(4096) char   threadStackForHid[ ThreadStackSize ];
        nn::os::ThreadType      threadForWaitEventSignalsForHid;

        nn::os::SystemEventType     systemEventForExitHid;
        nn::os::MultiWaitType       multiWaitForHid;
        nn::os::MultiWaitHolderType holderForHid;
        nn::os::MultiWaitHolderType holderForExitHid;
        nn::os::MultiWaitHolderType holderForHidReport;
        static void WaitEventSignalsForHid(void* arg);

        NN_ALIGNAS(4096) char   threadStackForBle[ ThreadStackSize ];
        nn::os::ThreadType      threadForWaitEventSignalsForBle;

        nn::os::SystemEventType     systemEventForExitBle;
        nn::os::MultiWaitType       multiWaitForBle;
        nn::os::MultiWaitHolderType holderForBle;
        nn::os::MultiWaitHolderType holderForBleCore;
        nn::os::MultiWaitHolderType holderForBleHid;
        nn::os::MultiWaitHolderType holderForExitBle;
        static void WaitEventSignalsForBle(void* arg);


    public:
        nn::os::SystemEventType     systemEventForInputReport;
        nn::os::SystemEventType     systemEventForHid;
        nn::os::SystemEventType     systemEventForCore;
        nn::os::SystemEventType     systemEventForBle;
        nn::os::SystemEventType     systemEventForBleCore;
        nn::os::SystemEventType     systemEventForBleHid;

        enum {
            NOT_PRIMARY = 0,
            PRIMARY
        };

        enum {
            AUTH_REQ_NONE = 0,
            AUTH_REQ_NO_MITM,
            AUTH_REQ_MITM,
            AUTH_REQ_SIGNED_NO_MITIM,
            AUTH_REQ_SIGNED_MITIM,
        };

        enum { NOT_BONDING, CANCELING_DISCOVERY, CREATING_BOND } bondingState;
        nn::bluetooth::BluetoothAddress bondingAddr;

        bool fatalErrorOccurred = false;

        int devicesPaired = 0;
        nn::settings::system::BluetoothDevicesSettings devicesSettingsArray[nn::settings::system::BluetoothDevicesSettingsCountMax];

        enum { MAX_CTRL = 10 };
        struct {
            bool                            valid;
            int                             hidCount;
            int                             tsi;
            bool                            tsiChangeAcked;
            nn::bluetooth::BluetoothAddress bdAddr;
            nn::bluetooth::BluetoothHhConnectionState      connectState;
        } controllerInfo[MAX_CTRL];

        void initControllerInfo();
        void printControllerHidCount();
        void setControllerInfoState(const nn::bluetooth::BluetoothAddress &BdAddr, nn::bluetooth::BluetoothHhConnectionState connectState);
        void setControllerHidCount(const nn::bluetooth::BluetoothAddress &BdAddr, int increment = 1);
        void setControllerTsiState(const nn::bluetooth::BluetoothAddress &BdAddr, int tsi, bool ack = 0);

        int  startBluetooth(void);
        void finishBluetooth(void);
        void restartBluetooth(void);

        void startBluetoothLowEnergy();
        void finishBluetoothLowEnergy();
        void startBluetoothLowEnergyClient(nn::bluetooth::GattAttributeUuid uuid);
        void startBluetoothLowEnergyServer(nn::bluetooth::GattAttributeUuid uuid);

        void AddPairedDevices();
        void GetPairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr);
        void RemovePairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr);

        void setTsi(const nn::bluetooth::BluetoothAddress &bdAddr, int tsi);

        static bool        isNintendoGamepad(const nn::bluetooth::Btbdname &name);
        static const char* toHexString(uint16_t length, const uint8_t buffer[]);
        static const char* toHexString(const nn::bluetooth::BluetoothAddress &BdAddress);

        // core events
        virtual void EventFromDeviceFoundCallback(              const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo);
        virtual void EventFromPinRequestCallback(               const nn::bluetooth::InfoFromPinRequestCallback* pInfo);
        virtual void EventFromDiscoveryStateChangedCallback(    const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo);
        virtual void EventFromBondStateChangedCallback(         const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo);

        virtual void EventFromLowEnergyCallbacks(               const nn::bluetooth::InfoFromLowEnergyCallbacks* pInfo);

        virtual void EventFromLeConnParamUpdateCallback(        const nn::bluetooth::InfoFromLeConnParamUpdateCallback* pInfo) { return; }

        virtual void EventFromLeScanStateChangedCallback(       const nn::bluetooth::InfoFromLeScanStateChangedCallback* pInfo) { return; }
        virtual void EventFromLeScanFilterStateChangedCallback( const nn::bluetooth::InfoFromLeScanFilterStateChangedCallback* pInfo) { return; }

        virtual void EventFromLeClientStateChangedCallback(             const nn::bluetooth::InfoFromLeAppStateChangedCallback* pInfo) { return; }
        virtual void EventFromLeClientConnStateChangedCallback(         const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo) { return; }
        virtual void EventFromLeClientGattOperationCallback(            const nn::bluetooth::InfoFromLeGattOperationCallback* pInfo) { return; }
        virtual void EventFromLeClientGattSrvcDiscCallback(             const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback* pInfo) { return; }
        virtual void EventFromLeClientGattSrvcDiscStateChangedCallback( const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback* pInfo) { return; }
        virtual void EventFromLeClientMtuConfigCallback(                const nn::bluetooth::InfoFromLeClientMtuConfigurationCallback* pInfo) { return; }

        virtual void EventFromLeServerStateChangedCallback(     const nn::bluetooth::InfoFromLeAppStateChangedCallback* pInfo) { return; }
        virtual void EventFromLeServerConnStateChangedCallback( const nn::bluetooth::InfoFromLeConnStateChangedCallback* pInfo) { return; }
        virtual void EventFromLeServerProfileChangedCallback(   const nn::bluetooth::InfoFromLeServerProfileChangedCallback* pInfo) { return; }
        virtual void EventFromLeServerGattReqCallback(          const nn::bluetooth::InfoFromLeServerGattReqCallback* pInfo) { return; }

        // hid events
        virtual void EventFromConnectionStateCallback(       const nn::bluetooth::InfoFromConnectionStateCallback* pInfo);
        virtual void EventFromExtensionCallbacks(            const nn::bluetooth::InfoFromExtensionCallbacks* pInfo);
        virtual void EventFromGetReportDataCallback(         const nn::bluetooth::InfoFromGetReportCallback* pInfo);
        virtual void EventFromSetReportStatusCallback(       const nn::bluetooth::InfoFromSetReportStatusCallback* pInfo);
        virtual void EventFromGetReportStatusCallback(       const nn::bluetooth::InfoFromGetReportStatusCallback* pInfo);

};
