﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
// This is skeleton code for bluetooth client process.
//-----------------------------------------------------------------------------
#include "bluetooth_Client.h"

#include <cstdlib>
#include <cstring>
#include <cstdio>

enum {CLIENT_VERSION = 1};

enum EventType
{
    EVENTTYPE_FROM_CORE = 1,
    EVENTTYPE_FROM_HID,
    EVENTTYPE_FROM_CLIENT,
    EVENTTYPE_FROM_HID_REPORT,
    EVENTTYPE_FROM_BLE,
    EVENTTYPE_FROM_BLE_CORE,
    EVENTTYPE_FROM_BLE_HID,
};



// PRINT (OR DON'T) TIMING INFORMATION
//-----------------------------------------------------------------------------
#if 0
    static nn::os::Tick startTime;
    void START_TIMER()
    {
        startTime = nn::os::GetSystemTick();
    }
    void STOP_TIMER(const char *comment)
    {
        nn::TimeSpan timespan = (nn::os::GetSystemTick() - startTime).ToTimeSpan();

        int64_t ms = timespan.GetMicroSeconds();
        NN_LOG("......................................................... %s %lld us\n", comment, ms);
    }
#else
    #define START_TIMER() (void*)0
    #define STOP_TIMER(str) (void*)0
#endif
//-----------------------------------------------------------------------------


#if 0
#define DEBUG_CONTROLLER_STATE \
        NN_LOG("%s: ctrl:%d state:%d tsi:%d acked:%d valid:%d\n", \
                __func__, k, \
                controllerInfo[k].connectState, \
                controllerInfo[k].tsi, \
                controllerInfo[k].tsiChangeAcked, \
                controllerInfo[k].valid)
#else
#define DEBUG_CONTROLLER_STATE
#endif


bool bluetoothClient::isNintendoGamepad(const nn::bluetooth::Btbdname &name)
{
    static const char ctrlName[] = "Joy-Con (";
    return(0 == memcmp(name.name, ctrlName, strlen(ctrlName)));
}


//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    nn::os::SetMemoryHeapSize( nn::os::MemoryBlockUnitSize * 4 );
}



//-----------------------------------------------------------------------------
// toHexString: convert uint8_t to string
//
const char* bluetoothClient::toHexString(uint16_t length, const uint8_t buffer[])
{
    static char       hexString[128];                // for output
    static const char hex[] = "0123456789ABCDEF";    // for conversion

    int i = 0;
    while(length && i < sizeof(hexString) - 3)
    {
        hexString[i++] = hex[(*buffer) >> 4];
        hexString[i++] = hex[(*buffer++) & 0xF];
        hexString[i++] = ':';
        length--;
    }

    hexString[i - 1] = 0;  // null-terminate and remove the trailing ":"

    return hexString;
}

//-----------------------------------------------------------------------------
// toHexString: convert BluetoothAddress to string
//
const char* bluetoothClient::toHexString(const nn::bluetooth::BluetoothAddress &BdAddress)
{
    return toHexString(6, BdAddress.address);
}

//-----------------------------------------------------------------------------
void bluetoothClient::initControllerInfo()
{
    memset(&controllerInfo, 0, sizeof(controllerInfo));

    for (int i=0; i<MAX_CTRL; i++)
    {
        controllerInfo[i].connectState   = nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED;
        controllerInfo[i].tsi            = 0xFF;
        controllerInfo[i].tsiChangeAcked = true;
        controllerInfo[i].hidCount       = 0;
        controllerInfo[i].valid          = false;
    }
}

//-----------------------------------------------------------------------------
void bluetoothClient::setControllerTsiState(const nn::bluetooth::BluetoothAddress &bdAddr, int tsi, bool acked)
{
    for (int k=0; k<MAX_CTRL; k++)
    {
        if (controllerInfo[k].valid && !memcmp(controllerInfo[k].bdAddr.address, bdAddr.address, 6))
        {
            if (acked)
            {
                // ignore the tsi param since it isn't valid in this use case
                controllerInfo[k].tsiChangeAcked = true;
            }

            else
            {
                controllerInfo[k].tsi        = tsi;
                controllerInfo[k].tsiChangeAcked = false;
            }

            DEBUG_CONTROLLER_STATE;
            return;
        }
    }
}

//-----------------------------------------------------------------------------
void bluetoothClient::setControllerInfoState(const nn::bluetooth::BluetoothAddress &bdAddr, nn::bluetooth::BluetoothHhConnectionState connectState)
{
    int k;

    // find the bd_addr in controllerInfo
    for (k=0; k<MAX_CTRL; k++)
    {
        if (controllerInfo[k].valid && !memcmp(controllerInfo[k].bdAddr.address, bdAddr.address, 6))
        {
            break;
        }
    }

    // if connecting and not in the database, add it (if not full)
    if (connectState == nn::bluetooth::BTHH_CONN_STATE_CONNECTED && k >= MAX_CTRL)
    {
        for (k=0; k<MAX_CTRL; k++)
        {
            if (!controllerInfo[k].valid)
            {
                controllerInfo[k].bdAddr       = bdAddr;
                controllerInfo[k].valid        = true;
                break;
            }
        }
    }

    // if in the database, update the state.
    if (k < MAX_CTRL)
    {
        controllerInfo[k].connectState = connectState;
        if (connectState == nn::bluetooth::BTHH_CONN_STATE_CONNECTED) {
            controllerInfo[k].tsiChangeAcked = true;
            controllerInfo[k].tsi = 0xFF;
        }

        DEBUG_CONTROLLER_STATE;
    }
}

//-----------------------------------------------------------------------------
void bluetoothClient::setControllerHidCount(const nn::bluetooth::BluetoothAddress &bdAddr, int increment)
{
    for (int k=0; k<MAX_CTRL; k++)
    {
        if (controllerInfo[k].valid && !memcmp(controllerInfo[k].bdAddr.address, bdAddr.address, 6))
        {
            controllerInfo[k].hidCount += increment;
            return;
        }
    }
}


//-----------------------------------------------------------------------------
void bluetoothClient::printControllerHidCount()
{
    char printBuf[100];
    int  index = 0;
    int  totalHid = 0;
    for (int k=0; k<MAX_CTRL; k++)
    {
        if (controllerInfo[k].valid && controllerInfo[k].connectState == nn::bluetooth::BTHH_CONN_STATE_CONNECTED)
            index += snprintf(printBuf + index, sizeof(printBuf) - index - 1, " %3d", controllerInfo[k].hidCount);
        else
            index += snprintf(printBuf + index, sizeof(printBuf) - index - 1, " %3s", "-");
        totalHid += controllerInfo[k].hidCount;
        controllerInfo[k].hidCount = 0;
    }

    if (totalHid > 0)
    {
        printBuf[index] = 0;
        NN_LOG("HID Stats: %s\n", printBuf);
    }
}


//-----------------------------------------------------------------------------
// EventFromDeviceFoundCallback
//
void bluetoothClient::EventFromDeviceFoundCallback(const nn::bluetooth::InfoFromDeviceFoundCallback* pInfo)
{
    nn::bluetooth::BluetoothAddress address;
    memcpy(address.address, pInfo->BdAddress.address, nn::bluetooth::AddressLength);
    NN_LOG("DeviceFound: %s, %s\n", pInfo->BdName.name, toHexString(address));
    if (isNintendoGamepad(pInfo->BdName))
    {
        NN_LOG("  [CancelDiscovery]\n");
        nn::bluetooth::CancelDiscovery();

        bondingState = CANCELING_DISCOVERY;
        memcpy(bondingAddr.address, pInfo->BdAddress.address, nn::bluetooth::AddressLength);
    }
}

//-----------------------------------------------------------------------------
// EventFromPinRequestCallback
//
void bluetoothClient::EventFromPinRequestCallback(const nn::bluetooth::InfoFromPinRequestCallback* pInfo)
{
    NN_LOG("PinRequest: %s\n",
            toHexString(sizeof(nn::bluetooth::InfoFromPinRequestCallback), reinterpret_cast<const uint8_t*>(pInfo)));
    //Pin request is came.
    //Reply for it.
    nn::bluetooth::BluetoothPinCode pinCode = { .pin = { 0x30, 0x30, 0x30, 0x30 } };
    NN_LOG("  [PinReply]\n");
    nn::bluetooth::PinReply(&pInfo->bluetoothAddress, 1, &pinCode, 4);
}

//-----------------------------------------------------------------------------
// EventFromDiscoveryStateChangedCallback
//
void bluetoothClient::EventFromDiscoveryStateChangedCallback( const nn::bluetooth::InfoFromDiscoveryStateChangedCallback* pInfo)
{
    const char* discStrP;
    switch (pInfo->state)
    {
        case nn::bluetooth::BT_DISCOVERY_STOPPED:  discStrP = "BT_DISCOVERY_STOPPED"; break;
        case nn::bluetooth::BT_DISCOVERY_STARTED:  discStrP = "BT_DISCOVERY_STARTED"; break;
        default:
            NN_LOG("ERROR: Invalid DiscState: 0x%X\n", pInfo->state);
            discStrP = "UNKNOWN!";
    }

    NN_LOG("DiscoveryStateChanged: %s\n", discStrP);

    if (pInfo->state == nn::bluetooth::BT_DISCOVERY_STOPPED && bondingState == CANCELING_DISCOVERY)
    {
        NN_LOG("  [CreateBond]\n");
        nn::bluetooth::CreateBond(&bondingAddr, nn::bluetooth::NoPreference);
        bondingState = CREATING_BOND;
    }
}

//-----------------------------------------------------------------------------
// EventFromBondStateChangedCallback
//
void bluetoothClient::EventFromBondStateChangedCallback( const nn::bluetooth::InfoFromBondStateChangedCallback* pInfo)
{
    const char* bondStrP;
    switch (pInfo->state)
    {
        case nn::bluetooth::BT_BOND_STATE_NONE:       bondStrP = "BT_BOND_STATE_NONE"; break;
        case nn::bluetooth::BT_BOND_STATE_BONDING:    bondStrP = "BT_BOND_STATE_BONDING"; break;
        case nn::bluetooth::BT_BOND_STATE_BONDED:     bondStrP = "BT_BOND_STATE_BONDED"; break;
        default:
            NN_LOG("ERROR: Invalid BondState: 0x%X\n", pInfo->status);
            bondStrP = "UNKNOWN!";
    };
    NN_LOG("BondStateChanged: %s %s\n", bondStrP, toHexString(pInfo->bluetoothAddress));

    if(CLIENT_VERSION != 0)
    {
        if (pInfo->state == nn::bluetooth::BT_BOND_STATE_BONDED)
        {
            GetPairedDevice(pInfo->bluetoothAddress);

            // Save paired device information to nand
            nn::settings::system::SetBluetoothDevicesSettings(devicesSettingsArray, devicesPaired);
            NN_LOG("Paired Bluetooth devices: %d\n", devicesPaired);
        }
    }

    if (pInfo->state == nn::bluetooth::BT_BOND_STATE_NONE ||
        pInfo->state == nn::bluetooth::BT_BOND_STATE_BONDED)
    {
        bondingState = NOT_BONDING;
    }
}

//-----------------------------------------------------------------------------
// EventFromLowEnergyCallbacks
//

void bluetoothClient::EventFromLowEnergyCallbacks( const nn::bluetooth::InfoFromLowEnergyCallbacks* pInfo)
{
    NN_LOG("***** %s *****\n", __func__);
    NN_LOG("Event: %d\n", pInfo->eventType);

    return;
}

//-----------------------------------------------------------------------------
// WaitEventSignals: thread for handling core events
//
void bluetoothClient::WaitEventSignals(void *arg)
{
    bluetoothClient *myclient = reinterpret_cast<bluetoothClient*>(arg);

    nn::bluetooth::EventType eventType;

    uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT];

    while(true)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&myclient->multiWaitForCore);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
            case EVENTTYPE_FROM_CORE:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForCore);

                nn::bluetooth::GetEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_CORE_OUT);

                switch(eventType)
                {
                    case nn::bluetooth::EventFromAdapterPropertiesCallback:
                        {
                            nn::bluetooth::InfoFromAdapterPropertiesCallback* pInfo =
                                reinterpret_cast<nn::bluetooth::InfoFromAdapterPropertiesCallback*>(buffer);
                            NN_LOG("AdapterProperties: 0x%X\n", pInfo->status);
                            break;
                        }
                    case nn::bluetooth::EventFromDiscoveryStateChangedCallback:
                        {
                            myclient->EventFromDiscoveryStateChangedCallback(
                                    reinterpret_cast<nn::bluetooth::InfoFromDiscoveryStateChangedCallback*>(buffer));
                            break;
                        }
                    case nn::bluetooth::EventFromDeviceFoundCallback:
                        {
                            myclient->EventFromDeviceFoundCallback(
                                    reinterpret_cast<const nn::bluetooth::InfoFromDeviceFoundCallback*>(buffer));
                            break;
                        }
                    case nn::bluetooth::EventFromSspRequestCallback:
                        {
                            //SSP request is came.
                            //Reply for it.
                            nn::bluetooth::InfoFromSspRequestCallback* pInfo =
                                reinterpret_cast<nn::bluetooth::InfoFromSspRequestCallback*>(buffer);
                            NN_LOG("SspRequest: name:%s %s\n", pInfo->bluetoothName.name, toHexString(pInfo->bluetoothAddress));
                            NN_LOG("  [SspReply]\n");
                            nn::bluetooth::SspReply(&pInfo->bluetoothAddress, nn::bluetooth::BT_SSP_VARIANT_PASSKEY_CONFIRMATION, true, 0);
                            break;
                        }
                    case nn::bluetooth::EventFromPinRequestCallback:
                        {
                            myclient->EventFromPinRequestCallback(
                                    reinterpret_cast<nn::bluetooth::InfoFromPinRequestCallback*>(buffer));
                            break;
                        }
                    case nn::bluetooth::EventFromBondStateChangedCallback:
                        {
                            myclient->EventFromBondStateChangedCallback(
                                    reinterpret_cast<const nn::bluetooth::InfoFromBondStateChangedCallback*>(buffer));
                            break;
                        }
                    case nn::bluetooth::EventFromFatalErrorCallback:
                        {
                            NN_LOG("EventFromFatalErrorCallback: Fatal error occured!\n");
                            myclient->fatalErrorOccurred = true;
                            break;
                        }
                    default:
                        NN_LOG("ERROR: Un-handled System Event: 0x%X\n", eventType);
                }
                break;
            }
            case EVENTTYPE_FROM_CLIENT:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForExitCore);
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
} // NOLINT(impl/function_size)


//-----------------------------------------------------------------------------
// EventFromConnectionStateCallback
//
void bluetoothClient::EventFromConnectionStateCallback(const nn::bluetooth::InfoFromConnectionStateCallback* pInfo)
{
    const char* connStrP;
    switch(pInfo->state)
    {
        case nn::bluetooth::BTHH_CONN_STATE_CONNECTED:               connStrP = "BTHH_CONN_STATE_CONNECTED"; break;
        case nn::bluetooth::BTHH_CONN_STATE_CONNECTING:              connStrP = "BTHH_CONN_STATE_CONNECTING"; break;
        case nn::bluetooth::BTHH_CONN_STATE_DISCONNECTED:            connStrP = "BTHH_CONN_STATE_DISCONNECTED"; break;
        case nn::bluetooth::BTHH_CONN_STATE_DISCONNECTING:           connStrP = "BTHH_CONN_STATE_DISCONNECTING"; break;
        case nn::bluetooth::BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST:  connStrP = "BTHH_CONN_STATE_FAILED_MOUSE_FROM_HOST"; break;
        case nn::bluetooth::BTHH_CONN_STATE_FAILED_KBD_FROM_HOST:    connStrP = "BTHH_CONN_STATE_FAILED_KBD_FROM_HOST"; break;
        case nn::bluetooth::BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES: connStrP = "BTHH_CONN_STATE_FAILED_TOO_MANY_DEVICES"; break;
        case nn::bluetooth::BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER:  connStrP = "BTHH_CONN_STATE_FAILED_NO_BTHID_DRIVER"; break;
        case nn::bluetooth::BTHH_CONN_STATE_FAILED_GENERIC:          connStrP = "BTHH_CONN_STATE_FAILED_GENERIC"; break;
        case nn::bluetooth::BTHH_CONN_STATE_UNKNOWN:                 connStrP = "BTHH_CONN_STATE_UNKNOWN"; break;
        default:
            NN_LOG("Invalid ConnectionState: 0x%X\n", pInfo->state);
            connStrP = "UNKNOWN";
    }
    NN_LOG("ConnectionState(HID): %s %s\n", connStrP, toHexString(pInfo->bluetoothAddress));

    setControllerInfoState(pInfo->bluetoothAddress, pInfo->state);

    if (pInfo->state==nn::bluetooth::BTHH_CONN_STATE_CONNECTED)
    {
        // SET THE LEDS
        for (int k=0; k<MAX_CTRL; k++)
        {
            if (!memcmp(controllerInfo[k].bdAddr.address, pInfo->bluetoothAddress.address, 6))
            {
                nn::bluetooth::HidData hidData;
                memset(&hidData, 0, sizeof(hidData));
                hidData.length=12;
                hidData.data[0]=0x01;
                hidData.data[10]=0x30;
                hidData.data[11]=k + 1;
                nn::bluetooth::HidSendData(&pInfo->bluetoothAddress, &hidData);
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
                break;
            }
        }

        //change the MC2 mode.
        //nn::bluetooth::ChannelInfo channelInfo={{1,1,0,0,0,0,0,0,0,0}};
        //nn::bluetooth::ExtSetMc2(true, channelInfo, 0);
    }
}

//-----------------------------------------------------------------------------
void bluetoothClient::setTsi(const nn::bluetooth::BluetoothAddress &bdAddr, int tsi)
{
    NN_LOG("  [ExtSetTsi] %d\n", tsi);
    auto result = nn::bluetooth::ExtSetTsi(&bdAddr, tsi);
    if (result.IsSuccess())
        setControllerTsiState(bdAddr, tsi);
    else
    {
        NN_LOG("Failed!\n");
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
    }
}

//-----------------------------------------------------------------------------
// EventFromExtensionCallbacks
//
void bluetoothClient::EventFromExtensionCallbacks(const nn::bluetooth::InfoFromExtensionCallbacks* pInfo)
{
    const char* extStrP;
    switch (pInfo->eventType)
    {
        case nn::bluetooth::EventFromTsiSetCallback:    extStrP = "EventFromTsiSetCallback"; break;
        case nn::bluetooth::EventFromTsiExitCallback:   extStrP = "EventFromTsiExitCallback"; break;
        case nn::bluetooth::EventFromBurstSetCallback:  extStrP = "EventFromBurstSetCallback"; break;
        case nn::bluetooth::EventFromBurstExitCallback: extStrP = "EventFromBurstExitCallback"; break;
        case nn::bluetooth::EventFromSetZeroRetranCallback: extStrP = "EventFromSetZeroRetranCallback"; break;
        case nn::bluetooth::EventFromGetPendingConnectionsCallback: extStrP = "EventFromGetPendingConnectionsCallback"; break;
        default:
            NN_LOG("Invalid Extension Event: 0x%X\n", pInfo->eventType);
            extStrP = "UNKNOWN";
    }
    NN_LOG("Extension(HID): %s\n", extStrP);

    if (pInfo->eventType == nn::bluetooth::EventFromTsiSetCallback ||
        pInfo->eventType == nn::bluetooth::EventFromTsiExitCallback)
    {
        setControllerTsiState(pInfo->bluetoothAddress, 0, 1);
    }
    else if (pInfo->eventType == nn::bluetooth::EventFromSetZeroRetranCallback)
    {
        NN_LOG("Zero re-tx mode: %s\n", pInfo->zeroRetran.enable ? "ENABLED" : "DISABLED");
    }
    else if (pInfo->eventType == nn::bluetooth::EventFromGetPendingConnectionsCallback)
    {
        NN_LOG("Pending Connections: %d\n", pInfo->connections.numPending);
    }
}


//-----------------------------------------------------------------------------
// EventFromGetReportDataCallback
//
void bluetoothClient::EventFromGetReportDataCallback(const nn::bluetooth::InfoFromGetReportCallback* pInfo)
{
    const nn::bluetooth::HidReport* pReport = &pInfo->rptData;// NOLINT

    //"Basic Input Report" came.
    if(pReport->reportData.data[0]==0x30)
    {
        setControllerHidCount(pInfo->bluetoothAddress);
    }

    //"Reply for the Get Device Info" came.
    else if(pReport->reportData.data[0]==0x21)
    {
        NN_LOG("  [!!!!]\n");
        //Time stamp for measuring the latency of data transaction.
        //NN_LOG("[MeasuringLatency]ending time   = %d (msec)\n",nn::os::ConvertToTimeSpan(nn::os::GetSystemTick()).GetMilliSeconds());
        //PrintByteDataAsHex(32,&pUipcReport->report.report_data.data[0]);
    }
    else
    {
        NN_LOG("  [????]\n");
    }
}

//-----------------------------------------------------------------------------
// EventFromSetReportStatusCallback
//
void bluetoothClient::EventFromSetReportStatusCallback(const nn::bluetooth::InfoFromSetReportStatusCallback* pInfo)
{
    NN_ASSERT_NOT_NULL(pInfo);
    NN_LOG("  %s: got status %d for %s\n", __func__, pInfo->status, toHexString(pInfo->bluetoothAddress));
}

//-----------------------------------------------------------------------------
// EventFromGetReportStatusCallback
//
void bluetoothClient::EventFromGetReportStatusCallback(const nn::bluetooth::InfoFromGetReportStatusCallback* pInfo)
{
    NN_ASSERT_NOT_NULL(pInfo);
    NN_LOG("  %s: got status %d for %s, length=%d, type=0x%x\n", __func__, pInfo->status,
             toHexString(pInfo->bluetoothAddress), pInfo->length, pInfo->data[0]);

    // Test Checksum:
    uint8_t sum = 0;
    for (int i=0; i<pInfo->length; i++)
        sum += pInfo->data[i];
    NN_LOG("  %s CHECKSUM RESULT 0x%02x (%s)\n", __func__, sum, sum == 0 ? "PASS" : "FAIL");
}

//-----------------------------------------------------------------------------
// WaitEventSignalsForHid: thread for handling HID events
//
void bluetoothClient::WaitEventSignalsForHid(void *arg)
{
    bluetoothClient *myclient = reinterpret_cast<bluetoothClient*>(arg);

    nn::bluetooth::HidEventType eventTypeForHid;
    uint8_t bufferForHid[nn::bluetooth::BUFFER_SIZE_OF_HID_OUT];

    while(true)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&myclient->multiWaitForHid);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
        case EVENTTYPE_FROM_HID:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForHid);

                nn::bluetooth::HidGetEventInfo(&eventTypeForHid, bufferForHid, nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);

                switch(eventTypeForHid)
                {
                case nn::bluetooth::EventFromConnectionStateCallback:
                    {
                        myclient->EventFromConnectionStateCallback(
                            reinterpret_cast<const nn::bluetooth::InfoFromConnectionStateCallback*>(bufferForHid));
                        break;
                    }
                case nn::bluetooth::EventFromExtensionCallbacks:
                    {
                        myclient->EventFromExtensionCallbacks(
                            reinterpret_cast<nn::bluetooth::InfoFromExtensionCallbacks*>(bufferForHid));
                        break;
                    }
#ifndef SUPPORT_HID_REPORT_EVENT
                case nn::bluetooth::EventFromGetReportCallback:
                    {
                        myclient->EventFromGetReportDataCallback(
                                reinterpret_cast<nn::bluetooth::InfoFromGetReportCallback*>(bufferForHid));
                        break;
                    }
                case nn::bluetooth::EventFromSetReportStatusCallback:
                    {
                        myclient->EventFromSetReportStatusCallback(
                            reinterpret_cast<nn::bluetooth::InfoFromSetReportStatusCallback*>(bufferForHid));
                        break;
                    }
                case nn::bluetooth::EventFromGetReportStatusCallback:
                    {
                        myclient->EventFromGetReportStatusCallback(
                            reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(bufferForHid));
                        break;
                    }
#endif
                default:
                    {
                        NN_LOG("ERROR: Un-handled Hid System Event: 0x%X\n", eventTypeForHid);
                    }
                }

                break;
            }
        case EVENTTYPE_FROM_CLIENT:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForExitHid);
                return;
            }
        case EVENTTYPE_FROM_HID_REPORT:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForInputReport);

                nn::bluetooth::HidGetReportEventInfo(&eventTypeForHid, bufferForHid, nn::bluetooth::BUFFER_SIZE_OF_HID_OUT);

                switch(eventTypeForHid)
                {
#ifdef SUPPORT_HID_REPORT_EVENT
                case nn::bluetooth::EventFromGetReportCallback:
                    {
                        myclient->EventFromGetReportDataCallback(
                                reinterpret_cast<nn::bluetooth::InfoFromGetReportCallback*>(bufferForHid));
                        break;
                    }
                case nn::bluetooth::EventFromSetReportStatusCallback:
                    {
                        myclient->EventFromSetReportStatusCallback(
                            reinterpret_cast<nn::bluetooth::InfoFromSetReportStatusCallback*>(bufferForHid));
                        break;
                    }
                case nn::bluetooth::EventFromGetReportStatusCallback:
                    {
                        myclient->EventFromGetReportStatusCallback(
                            reinterpret_cast<nn::bluetooth::InfoFromGetReportStatusCallback*>(bufferForHid));
                        break;
                    }
#endif
                default:
                    {
                        NN_LOG("ERROR: Un-handled Hid Report Event: 0x%X\n", eventTypeForHid);
                    }
                }

                break;
            }
        default: NN_UNEXPECTED_DEFAULT;
        }
    }
}

//-----------------------------------------------------------------------------
// WaitEventSignals: thread for handling core events
//
void bluetoothClient::WaitEventSignalsForBle(void *arg)
{
    bluetoothClient *myclient = reinterpret_cast<bluetoothClient*>(arg);

    nn::bluetooth::BleEventType eventType;

    uint8_t buffer[nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT];

    while(true)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&myclient->multiWaitForBle);
        int holderType = nn::os::GetMultiWaitHolderUserData(holder);
        switch (holderType)
        {
            case EVENTTYPE_FROM_BLE_CORE:
            {
                NN_LOG("EVENTTYPE_FROM_BLE_CORE\n");
                nn::os::TryWaitSystemEvent(&myclient->systemEventForBleCore);

                nn::bluetooth::GetLeCoreEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

                switch(eventType)
                {
                case nn::bluetooth::EventFromLeConnParamUpdateCallback:
                    myclient->EventFromLeConnParamUpdateCallback(
                        reinterpret_cast<nn::bluetooth::InfoFromLeConnParamUpdateCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeScanStateChangedCallback:
                    myclient->EventFromLeScanStateChangedCallback(
                        reinterpret_cast<nn::bluetooth::InfoFromLeScanStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeScanFilterStateChangedCallback:
                    myclient->EventFromLeScanFilterStateChangedCallback(
                        reinterpret_cast<nn::bluetooth::InfoFromLeScanFilterStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientStateChangedCallback:
                    myclient->EventFromLeClientStateChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeAppStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerStateChangedCallback:
                    myclient->EventFromLeServerStateChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeAppStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientConnStateChangedCallback:
                    myclient->EventFromLeClientConnStateChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeConnStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerConnStateChangedCallback:
                    myclient->EventFromLeServerConnStateChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeConnStateChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientGattOpCallback:
                    // Use the same callback for general BLE (for application) and BLE HID
                    myclient->EventFromLeClientGattOperationCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeGattOperationCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientServiceDiscoveryCallback:
                    myclient->EventFromLeClientGattSrvcDiscCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientServiceDiscoveryStateChangedCallback:
                    myclient->EventFromLeClientGattSrvcDiscStateChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeClientGattServiceDiscoveryCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeClientConfigureMtuCallback:
                    myclient->EventFromLeClientMtuConfigCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeClientMtuConfigurationCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerProfileChangedCallback:
                    myclient->EventFromLeServerProfileChangedCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeServerProfileChangedCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerGattReqCallback:
                    // Use the same callback for general BLE (for application) and BLE HID
                    myclient->EventFromLeServerGattReqCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeServerGattReqCallback*>(buffer));
                    break;
                default:
                        NN_LOG("ERROR: Un-handled System Event(BLE CORE): 0x%X\n", eventType);
                }
                break;
            }
            case EVENTTYPE_FROM_BLE:
                NN_LOG("EVENTTYPE_FROM_BLE\n");
                nn::os::TryWaitSystemEvent(&myclient->systemEventForBle);

                nn::bluetooth::user::GetLeEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

                switch (eventType)
                {
                case nn::bluetooth::EventFromLeClientGattOpCallback:
                    // Use the same callback for BLE HID and BLE Core
                    myclient->EventFromLeClientGattOperationCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeGattOperationCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerGattReqCallback:
                    // Use the same callback for BLE HID and BLE Core
                    myclient->EventFromLeServerGattReqCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeServerGattReqCallback*>(buffer));
                    break;
                default:
                    NN_LOG("ERROR: Un-expected System Event(BLE): 0x%X\n", eventType);
                    break;
                }
                break;
            case EVENTTYPE_FROM_BLE_HID:
                NN_LOG("EVENTTYPE_FROM_BLE_HID\n");
                nn::os::TryWaitSystemEvent(&myclient->systemEventForBleHid);

                nn::bluetooth::GetLeHidEventInfo(&eventType, buffer, nn::bluetooth::BUFFER_SIZE_OF_BLE_OUT);

                switch (eventType)
                {
                case nn::bluetooth::EventFromLeClientGattOpCallback:
                    // Use the same callback for general BLE (for application) and BLE Core
                    myclient->EventFromLeClientGattOperationCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeGattOperationCallback*>(buffer));
                    break;
                case nn::bluetooth::EventFromLeServerGattReqCallback:
                    // Use the same callback for general BLE (for application) and BLE Core
                    myclient->EventFromLeServerGattReqCallback(
                        reinterpret_cast<const nn::bluetooth::InfoFromLeServerGattReqCallback*>(buffer));
                    break;
                default:
                    NN_LOG("ERROR: Un-expected System Event(BLE HID): 0x%X\n", eventType);
                    break;
                }
                break;
            case EVENTTYPE_FROM_CLIENT:
            {
                nn::os::TryWaitSystemEvent(&myclient->systemEventForExitBle);
                return;
            }
            default: NN_UNEXPECTED_DEFAULT;
        }
    }
} // NOLINT(impl/function_size)

//-----------------------------------------------------------------------------
// AddPairedDevices: Add paired device information to BSA
//
void bluetoothClient::AddPairedDevices()
{
    // read the pairing data
    devicesPaired = nn::settings::system::GetBluetoothDevicesSettings(devicesSettingsArray, nn::settings::system::BluetoothDevicesSettingsCountMax);
    NN_LOG("Paired Bluetooth devices: %d\n", devicesPaired);

    for (int i = 0; i < devicesPaired; i++)
    {
        START_TIMER();
        nn::bluetooth::HidAddPairedDevice(&devicesSettingsArray[i]);
        STOP_TIMER("HidAddPairedDevice");

    }
}

//-----------------------------------------------------------------------------
// GetPairedDevices: Get paired device information for nand
//
void bluetoothClient::GetPairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr)
{
    for (int i = 0; i < devicesPaired; i++)
    {
        if (memcmp(BdAddr.address, devicesSettingsArray[i].bd_addr, 6)==0)
        {
            // already in the table. overwrite it.
            nn::bluetooth::HidGetPairedDevice(&BdAddr, &devicesSettingsArray[i]);
            return;
        }
    }

    if(devicesPaired < nn::settings::system::BluetoothDevicesSettingsCountMax)
    {
        nn::bluetooth::HidGetPairedDevice(&BdAddr, &devicesSettingsArray[devicesPaired]);
        devicesPaired++;
    }
    else
    {
        NN_LOG("GetPairedDevice failed, max devices already paired\n");
    }
}

//-----------------------------------------------------------------------------
// RemovePairedDevice: Remove paired device information
//
void bluetoothClient::RemovePairedDevice(const nn::bluetooth::BluetoothAddress &BdAddr)
{
    for (int i = 0; i < devicesPaired; i++)
    {
        if (memcmp(BdAddr.address, devicesSettingsArray[i].bd_addr, 6)==0)
        {
            devicesPaired--;

            // Replace the entry with a device from the top of the table
            devicesSettingsArray[i] = devicesSettingsArray[devicesPaired];

            // Save the whole device table to nand
            nn::settings::system::SetBluetoothDevicesSettings(devicesSettingsArray, devicesPaired);
            NN_LOG("Paired Bluetooth devices: %d\n", devicesPaired);
            return;
        }
    }
}

static bluetoothClient client;
//-----------------------------------------------------------------------------
// startBluetooth
//
int bluetoothClient::startBluetooth()
{
    nn::Result result;

    START_TIMER();
    nn::bluetooth::InitializeBluetoothDriver();
    STOP_TIMER("InitializeBluetoothDriver");

    START_TIMER();
    nn::bluetooth::InitializeBluetooth(&systemEventForCore);
    STOP_TIMER("InitializeBluetooth");

    START_TIMER();
    nn::bluetooth::InitializeHid(&systemEventForHid, CLIENT_VERSION);
    STOP_TIMER("InitializeHid");

    START_TIMER();
    result = nn::bluetooth::EnableBluetooth();
    STOP_TIMER("EnableBluetooth");

    if (!result.IsSuccess())
    {
        nn::bluetooth::CleanupBluetooth();
        nn::bluetooth::DisableBluetooth();
        nn::bluetooth::FinalizeBluetoothDriver();
        return -1;
    }


    if(CLIENT_VERSION != 0)
    {
        AddPairedDevices();
    }

    nn::os::InitializeMultiWaitHolder(&holderForCore, &systemEventForCore);
    nn::os::SetMultiWaitHolderUserData(&holderForCore, EVENTTYPE_FROM_CORE);

    nn::os::InitializeMultiWaitHolder(&holderForHid, &systemEventForHid);
    nn::os::SetMultiWaitHolderUserData(&holderForHid, EVENTTYPE_FROM_HID);

    nn::os::CreateSystemEvent(&systemEventForExitCore, nn::os::EventClearMode_AutoClear, false);
    nn::os::InitializeMultiWaitHolder(&holderForExitCore, &systemEventForExitCore);
    nn::os::SetMultiWaitHolderUserData(&holderForExitCore, EVENTTYPE_FROM_CLIENT);

    nn::os::CreateSystemEvent(&systemEventForExitHid, nn::os::EventClearMode_AutoClear, false);
    nn::os::InitializeMultiWaitHolder(&holderForExitHid, &systemEventForExitHid);
    nn::os::SetMultiWaitHolderUserData(&holderForExitHid, EVENTTYPE_FROM_CLIENT);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::bluetooth::RegisterHidReportEvent(&systemEventForInputReport));

    nn::os::InitializeMultiWaitHolder(&holderForHidReport, &systemEventForInputReport);
    nn::os::SetMultiWaitHolderUserData(&holderForHidReport, EVENTTYPE_FROM_HID_REPORT);

    nn::os::InitializeMultiWait(&multiWaitForCore);
    nn::os::LinkMultiWaitHolder(&multiWaitForCore, &holderForCore);
    nn::os::LinkMultiWaitHolder(&multiWaitForCore, &holderForExitCore);

    nn::os::InitializeMultiWait(&multiWaitForHid);
    nn::os::LinkMultiWaitHolder(&multiWaitForHid, &holderForHid);
    nn::os::LinkMultiWaitHolder(&multiWaitForHid, &holderForExitHid);
    nn::os::LinkMultiWaitHolder(&multiWaitForHid, &holderForHidReport);

    NN_LOG("  [SetAdapterProperty(BT_PROPERTY_BDNAME)]\n");
    nn::bluetooth::BluetoothName myName={"Nintendo Switch"};
    START_TIMER();
    nn::bluetooth::SetAdapterProperty(nn::bluetooth::BT_PROPERTY_BDNAME, myName.name, std::strlen(reinterpret_cast<char *>(myName.name)));
    STOP_TIMER("SetAdapterProperty");

    result = nn::os::CreateThread( &threadForWaitEventSignals, WaitEventSignals, this,
            threadStack, ThreadStackSize, nn::os::HighestThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create threadForWaitEventSignals." );
    nn::os::StartThread( &threadForWaitEventSignals );

    result = nn::os::CreateThread( &threadForWaitEventSignalsForHid, WaitEventSignalsForHid, this,
            threadStackForHid, ThreadStackSize, nn::os::HighestThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create threadForWaitEventSignalsForHid." );
    nn::os::StartThread( &threadForWaitEventSignalsForHid );

    START_TIMER();
    nn::bluetooth::ExtSetVisibility(false,true);
    STOP_TIMER("SetVisibility");

    START_TIMER();
    nn::bluetooth::ExtSetMcMode(0);
    STOP_TIMER("SetMcMode");

    START_TIMER();
    nn::bluetooth::AdapterProperty ap;
    nn::bluetooth::GetAdapterProperties(&ap);
    STOP_TIMER("GetAdapterProperties");
    NN_LOG("Properties: %s\n", toHexString(ap.bluetoothAddress));
    NN_LOG("Properties: %s\n", toHexString(3, (uint8_t*)&ap.classOfDevice));


    return 0;

}

//-----------------------------------------------------------------------------
// finishBluetooth
//
void bluetoothClient::finishBluetooth()
{
    // Exit core & hid threads
    nn::os::SignalSystemEvent(&systemEventForExitCore);
    nn::os::SignalSystemEvent(&systemEventForExitHid);

    nn::os::DestroyThread( &threadForWaitEventSignalsForHid );
    nn::os::DestroyThread( &threadForWaitEventSignals );

    nn::os::UnlinkAllMultiWaitHolder(&multiWaitForCore);
    nn::os::UnlinkAllMultiWaitHolder(&multiWaitForHid);
    nn::os::FinalizeMultiWaitHolder(&holderForCore);
    nn::os::FinalizeMultiWaitHolder(&holderForExitCore);
    nn::os::FinalizeMultiWaitHolder(&holderForHid);
    nn::os::FinalizeMultiWaitHolder(&holderForExitHid);
    nn::os::FinalizeMultiWaitHolder(&holderForHidReport);
    nn::os::FinalizeMultiWait(&multiWaitForCore);
    nn::os::FinalizeMultiWait(&multiWaitForHid);

    nn::os::DestroySystemEvent(&systemEventForCore);
    nn::os::DestroySystemEvent(&systemEventForExitCore);
    nn::os::DestroySystemEvent(&systemEventForHid);
    nn::os::DestroySystemEvent(&systemEventForExitHid);
    nn::os::DestroySystemEvent(&systemEventForInputReport);

    nn::bluetooth::DisableBluetooth();
    nn::bluetooth::CleanupBluetooth();
    nn::bluetooth::CleanupHid();

    // Disconnect from bluetooth server
    nn::bluetooth::FinalizeBluetoothDriver();
}

//-----------------------------------------------------------------------------
// restartBluetooth
//
void bluetoothClient::restartBluetooth()
{
    NN_LOG("Restarting Bluetooth\n");

    // Safety precaution, not required for restart
    nn::bluetooth::CancelDiscovery();
    nn::bluetooth::ExtSetVisibility(false,false);

    START_TIMER();
    nn::bluetooth::DisableBluetooth();
    STOP_TIMER("DisableBluetooth");

    START_TIMER();
    nn::bluetooth::EnableBluetooth();
    STOP_TIMER("EnableBluetooth");

    if(CLIENT_VERSION != 0)
    {
        AddPairedDevices();
    }

    // Allow connections again
    nn::bluetooth::ExtSetVisibility(false,true);

    NN_LOG("Restart Complete\n");
}

//-----------------------------------------------------------------------------
// startBluetoothLowEnergy
//
void bluetoothClient::startBluetoothLowEnergy()
{
    nn::bluetooth::InitializeBluetoothLe(&systemEventForBleCore);
    nn::bluetooth::EnableBluetoothLe();

    nn::bluetooth::user::InitializeBluetoothUserInterface();
    nn::bluetooth::user::RegisterBleEvent(&systemEventForBle);
    nn::bluetooth::RegisterBleHidEvent(&systemEventForBleHid);

    nn::os::InitializeMultiWaitHolder(&holderForBleCore, &systemEventForBleCore);
    nn::os::SetMultiWaitHolderUserData(&holderForBleCore, EVENTTYPE_FROM_BLE_CORE);
    nn::os::InitializeMultiWaitHolder(&holderForBle, &systemEventForBle);
    nn::os::SetMultiWaitHolderUserData(&holderForBle, EVENTTYPE_FROM_BLE);
    nn::os::InitializeMultiWaitHolder(&holderForBleHid, &systemEventForBleHid);
    nn::os::SetMultiWaitHolderUserData(&holderForBleHid, EVENTTYPE_FROM_BLE_HID);

    nn::os::CreateSystemEvent(&systemEventForExitBle, nn::os::EventClearMode_AutoClear, false);
    nn::os::InitializeMultiWaitHolder(&holderForExitBle, &systemEventForExitBle);
    nn::os::SetMultiWaitHolderUserData(&holderForExitBle, EVENTTYPE_FROM_CLIENT);

    nn::os::InitializeMultiWait(&multiWaitForBle);
    nn::os::LinkMultiWaitHolder(&multiWaitForBle, &holderForBleCore);
    nn::os::LinkMultiWaitHolder(&multiWaitForBle, &holderForBle);
    nn::os::LinkMultiWaitHolder(&multiWaitForBle, &holderForBleHid);
    nn::os::LinkMultiWaitHolder(&multiWaitForBle, &holderForExitBle);

    nn::Result result = nn::os::CreateThread( &threadForWaitEventSignalsForBle, WaitEventSignalsForBle, this,
            threadStackForBle, ThreadStackSize, nn::os::HighestThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create threadForWaitEventSignalsForBle." );
    nn::os::StartThread( &threadForWaitEventSignalsForBle );
}

void bluetoothClient::finishBluetoothLowEnergy()
{
    nn::bluetooth::DisableBluetoothLe();

    nn::os::UnlinkAllMultiWaitHolder(&multiWaitForBle);
    nn::os::FinalizeMultiWaitHolder(&holderForBleCore);
    nn::os::FinalizeMultiWaitHolder(&holderForBle);
    nn::os::FinalizeMultiWaitHolder(&holderForBleHid);
    nn::os::FinalizeMultiWaitHolder(&holderForExitBle);

    nn::os::DestroySystemEvent(&systemEventForBleCore);
    nn::os::DestroySystemEvent(&systemEventForBle);
    nn::os::DestroySystemEvent(&systemEventForBleHid);
    nn::os::DestroySystemEvent(&systemEventForExitBle);

    nn::bluetooth::CleanupBluetoothLe();
    nn::bluetooth::user::FinalizeBluetoothUserInterface();
}

//-----------------------------------------------------------------------------
//
//
extern "C" void nnMain2()
{
    NN_LOG("NBM_CLIENT NOW RUNNING \n");

    client.startBluetooth();
    nn::bluetooth::ExtSetVisibility(false,true);

    int loopTimeMs=0;

    for(;;)
    {
        if(loopTimeMs==0)
        {
            //Discovery
            NN_LOG("  [StartDiscovery]\n");
            nn::bluetooth::StartDiscovery();
        }

        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(100));
        loopTimeMs += 100;
        if(loopTimeMs == 30000)
        {
           loopTimeMs=0;
        }

        if (client.fatalErrorOccurred)
        {
            client.restartBluetooth();
            client.fatalErrorOccurred = false;
        }
    }

    client.finishBluetooth();
}



//-----------------------------------------------------------------------------

