﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AudioMultipleRenderer.cpp,PageSampleAudioAudioMultipleRenderer}
 *
 * @brief
 * マルチオーディオレンダラのサンプルプログラム
 */

/**
 * @page PageSampleAudioAudioMultipleRenderer マルチオーディオレンダラ
 * @tableofcontents
 *
 * @brief
 * マルチオーディオレンダラのサンプルプログラムの解説です。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionBrief 概要
 * 複数のオーディオレンダラを利用して、オーディオ再生を行うサンプルです。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/AudioMultipleRenderer
 * Samples/Sources/Applications/AudioMultipleRenderer @endlink 以下にあります。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionNecessaryEnvironment 必要な環境
 * オーディオ出力が利用可能である必要があります。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、2 つのオーディオレンダラによりオーディオ再生を開始されます。
 *
 * <p>
 * <table>
 * <tr><th> 入力 </th><th> 動作 </th></tr>
 * <tr><td> [A]             </td><td> AudioRenderer 1 を起動/終了          </td></tr>
 * <tr><td> [B]             </td><td> AudioRenderer 2 を起動/終了          </td></tr>
 * <tr><td> [Start/Space]   </td><td> プログラムの終了                     </td></tr>
 * </table>
 * </p>
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionPrecaution 注意事項
 * 特にありません。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAudioAudioMultipleRenderer_SectionDetail 解説
 * このサンプルプログラムは、複数のオーディオレンダラを用いてオーディオレンダリングを行うものです。
 * オーディオレンダラは、同時に最大 nn::audio::AudioRendererCountMax 個を使用することができます。
 * オーディオレンダラの使い方については、@link ../../../Samples/Sources/Applications/AudioRenderer Samples/Sources/Applications/AudioRenderer @endlink を併せてご参照ください。
 *
 * オーディオレンダラは、自身に紐づくリソースをそれぞれ独自に管理します。
 * よって、VoiceType や MemoryPoolType 、AudioRendererConfig など特定の AudioRenderer に紐づくリソースは、各 AudioRenderer の間で共有できないことにご注意ください。
 *
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - オーディオレンダラで再生を行うスレッドを 2 個作成
 * 作成されたスレッドでは以下の処理が実行されます。
 * - オーディオレンダラの初期化
 * - FinalMix の準備
 * - 出力先デバイスの設定
 * - 入力波形の準備、Voice の確保およびパラメータ設定
 * - オーディオレンダリングの開始
 * - レンダリング処理の終了要求の通知を確認
 * - レンダリングを終了
 * - オーディオレンダラをクローズ
 * - 入力波形を破棄
 *
 */

#include <atomic>
#include <cstdlib>
#include <cmath>

#include <nns/nns_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_TimeSpan.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

const char Title[] = "AudioMultipleRenderer";

class AudioRendererThread
{
private:
    NN_ALIGNAS(4096) char m_WorkBuffer[8 * 1024 * 1024];
    NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char m_WaveBufferPoolMemory[14 * 1024 * 1024];

    nn::mem::StandardAllocator m_Allocator;
    nn::mem::StandardAllocator m_WaveBufferAllocator;

    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStack[8192];
    nn::os::ThreadType m_Thread;
    int m_Frequency;
    std::atomic<bool> m_IsInitialized;
    int m_Id;

    std::size_t GenerateSineWave(void** data, int sampleRate, int frequency, int sampleCount)
    {
        // m_WaveBufferAllocator が管理するメモリ領域はすべて waveBufferMemoryPool メモリプールに追加されています。
        int16_t* p = static_cast<int16_t*>(m_WaveBufferAllocator.Allocate(sampleCount * sizeof(int16_t), nn::audio::BufferAlignSize));
        NN_ABORT_UNLESS_NOT_NULL(p);
        const float Pi = 3.1415926535897932384626433f;
        for (auto i = 0; i < sampleCount; ++i)
        {
            p[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * frequency * i / sampleRate));
        }
        *data = p;
        return sampleCount * sizeof(int16_t);
    }

    void ThreadFunc(void* args)
    {
        NN_UNUSED(args);

        // レンダラのパラメータを指定します。
        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.sampleRate = RenderRate;
        parameter.sampleCount = RenderCount;
        parameter.mixBufferCount = 2;
        parameter.voiceCount = 1;
        parameter.subMixCount = 0;
        parameter.sinkCount = 1;
        parameter.effectCount = 0;
        parameter.performanceFrameCount = 0;

        // ミックスバッファとオーディオバスの関係を定義します。
        const int channelCount = 2;
        int8_t mainBus[channelCount];
        mainBus[nn::audio::ChannelMapping_FrontLeft] = 0;
        mainBus[nn::audio::ChannelMapping_FrontRight] = 1;

        // パラメータがシステムでサポートされているかどうかを確認します。
        NN_ABORT_UNLESS(
            nn::audio::IsValidAudioRendererParameter(parameter),
            "Invalid AudioRendererParameter specified."
        );

        // レンダラのワークバッファを準備します。
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        void* workBuffer = m_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer);

        nn::os::SystemEvent systemEvent;

        // レンダラを初期化します。
        nn::audio::AudioRendererHandle handle;
        NN_ABORT_UNLESS(
            nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
            "Failed to open AudioRenderer"
        );

        // AudioRendererConfig を初期化します。
        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        void* configBuffer = m_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(configBuffer);
        nn::audio::AudioRendererConfig config;
        nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

        nn::audio::FinalMixType finalMix;
        NN_ABORT_UNLESS(nn::audio::AcquireFinalMix(&config, &finalMix, 2));

        // レンダラの出力先を用意します。
        nn::audio::DeviceSinkType deviceSink;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut"));

        // 設定したパラメータをレンダラに反映させます。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));

        // レンダリングを開始します。
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::StartAudioRenderer(handle));

        // Voice を用意します。
        nn::audio::VoiceType voiceSine;
        nn::audio::WaveBuffer waveBufferSine;
        void* dataSine;

        // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
        nn::audio::MemoryPoolType waveBufferMemoryPool;
        NN_ABORT_UNLESS(nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory)));
        NN_ABORT_UNLESS(nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool));

        // Voice で再生する波形を準備します。
        {
            const int sineSampleRate = 32000;
            const int sineFrequency = m_Frequency;
            const int sineSampleCount = 32000;
            std::size_t size = GenerateSineWave(&dataSine, sineSampleRate, sineFrequency, sineSampleCount);

            nn::audio::AcquireVoiceSlot(&config, &voiceSine, sineSampleRate, 1, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
            nn::audio::SetVoiceDestination(&config, &voiceSine, &finalMix);

            waveBufferSine.buffer = dataSine;
            waveBufferSine.size = size;
            waveBufferSine.startSampleOffset = 0;
            waveBufferSine.endSampleOffset = sineSampleCount;
            waveBufferSine.loop = 0;
            waveBufferSine.isEndOfStream = false;

            // 波形を Voice に追加します。
            nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
            nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
            nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
            nn::audio::AppendWaveBuffer(&voiceSine, &waveBufferSine);
            nn::audio::SetVoicePlayState(&voiceSine, nn::audio::VoiceType::PlayState_Play);
            nn::audio::SetVoiceMixVolume(&voiceSine, &finalMix, 0.707f / 2, 0, 0);
            nn::audio::SetVoiceMixVolume(&voiceSine, &finalMix, 0.707f / 2, 0, 1);
        }

        // メインスレッドから終了指示が来るまで音を再生し続けます。
        while(m_IsInitialized)
        {
            systemEvent.Wait();

            if (const nn::audio::WaveBuffer* pWaveBuffer = nn::audio::GetReleasedWaveBuffer(&voiceSine))
            {
                nn::audio::AppendWaveBuffer(&voiceSine, pWaveBuffer);
            }

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::audio::RequestUpdateAudioRenderer(handle, &config));
        }

        nn::audio::SetVoicePlayState(&voiceSine, nn::audio::VoiceType::PlayState_Stop);
        nn::audio::RequestUpdateAudioRenderer(handle, &config);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(20));

        // レンダリングを終了します。
        nn::audio::StopAudioRenderer(handle);
        nn::audio::CloseAudioRenderer(handle);
        nn::os::DestroySystemEvent(systemEvent.GetBase());

        if (dataSine)
        {
            m_WaveBufferAllocator.Free(dataSine);
            dataSine = nullptr;
        }
        if (configBuffer)
        {
            m_Allocator.Free(configBuffer);
            configBuffer = nullptr;
        }
        if (workBuffer)
        {
            m_Allocator.Free(workBuffer);
            workBuffer = nullptr;
        }
    } // NOLINT(readability/fn_size)

public:
    void Initialize(int id, int frequency) NN_NOEXCEPT
    {
        m_Allocator.Initialize(m_WorkBuffer, sizeof(m_WorkBuffer));
        m_WaveBufferAllocator.Initialize(m_WaveBufferPoolMemory, sizeof(m_WaveBufferPoolMemory));

        m_IsInitialized = true;
        m_Id = id;
        m_Frequency = frequency;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadEntry, this, m_ThreadStack, sizeof(m_ThreadStack), nn::os::DefaultThreadPriority));
        nn::os::StartThread(&m_Thread);

        NNS_LOG("Start AudioRenderer %d.\n", m_Id);
    }

    void Finalize() NN_NOEXCEPT
    {
        m_IsInitialized = false;
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);

        m_Allocator.Finalize();
        m_WaveBufferAllocator.Finalize();

        NNS_LOG("Stop AudioRenderer %d.\n", m_Id);
    }

    bool IsInitialzied() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    static void ThreadEntry(void *args) NN_NOEXCEPT
    {
        reinterpret_cast<AudioRendererThread*>(args)->ThreadFunc(args);
    }
};

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonX     = nn::hid::KeyboardKey::X::Index;
    map.buttonY     = nn::hid::KeyboardKey::Y::Index;
    map.buttonL     = nn::hid::KeyboardKey::L::Index;
    map.buttonR     = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft  = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp    = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown  = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart = nn::hid::KeyboardKey::Space::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------------\n");
    NNS_LOG("[A]             Start/Stop AudioRenderer 1\n");
    NNS_LOG("[B]             Start/Stop AudioRenderer 2\n");
    NNS_LOG("[Start/Space]   Shut down sample program\n");
    NNS_LOG("-------------------------------------------------------\n");
}

} // anonymous namespace

AudioRendererThread g_AudioRendererThread[nn::audio::AudioRendererCountMax];

extern "C" void nnMain()
{
    InitializeHidDevices();
    PrintUsage();

    // オーディオレンダラ用スレッドを初期化します。
    g_AudioRendererThread[0].Initialize(1, 440);
    g_AudioRendererThread[1].Initialize(2, 880);

    for(;;)
    {
        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        // ボタン入力に応じてオーディオレンダラを初期化・終了します。
        if(npadButtonDown.Test< ::nn::hid::NpadButton::A >())
        {
            if(g_AudioRendererThread[0].IsInitialzied() == false)
            {
                g_AudioRendererThread[0].Initialize(1, 440);
            }
            else
            {
                g_AudioRendererThread[0].Finalize();
            }
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::B >())
        {
            if(g_AudioRendererThread[1].IsInitialzied() == false)
            {
                g_AudioRendererThread[1].Initialize(2, 880);
            }
            else
            {
                g_AudioRendererThread[1].Finalize();
            }
        }

        if(npadButtonDown.Test< ::nn::hid::NpadButton::Plus >())
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    for(int i = 0; i < nn::audio::AudioRendererCountMax; ++i)
    {
        if(g_AudioRendererThread[i].IsInitialzied())
        {
            g_AudioRendererThread[i].Finalize();
        }
    }
}
