﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>  // std::nothrow

#include <nn/nn_Abort.h>
#include <nns/nns_Log.h>

#include <nn/os.h>
#include <nn/mem.h>
#include <nn/fs.h>

#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/afx.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>

#include <nn/settings/settings_DebugPad.h>

namespace {
// choose rendering engine sample rate
//const int RenderRate = 48000;
const int RenderRate = 32000;
const int RenderCount = (RenderRate / 200);

// choose number of files to play
const int SeCount = 2;

const char Title[] = "AudioEffectLegacy";

// - and add / remove them to / from the files lists
const char* g_SeFileNames[SeCount] =
{
    "asset:/AudioCommon/SampleSe0.adpcm",
    "asset:/AudioCommon/SampleSe1.adpcm",
};

NN_ALIGNAS(4096) char g_WorkBuffer[16 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_WaveBufferPoolMemory[14 * 1024 * 1024];
NN_AUDIO_ALIGNAS_MEMORY_POOL_ALIGN char g_EffectBufferPoolMemory[16 * 1024 * 1024];

nn::mem::StandardAllocator g_Allocator;
nn::mem::StandardAllocator g_WaveBufferAllocator;
nn::mem::StandardAllocator g_EffectBufferAllocator;

char* g_MountRomCacheBuffer = NULL;

struct AudioEffectSampleStatus
{
    bool seSourceFlag;
};
}

void* Allocate(size_t size)
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

std::size_t ReadAdpcmFile(nn::audio::AdpcmHeaderInfo* header, void** adpcmData, const char* filename)
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filename, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS(result.IsSuccess());

    int64_t size;
    uint8_t adpcmheader[nn::audio::AdpcmHeaderSize];

    result = nn::fs::GetFileSize(&size, handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    *adpcmData = g_WaveBufferAllocator.Allocate(static_cast<std::size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS_NOT_NULL(*adpcmData);

    result = nn::fs::ReadFile(handle, 0, adpcmheader, sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    result = nn::fs::ReadFile(handle, sizeof(adpcmheader), *adpcmData, static_cast<size_t>(size) - sizeof(adpcmheader));
    NN_ABORT_UNLESS(result.IsSuccess());
    nn::fs::CloseFile(handle);

    nn::audio::ParseAdpcmHeader(header, adpcmheader, sizeof(adpcmheader));

    return static_cast<std::size_t>(size) - sizeof(adpcmheader);
}

void InitializeFileSystem()
{
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("asset", g_MountRomCacheBuffer, cacheSize)
    );
}

void FinalizeFileSystem()
{
    nn::fs::Unmount("asset");

    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

void InitializeHidDevices()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    const nn::hid::NpadIdType npadIds[2] = { nn::hid::NpadId::No1, nn::hid::NpadId::Handheld };
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    //キーボードのキーを DebugPad のボタンに割り当てます。
    nn::settings::DebugPadKeyboardMap map;
    nn::settings::GetDebugPadKeyboardMap(&map);
    map.buttonA      = nn::hid::KeyboardKey::A::Index;
    map.buttonB      = nn::hid::KeyboardKey::B::Index;
    map.buttonX      = nn::hid::KeyboardKey::X::Index;
    map.buttonY      = nn::hid::KeyboardKey::Y::Index;
    map.buttonL      = nn::hid::KeyboardKey::L::Index;
    map.buttonR      = nn::hid::KeyboardKey::R::Index;
    map.buttonLeft   = nn::hid::KeyboardKey::LeftArrow::Index;
    map.buttonRight  = nn::hid::KeyboardKey::RightArrow::Index;
    map.buttonUp     = nn::hid::KeyboardKey::UpArrow::Index;
    map.buttonDown   = nn::hid::KeyboardKey::DownArrow::Index;
    map.buttonStart  = nn::hid::KeyboardKey::Space::Index;
    map.buttonSelect = nn::hid::KeyboardKey::Minus::Index;
    nn::settings::SetDebugPadKeyboardMap(map);
}

void PrintUsage()
{
    NNS_LOG("---------------------------------------------- \n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("Renderer Sample Rate %d\n", RenderRate);
    NNS_LOG("---------------------------------------------- \n");
    NNS_LOG("[A]             StartSound     (SampleSe0)     \n");
    NNS_LOG("[B]             StartSound     (SampleSe1)     \n");
    NNS_LOG("[L]             Switch effect target SE        \n");
    NNS_LOG("[-]             Print Usage                    \n");
    NNS_LOG("[Start/Space]   Finish Sample Program          \n");
    NNS_LOG("---------------------------------------------- \n");
}

namespace {

bool g_IsSimpleChorusInitialized = false;
nns::afx::SimpleChorus g_SimpleChorus;
nns::afx::SimpleChorus::Constant g_SimpleChorusConstant;
nns::afx::SimpleChorus::Parameter g_SimpleChorusParameter;

}

namespace {

bool g_IsSimpleDelayInitialized = false;
nns::afx::SimpleDelay g_SimpleDelay;
nns::afx::SimpleDelay::Constant g_SimpleDelayConstant;
nns::afx::SimpleDelay::Parameter g_SimpleDelayParameter;

}

void ProcessUserEffect(nn::audio::AuxType aux, int32_t* readBuffer, int sampleCount, int channelCount, int auxInputChannelCount)
{
    if( !g_IsSimpleChorusInitialized )
    {
        g_SimpleChorusConstant.channelCount = channelCount;
        g_SimpleChorusConstant.sampleRate = RenderRate;
        auto size = nns::afx::GetRequiredMemorySizeForSimpleChorus(&g_SimpleChorusConstant);
        auto buffer = std::malloc(size);

        g_SimpleChorus.Initialize(buffer, size, &g_SimpleChorusConstant);

        g_SimpleChorusParameter.baseDelay = 25;
        g_SimpleChorusParameter.vibration = 10;
        g_SimpleChorusParameter.period = 1000;

        g_SimpleChorus.SetParameter(&g_SimpleChorusParameter);

        g_IsSimpleChorusInitialized = true;
    }

    if( !g_IsSimpleDelayInitialized )
    {
        g_SimpleDelayConstant.channelCount = channelCount;
        g_SimpleDelayConstant.sampleRate = RenderRate;
        g_SimpleDelayConstant.delayTimeMax = 1000;
        auto size = nns::afx::GetRequiredMemorySizeForSimpleDelay(&g_SimpleDelayConstant);
        auto buffer = std::malloc(size);

        g_SimpleDelay.Initialize(buffer, size, &g_SimpleDelayConstant);

        g_SimpleDelayParameter.delayTime = 250;
        g_SimpleDelayParameter.feedbackGain = 0.4f;
        g_SimpleDelayParameter.damping = 0.5f;
        g_SimpleDelay.SetParameter(&g_SimpleDelayParameter);

        g_IsSimpleDelayInitialized = true;
    }
    // Aux エフェクトによって受け取ったサンプルデータを加工する処理を行います。
    // ここではごく単純なボリューム調整とパン処理を行います。

    // オーディオレンダラからサンプルデータを読み込みます。
    int readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, channelCount * sampleCount * 2);
    while( readCount )
    {
        // Aux エフェクトによって得られるサンプルデータは、チャンネルごとにブロックインタリーブされています。
        // 各ブロックは Aux エフェクトを追加した FinalMix / SubMix が保持するサンプル数です。
        // 今回は FinalMix に追加したので、各ブロックに含まれるサンプル数は parameter.sampleCount に一致します。
        int samplesPerFrame = sampleCount;

        // 含まれるチャンネル数は SetAuxInputOutput() で設定した数となります。
        int channelBlockCount = auxInputChannelCount;

        // ReadAuxSendBuffer() で読み込まれたサンプルが、何フレーム分に相当するか取得します。
        int frameCount = readCount / (samplesPerFrame * channelCount);

        for( auto frame = 0; frame < frameCount; ++frame )
        {
            int32_t* bufferBase = readBuffer + frame * samplesPerFrame * channelBlockCount;

            int32_t* buffers[24] = { nullptr, };
            for( auto ch = 0; ch < channelBlockCount; ++ch )
            {
                buffers[ch] = bufferBase + ch * samplesPerFrame;
            }

            g_SimpleChorus.Apply(buffers, buffers, channelBlockCount, samplesPerFrame);
            g_SimpleDelay.Apply(buffers, buffers, channelBlockCount, samplesPerFrame);
        }

        // オーディオレンダラに加工したサンプルを書き戻します。
        int writeCount = nn::audio::WriteAuxReturnBuffer(&aux, readBuffer, readCount);
        if( writeCount != readCount )
        {
            // オーディオレンダラにすべてのサンプルを書き戻すことができませんでした。
            // 一般的には原因として以下の点が考えられます。
            //
            // - nn::audio::ReadAuxSendBuffer() の読み出し頻度が、処理負荷増等の理由により、低くなっている。
            // - その結果 nn::audio::ReadAuxSendBuffer() で読み込まれるサンプル数が増えている。
            // - 増えたすべてのサンプルを、 一度に nn::audio::WriteAuxReturnBuffer() で書き戻そうとしている。
            // - その結果 ReturnBuffer のバッファが不足しており、readCount に指定したすべてのサンプルを書き戻すことができなかった。
            //
            // この問題を回避するためには、以下の改善を検討してください。
            // - nn::audio::ReadAuxSendBuffer() で読み込むサンプル数を減らす
            // - nn::audio::ReadAuxSendBuffer() を呼び出す頻度を上げる( == Aux エフェクト処理の更新頻度を上げる)
            //
            NNS_LOG("Notice: Could not write back whole sample\n");
        }
        readCount = nn::audio::ReadAuxSendBuffer(&aux, readBuffer, channelCount * sampleCount);
    }
}

void UpdateEffectInputOutput(nn::audio::AuxType* aux, int8_t* targetBus, int busCount)
{
    nn::audio::SetAuxInputOutput(aux, targetBus,targetBus, busCount);
}

extern "C" void nnMain()
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));
    g_WaveBufferAllocator.Initialize(g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    g_EffectBufferAllocator.Initialize(g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));

    InitializeFileSystem();
    InitializeHidDevices();

    AudioEffectSampleStatus sampleStatus = { 0 };

    // レンダラのパラメータを指定します。
    nn::audio::AudioRendererParameter parameter;
    nn::audio::InitializeAudioRendererParameter(&parameter);
    parameter.sampleRate = RenderRate;
    parameter.sampleCount = RenderCount;
    parameter.mixBufferCount = 6;
    parameter.voiceCount = 24;
    parameter.subMixCount = 0;
    parameter.sinkCount = 1;
    parameter.effectCount = 3; // Aux, BufferMixer * 2
    parameter.performanceFrameCount = 0;

    // ミックスバッファとオーディオバスの関係を定義します。
    int channelCount = 2;
    int8_t mainBus[2];
    mainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
    mainBus[nn::audio::ChannelMapping_FrontRight] = 5;
    int8_t subBusA[2];
    subBusA[nn::audio::ChannelMapping_FrontLeft] = 2;
    subBusA[nn::audio::ChannelMapping_FrontRight] = 3;
    int8_t subBusB[2];
    subBusB[nn::audio::ChannelMapping_FrontLeft] = 0;
    subBusB[nn::audio::ChannelMapping_FrontRight] = 1;


    // パラメータがシステムでサポートされているかどうかを確認します。
    NN_ABORT_UNLESS(
        nn::audio::IsValidAudioRendererParameter(parameter),
        "Invalid AudioRendererParameter specified."
    );

    // レンダラのワークバッファを準備し AudioRenderer に登録します。
    size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
    void* workBuffer = g_Allocator.Allocate(workBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(workBuffer);

    nn::os::SystemEvent systemEvent;

    // レンダラを初期化します。
    nn::audio::AudioRendererHandle handle;
    NN_ABORT_UNLESS(
        nn::audio::OpenAudioRenderer(&handle, &systemEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
        "Failed to open AudioRenderer"
    );

    // AudioRendererConfig を初期化します。
    size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
    void* configBuffer = g_Allocator.Allocate(configBufferSize, nn::os::MemoryPageSize);
    NN_ABORT_UNLESS_NOT_NULL(configBuffer);
    nn::audio::AudioRendererConfig config;
    nn::audio::InitializeAudioRendererConfig(&config, parameter, configBuffer, configBufferSize);

    nn::audio::FinalMixType finalMix;
    nn::audio::AcquireFinalMix(&config, &finalMix, 4);

    // レンダラの出力先を用意します。
    nn::audio::DeviceSinkType deviceSink;
    // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
    // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
    // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
    nn::Result result = nn::audio::AddDeviceSink(&config, &deviceSink, &finalMix, mainBus, channelCount, "MainAudioOut");
    NN_ABORT_UNLESS(result.IsSuccess());

    // エフェクトバッファ用のメモリプールを準備します。
    // MemoryPool は nn::audio::AcquireMemoryPool() で渡される address 、 size の領域を書き換えることはありません。
    // そのため g_EffectBufferPoolMemory として準備した領域をすべてをメモリプールに利用することができます。
    // またこのサンプルでは WaveBuffer, Effect とメモリの用途別にアロケータを準備しています。
    // MemoryPool はその領域全体を「アタッチ」「デタッチ」しますので、
    // MemoryPool の操作に必要な粒度に応じてメモリ領域を分けておくことをおすすめします。
    // MemoryPool の詳しい利用法は @link ../../../Samples/Sources/Applications/AudioMemoryPool
    // Samples/Sources/Applications/AudioMemoryPool @endlink を参照してください。
    nn::audio::MemoryPoolType effectBufferPool;
    bool rt = nn::audio::AcquireMemoryPool(&config, &effectBufferPool, g_EffectBufferPoolMemory, sizeof(g_EffectBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
    rt = nn::audio::RequestAttachMemoryPool(&effectBufferPool);
    NN_ABORT_UNLESS(rt);

    // Aux エフェクトを用意します。
    nn::audio::AuxType aux;

    // Aux エフェクトで利用する Send Buffer & Return Buffer を確保します。
    // ここでは 4 オーディオフレーム分のバッファを確保しますので、最大で 5ms * 4 フレーム = 20ms の遅延が発生します。
    // このバッファサイズを小さくすることで遅延量も減りますが、後述するユーザー側の処理をより高頻度に行う必要がでてきます。
    const int AuxBufferFrameCount = 16;
    size_t auxBufferSize = nn::audio::GetRequiredBufferSizeForAuxSendReturnBuffer(&parameter, AuxBufferFrameCount, sizeof(subBusA));
    void* sendBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(sendBuffer);
    void* returnBuffer = g_EffectBufferAllocator.Allocate(auxBufferSize, nn::audio::BufferAlignSize);
    NN_ABORT_UNLESS_NOT_NULL(returnBuffer);

    // Aux エフェクトを FinalMix に追加します。
    result = nn::audio::AddAux(&config, &aux, &finalMix, sendBuffer, returnBuffer, auxBufferSize);
    NN_ABORT_UNLESS(result.IsSuccess());

    // Aux エフェクトへの入力チャンネルを設定します。
    int auxInputChannelCount = sizeof(subBusA);
    nn::audio::SetAuxInputOutput(&aux, subBusA, subBusA, auxInputChannelCount);

    // BufferMixer エフェクトを用意します。
    nn::audio::BufferMixerType mixerA;
    nn::audio::BufferMixerType mixerB;
    nn::audio::AddBufferMixer(&config, &mixerA, &finalMix);
    nn::audio::AddBufferMixer(&config, &mixerB, &finalMix);
    nn::audio::SetBufferMixerVolume(&mixerA, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerA, 1, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerB, 0, 1.0f);
    nn::audio::SetBufferMixerVolume(&mixerB, 1, 1.0f);
    nn::audio::SetBufferMixerInputOutput(&mixerA, subBusA, mainBus, sizeof(mainBus));
    nn::audio::SetBufferMixerInputOutput(&mixerB, subBusB, mainBus, sizeof(mainBus));

    // 設定したパラメータをレンダラに反映させます。
    result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
    NN_ABORT_UNLESS(result.IsSuccess());

    // レンダリングを開始します。
    result = nn::audio::StartAudioRenderer(handle);
    NN_ABORT_UNLESS(result.IsSuccess());

    // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
    nn::audio::MemoryPoolType waveBufferMemoryPool;
    rt = nn::audio::AcquireMemoryPool(&config, &waveBufferMemoryPool, g_WaveBufferPoolMemory, sizeof(g_WaveBufferPoolMemory));
    NN_ABORT_UNLESS(rt);
    rt = nn::audio::RequestAttachMemoryPool(&waveBufferMemoryPool);
    NN_ABORT_UNLESS(rt);

    // SE を読み込みます。
    nn::audio::VoiceType voiceSe[SeCount];
    nn::audio::WaveBuffer waveBufferSe[SeCount];
    nn::audio::AdpcmHeaderInfo* header[SeCount];
    void* dataSe[SeCount];

    for (int i = 0; i < SeCount; ++i)
    {
        header[i] = reinterpret_cast<nn::audio::AdpcmHeaderInfo*>(g_WaveBufferAllocator.Allocate(sizeof(nn::audio::AdpcmHeaderInfo), NN_ALIGNOF(nn::audio::AdpcmHeaderInfo)));
        std::size_t dataSeSize = ReadAdpcmFile(header[i], &dataSe[i], g_SeFileNames[i]);
        nn::audio::AcquireVoiceSlot(&config, &voiceSe[i], header[i]->sampleRate, 1, nn::audio::SampleFormat_Adpcm, nn::audio::VoiceType::PriorityHighest, &header[i]->parameter, sizeof(nn::audio::AdpcmParameter));
        nn::audio::SetVoiceDestination(&config, &voiceSe[i], &finalMix);

        waveBufferSe[i].buffer = dataSe[i];
        waveBufferSe[i].size = dataSeSize;
        waveBufferSe[i].startSampleOffset = 0;
        waveBufferSe[i].endSampleOffset = header[i]->sampleCount;
        waveBufferSe[i].loop = false;
        waveBufferSe[i].isEndOfStream = false;
        waveBufferSe[i].pContext = &header[i]->loopContext;
        waveBufferSe[i].contextSize = sizeof(nn::audio::AdpcmContext);

        nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
        nn::audio::SetVoicePlayState(&voiceSe[i], nn::audio::VoiceType::PlayState_Play);
        auto bus = (i == 0) ? subBusA : subBusB;
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, bus[nn::audio::ChannelMapping_FrontLeft]);
        nn::audio::SetVoiceMixVolume(&voiceSe[i], &finalMix, 0.707f / 2, 0, bus[nn::audio::ChannelMapping_FrontRight]);
    }

    PrintUsage();

    // オーディオレンダラからサンプルデータを受け取るバッファを準備します。
    int32_t* readBuffer = static_cast<int32_t*>(g_Allocator.Allocate(auxBufferSize));
    NN_ABORT_UNLESS_NOT_NULL(readBuffer);
    memset(readBuffer, 0, auxBufferSize);

    bool running = true;

    while (running)
    {
        systemEvent.Wait();

        nn::hid::NpadButtonSet npadButtonCurrent = {};
        nn::hid::NpadButtonSet npadButtonDown = {};
        nn::hid::AnalogStickState analogStickStateL = {};
        nn::hid::AnalogStickState analogStickStateR = {};

        // Npad の入力を取得します。
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).Test<nn::hid::NpadStyleFullKey>())
        {
            static nn::hid::NpadFullKeyState npadFullKeyState = {};
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::No1);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadFullKeyState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadFullKeyState = state;
        }
        if (nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).Test<nn::hid::NpadStyleHandheld>())
        {
            static nn::hid::NpadHandheldState npadHandheldState = {};
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, nn::hid::NpadId::Handheld);
            npadButtonCurrent |= state.buttons;
            npadButtonDown |= state.buttons & ~npadHandheldState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            npadHandheldState = state;
        }

        // DebugPad の入力を取得します。
        {
            static nn::hid::DebugPadState debugPadState = {};
            nn::hid::DebugPadButtonSet debugPadButtonCurrent = {};
            nn::hid::DebugPadButtonSet debugPadButtonDown = {};
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            debugPadButtonCurrent |= state.buttons;
            debugPadButtonDown |= state.buttons & ~debugPadState.buttons;
            analogStickStateL.x += state.analogStickL.x;
            analogStickStateL.y += state.analogStickL.y;
            analogStickStateR.x += state.analogStickR.x;
            analogStickStateR.y += state.analogStickR.y;
            debugPadState = state;
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonCurrent, debugPadButtonCurrent);
            nns::audio::ConvertDebugPadButtonsToNpadButtons(&npadButtonDown, debugPadButtonDown);
        }

        // 各 Effect のパラメータを更新します。
        {
            // Aux については、独自定義のエフェクト処理も行います。
            ProcessUserEffect(aux, readBuffer, parameter.sampleCount, channelCount, auxInputChannelCount);

            // 全 Effect の InputOutput を更新します。
            if(npadButtonDown.Test<::nn::hid::NpadButton::L>())
            {
                int8_t* targetBus = (sampleStatus.seSourceFlag) ? subBusA : subBusB;
                UpdateEffectInputOutput(&aux, targetBus, channelCount);
                sampleStatus.seSourceFlag = !sampleStatus.seSourceFlag;
            }

            if (npadButtonDown.Test<::nn::hid::NpadButton::Minus>())
            {
                PrintUsage();
            }

            if ( npadButtonDown.Test<::nn::hid::NpadButton::A>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::B>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::X>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::Y>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::L>() ||
                 npadButtonDown.Test<::nn::hid::NpadButton::R>())
            {
                NNS_LOG("Effected SE No.%d\n",
                    sampleStatus.seSourceFlag ? 1 : 0);
            }
        }

        //SE を再生します（同じ SE を多重再生しません）。
        for (int i = 0; i < SeCount; ++i)
        {
            // SE番号 {0, 1, 2, 3, 4, 5} が、ボタン {A, B, X, Y, L, R} に対応します。
            if(npadButtonDown.Test(i))
            {
                if (nn::audio::GetReleasedWaveBuffer(&voiceSe[i]) != nullptr)
                {
                    nn::audio::AppendWaveBuffer(&voiceSe[i], &waveBufferSe[i]);
                }
            }
        }

        // 各エフェクトに施したパラメータ変更をオーディオレンダラに反映します。
        // 反映するまでは、パラメータ変更は有効になりません。
        // nn::audio::RequestUpdateAudioRenderer() の呼び出し中は、各エフェクトへのアクセスは避けてください。
        result = nn::audio::RequestUpdateAudioRenderer(handle, &config);
        NN_ABORT_UNLESS(result.IsSuccess());

        //  サンプルを終了する
        if(npadButtonDown.Test<::nn::hid::NpadButton::Plus>())
        {
            running = false;
        }
    }

    // レンダリングを終了します。
    nn::audio::StopAudioRenderer(handle);
    nn::audio::CloseAudioRenderer(handle);
    nn::os::DestroySystemEvent(systemEvent.GetBase());

    // メモリを解放します。
    for (int i = 0; i < SeCount; ++i)
    {
        if (dataSe[i])
        {
            g_WaveBufferAllocator.Free(dataSe[i]);
            dataSe[i] = nullptr;
        }
        if (header[i])
        {
            g_WaveBufferAllocator.Free(header[i]);
            header[i] = nullptr;
        }
    }

    if (configBuffer)
    {
        g_Allocator.Free(configBuffer);
        configBuffer = nullptr;
    }
    if (workBuffer)
    {
        g_Allocator.Free(workBuffer);
        workBuffer = nullptr;
    }
    if (sendBuffer)
    {
        g_EffectBufferAllocator.Free(sendBuffer);
        sendBuffer = nullptr;
    }
    if (returnBuffer)
    {
        g_EffectBufferAllocator.Free(returnBuffer);
        returnBuffer = nullptr;
    }
    if (readBuffer)
    {
        g_Allocator.Free(readBuffer);
        readBuffer = nullptr;
    }

    FinalizeFileSystem();
}  // NOLINT(readability/fn_size)
