﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkSoundActor.cpp,PageSampleAtkSoundActor}
 *
 * @brief
 *  nn::atk ライブラリのサウンドアクター機能の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkSoundActor サウンドアクターを用いた再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリのサウンドアクター機能に関するサンプルプログラムの解説です。
 *
 * @section PageSampleAtkSoundActor_SectionBrief 概要
 *  Atk ライブラリのサウンドアクター機能を用いてサウンドを再生するサンプルです。
 *
 * @section PageSampleAtkSoundActor_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkSoundActor Samples/Sources/Applications/AtkSoundActor @endlink 以下にあります。
 *
 * @section PageSampleAtkSoundActor_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkSoundActor_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkSoundActor_SectionPrecaution 注意事項
 *  実行中のログが、画面上とコンソールに表示され、
 *  起動直後のログとして操作方法が表示されます。
 *
 *  PC 版ではウィンドウは立ち上がらず、コンソール表示のみになる点にご注意ください。
 *
 * @section PageSampleAtkSoundActor_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkSoundActor_SectionDetail 解説
 *
 * @subsection PageSampleAtkSoundActor_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkSoundActor.cpp
 *  @includelineno AtkSoundActor.cpp
 *
 * @subsection PageSampleAtkSoundActor_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです。(AtkSimple と違う部分には追加要素と書かれています)。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - AtkSimple で実行している初期化処理
 *    - SoundActor の初期化(追加要素)
 *  - SoundHeap にウェーブサウンドとシーケンスサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キーによる処理
 *      - サウンドアクターでの再生・停止処理を実行(追加要素)
 *      - サウンドアクターの音量の増減(追加要素)
 *      - サウンドアクターのパンのかかり具合の変更(追加要素)
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *    - SoundActor の終了処理(追加要素)
 *
 *
 *  キー操作を行うことで、サウンドアクターのパラメータを変更する事ができます。
 *  変更することのできるパラメータは以下になります。
 *  - 音量
 *  - パン
 *
 *  また、このサンプルでは取り扱っていませんが、サウンドアクターでは以下のパラメータも変更可能です。
 *  - ピッチ
 *  - LPF
 *  - BiquadFilter
 *  - 同時に再生可能なサウンド数
 *
 */

#include "Common.fsid"

#include <nn/atk.h>

#include <nns/atk/atk_SampleCommon.h>

#include <nns/nns_Log.h>

namespace
{
    const char Title[] = "AtkSoundActor";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;

    nn::atk::SoundHandle        g_SoundHandle;
    nn::atk::SoundHandle        g_SoundHandleHold;

    const float ActorPanMin        = -1.0f;
    const float ActorPanMax        =  1.0f;
    const float ActorPanDefault    =  0.0f;
    const float ActorVolumeMin     =  0.1f;
    const float ActorVolumeMax     =  2.0f;
    const float ActorVolumeDefault =  1.0f;
    const float ControlPadStep     =  0.05f;

    float g_ActorPan;
    float g_ActorVolume;

    nn::atk::SoundActor g_SoundActor;
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    // SoundActor の初期化
    g_ActorPan    = ActorPanDefault;
    g_ActorVolume = ActorVolumeDefault;
    g_SoundActor.Initialize( &g_SoundArchivePlayer );
    g_SoundActor.SetPlayableSoundCount( 0, 2 );
}

void FinalizeAtk()
{
    g_SoundActor.Finalize();

    g_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );
}

void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    g_SoundHandle.Stop( 0 );

    bool result = g_SoundActor.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", debugLabelName, result);
}

void PlayWithHoldSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    bool result = g_SoundActor.HoldSound( &g_SoundHandleHold, soundId ).IsSuccess();
    NNS_LOG("HoldSound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]              StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]              StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]              Stop Sound\n");
    NNS_LOG("[RIGHT]          Modify Actor Pan(Up)\n");
    NNS_LOG("[UP]             Modify Actor Volume(Up)\n");
    NNS_LOG("[LEFT]           Modify Actor Pan(Down)\n");
    NNS_LOG("[DOWN]           Modify Actor Volume(Down)\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[+/Start][Space] Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

bool UpdateAtk()
{
    // StartSound / Stop
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
    {
        PlayWithStartSound(SE_YOSHI, "SE_YOSHI");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART");
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
    {
        g_SoundActor.StopAllSound(3);
    }

    // HoldSound
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Right >() )
    {
        g_ActorPan += ControlPadStep;
        if (g_ActorPan > ActorPanMax)
        {
            g_ActorPan = ActorPanMax;
        }
        g_SoundActor.SetOutputPan(nn::atk::OutputDevice_Main, g_ActorPan);
        PlayWithHoldSound(SE_WIHAHO, "SE_WIHAHO");
        NNS_LOG("SetActorPan ... (%f)\n", g_ActorPan);
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        g_ActorVolume += ControlPadStep;
        if (g_ActorVolume > ActorVolumeMax)
        {
            g_ActorVolume = ActorVolumeMax;
        }
        g_SoundActor.SetVolume(g_ActorVolume);
        PlayWithHoldSound(SE_WIHAHO, "SE_WIHAHO");
        NNS_LOG("SetActorVolume ... (%f)\n", g_ActorVolume);
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Left >() )
    {
        g_ActorPan -= ControlPadStep;
        if (g_ActorPan < ActorPanMin)
        {
            g_ActorPan = ActorPanMin;
        }
        g_SoundActor.SetOutputPan(nn::atk::OutputDevice_Main, g_ActorPan);
        PlayWithHoldSound(SE_WIHAHO, "SE_WIHAHO");
        NNS_LOG("SetActorPan ... (%f)\n", g_ActorPan);
    }
    else if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Down >() )
    {
        g_ActorVolume -= ControlPadStep;
        if (g_ActorVolume < ActorVolumeMin)
        {
            g_ActorVolume = ActorVolumeMin;
        }
        g_SoundActor.SetVolume(g_ActorVolume);
        PlayWithHoldSound(SE_WIHAHO, "SE_WIHAHO");
        NNS_LOG("SetActorVolume ... (%f)\n", g_ActorVolume);
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeAtk();

    LoadData();

    PrintUsage();

    for ( ;; )
    {
        nns::atk::UpdateHidDevices();

        if ( !UpdateAtk() )
        {
            break;
        }

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    FinalizeAtk();
    Finalize();
}

