﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include "SoundRecorder.h"

namespace
{
}

SoundRecorder::SoundRecorder() NN_NOEXCEPT
: m_IsInitialized(false)
, m_IsRecording(false)
, m_pOutputFileName("rec.wav")
, m_Recorder()
, m_RecordingFileStream()
{
}

SoundRecorder::~SoundRecorder() NN_NOEXCEPT
{
}

void SoundRecorder::Initialize(void* recordingBuffer, size_t recordingBufferSize, void* pThreadStack, size_t threadStackSize) NN_NOEXCEPT
{
    if ( m_IsInitialized )
    {
        NN_LOG( "[REC] SoundRecorder::Initialize : already initalized.\n" );
        return;
    }

    if ( threadStackSize < nn::atk::DeviceOutRecorder::RequiredThreadStackSize )
    {
        NN_LOG( "[REC] SoundRecorder::Initialize : invalid thread stack size (actual %u, required %u)\n",
            threadStackSize, nn::atk::DeviceOutRecorder::RequiredThreadStackSize);
        return;
    }

    NN_ABORT_UNLESS_NOT_NULL(recordingBuffer);
    NN_ABORT_UNLESS_NOT_NULL(pThreadStack);
    m_Recorder.Initialize(recordingBuffer, recordingBufferSize, pThreadStack, threadStackSize);
    m_IsInitialized = true;
}

void SoundRecorder::Finalize() NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        NN_LOG( "[REC] SoundRecorder::Finalize : still not initalized.\n" );
        return;
    }

    // 録音中の場合は録音停止後少し待ってから終了処理を行う
    if ( m_IsRecording )
    {
        StopRecording();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        m_IsRecording = false;
    }

    m_Recorder.Finalize();
    m_IsInitialized = false;
}

size_t SoundRecorder::GetRequiredMemorySize() NN_NOEXCEPT
{
    return m_Recorder.GetRequiredMemorySizeForRecording();
}

void SoundRecorder::StartRecording(uint32_t frames) NN_NOEXCEPT
{
    if ( !m_IsInitialized )
    {
        NN_LOG( "[REC] SoundRecorder::StartRecording : recorder is not initialized: %s\n" );
        return;
    }

    nn::Result result = nn::fs::DeleteFile(m_pOutputFileName);
    if(!(result.IsSuccess() || nn::fs::ResultPathNotFound::Includes(result)))
    {
        NN_LOG( "[REC] SoundRecorder::StartRecording : cannot delete output file: %s\n", m_pOutputFileName );
        return;
    }

    nn::fs::CreateFile(m_pOutputFileName, 0);

    if ( m_RecordingFileStream.Open(
        m_pOutputFileName,
        nn::atk::detail::fnd::FileStream::AccessMode_AllowAppendAndWrite
        ).IsFailed() )
    {
        NN_LOG( "[REC] SoundRecorder::StartRecording : failed to open wav file: %s\n", m_pOutputFileName );
        return;
    }

    // 先頭の無音区間のトリミングを有効にする
    nn::atk::WavOutRecorder::RecordingOptions options;
    options.SetLeadSilenceTrimmingEnabled( true );
    options.SetMaxFrames( frames );

    m_IsRecording = m_Recorder.Start( m_RecordingFileStream, options );

    if ( !m_IsRecording )
    {
        NN_LOG( "[REC] SoundRecorder::StartRecording : failed to start recording\n" );
        m_RecordingFileStream.Close();
    }
}

void SoundRecorder::StopRecording() NN_NOEXCEPT
{
    if ( !m_IsInitialized || !m_IsRecording )
    {
        return;
    }
    m_Recorder.Stop(true);
    m_RecordingFileStream.Flush();
    m_RecordingFileStream.Close();
    m_IsRecording = false;
}

void SoundRecorderUtil::MountContentsDirectory(const char* pMountName) NN_NOEXCEPT
{
    // 録音用に Contents ディレクトリをマウント
    // アプリケーションバイナリの配置されているディレクトリの隣に、Content ディレクトリがある
    char filePath[nn::fs::EntryNameLengthMax];
    char* pPath = nn::os::GetHostArgv()[ 0 ];
    char* pEscapeEnd = strrchr( pPath, '\\' );
    char* pSlashEnd = strrchr( pPath, '/');
    char* pEnd;
    if (pEscapeEnd > pSlashEnd)
    {
        pEnd = pEscapeEnd;
    }
    else
    {
        pEnd = pSlashEnd;
    }

    NN_ABORT_UNLESS_NOT_NULL( pEnd );
    ptrdiff_t length = pEnd - pPath + 1;
    strncpy( filePath, pPath, length );
#if defined(NN_BUILD_APISET_NX)
    strcpy( filePath + length, "..\\Contents\\NX" );
#elif defined(NN_BUILD_APISET_GENERIC)
    strcpy( filePath + length, "..\\Contents\\Generic" );
#else
    NN_ABORT("Unexpected spec.");
#endif
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::MountHost( pMountName, filePath ) );
}

void SoundRecorderUtil::UnmountContentsDirectory(const char* pMountName) NN_NOEXCEPT
{
    nn::fs::Unmount(pMountName);
}
