﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "StartInfo.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    StartInfoCheckModule::IStartInfoChecker* g_pCurrent = nullptr;
    StartInfoCheckModule::StartInfoCheckerForStartOffset g_StartInfoCheckerForStartOffset;
    StartInfoCheckModule::StartInfoCheckerForPlayerId g_StartInfoCheckerForPlayerId;
    StartInfoCheckModule::StartInfoCheckerForPlayerPriority g_StartInfoCheckerForPlayerPriority;
    StartInfoCheckModule::StartInfoCheckerForActorPlayerId g_StartInfoCheckerForActorPlayerId;
    StartInfoCheckModule::StartInfoCheckerForSoundStopCallback g_StartInfoCheckerForSoundStopCallback;
    StartInfoCheckModule::StartInfoCheckerForDelayCount g_StartInfoCheckerForDelayCount;
    StartInfoCheckModule::StartInfoCheckerForDelayTime g_StartInfoCheckerForDelayTime;
    StartInfoCheckModule::StartInfoCheckerForUpdateType g_StartInfoCheckerForUpdateType;

    StartInfoCheckModule::IStartInfoChecker* g_pCheckList[] =
    {
        &g_StartInfoCheckerForStartOffset,
        &g_StartInfoCheckerForPlayerId,
        &g_StartInfoCheckerForPlayerPriority,
        &g_StartInfoCheckerForActorPlayerId,
        &g_StartInfoCheckerForSoundStopCallback,
        &g_StartInfoCheckerForDelayCount,
        &g_StartInfoCheckerForDelayTime,
        &g_StartInfoCheckerForUpdateType
    };

    const int CheckListMin = 0;
    const int CheckListMax = sizeof(g_pCheckList) / sizeof(g_pCheckList[0]);
    const int CheckStartIndex = 0;

    FlagList g_LocalFlagList(nullptr, 0);
}

NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForStartOffset::StartOffsetForMilliSeconds);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForStartOffset::StartOffsetForTick);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForStartOffset::StartOffsetForSample);

NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayTime::DefaultDelayTime);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayTime::DelayTimeMin);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayTime::DelayTimeMax);

NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayCount::DefaultDelayCount);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayCount::DelayCountMin);
NN_DEFINE_STATIC_CONSTANT(const int StartInfoCheckModule::StartInfoCheckerForDelayCount::DelayCountMax);

void StartInfoCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_CommonObject.Initialize();

    m_StartInfoCheckerIndex = CheckStartIndex;

    g_pCurrent = GetCurrentStartInfoChecker();
    g_pCurrent->Initialize();
    g_pCurrent->SetCommonObjectPointer(&m_CommonObject);

    m_IsPause = false;

}

void StartInfoCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
}

void StartInfoCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void StartInfoCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    NN_LOG("[A]            StartSound SEQ  (SEQ_MARIOKART)\n");
    NN_LOG("[X]            StartSound WSD  (WSD_YOSHI_ADPCM)\n");
    NN_LOG("[R + X]        StartSound WSD  (WSD_YOSHI_PCM16)\n");
    NN_LOG("[Y]            StartSound STRM (STRM_PIANO16_ADPCM)\n");
    NN_LOG("[R + Y]        StartSound STRM (STRM_PIANO16_PCM16)\n");
    NN_LOG("[B]            Stop Sound\n");
    NN_LOG("[R + B]        Pause Sound\n");
    NN_LOG("[Up/Down]      Change Parameter\n");
    NN_LOG("[Left/Right]   Change StartInfo Type\n");
    m_CommonObject.GetAtkProfiler().PrintUsage();
}

void StartInfoCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(g_pCurrent);

    int processCount = m_CommonObject.GetAtkProfiler().UpdateInput();

    if (processCount == 0)
    {
        // StartSound / StopSound
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
        {
            g_pCurrent->PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART");
        }
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                g_pCurrent->PlayWithStartSound(WSD_YOSHI_PCM16, "WSD_YOSHI_PCM16");
            }
            else
            {
                g_pCurrent->PlayWithStartSound(WSD_YOSHI_ADPCM, "WSD_YOSHI_ADPCM");
            }
        }
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                g_pCurrent->PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16");
            }
            else
            {
                g_pCurrent->PlayWithStartSound(STRM_PIANO16_ADPCM, "STRM_PIANO16_ADPCM");
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B>())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                m_IsPause = !m_IsPause;
                m_CommonObject.GetSoundHandle().Pause(m_IsPause, 0);
            }
            else
            {
                m_CommonObject.GetSoundHandle().Stop(0);
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >())
        {
            g_pCurrent->IncrementValue();
            g_pCurrent->PrintValue();
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >())
        {
            g_pCurrent->DecrementValue();
            g_pCurrent->PrintValue();
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >())
        {
            PrevChecker();
        }
        else if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >())
        {
            NextChecker();
        }
    }

    m_CommonObject.GetAtkProfiler().Show(m_CommonObject.GetSoundHandle());
}

void StartInfoCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void StartInfoCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& StartInfoCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

StartInfoCheckModule::IStartInfoChecker* StartInfoCheckModule::GetCurrentStartInfoChecker() NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(m_StartInfoCheckerIndex, CheckListMin, CheckListMax);
    return g_pCheckList[m_StartInfoCheckerIndex];
}

void StartInfoCheckModule::NextChecker() NN_NOEXCEPT
{
    g_pCurrent->Finalize();
    g_pCurrent->SetCommonObjectPointer(nullptr);

    m_StartInfoCheckerIndex++;
    if (m_StartInfoCheckerIndex >= CheckListMax)
    {
        m_StartInfoCheckerIndex = CheckListMin;
    }

    g_pCurrent = GetCurrentStartInfoChecker();
    g_pCurrent->Initialize();
    g_pCurrent->SetCommonObjectPointer(&m_CommonObject);

    g_pCurrent->PrintCheckerType();
    g_pCurrent->PrintValue();
}

void StartInfoCheckModule::PrevChecker() NN_NOEXCEPT
{
    g_pCurrent->Finalize();
    g_pCurrent->SetCommonObjectPointer(nullptr);

    m_StartInfoCheckerIndex--;
    if (m_StartInfoCheckerIndex < CheckListMin)
    {
        m_StartInfoCheckerIndex = CheckListMax - 1;
    }

    g_pCurrent = GetCurrentStartInfoChecker();
    g_pCurrent->Initialize();
    g_pCurrent->SetCommonObjectPointer(&m_CommonObject);

    g_pCurrent->PrintCheckerType();
    g_pCurrent->PrintValue();
}
