﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MixParameter.h"
#include <nn/util/util_TFormatString.h>

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    const float MixVolumeDelta = 0.04f;
    const float MixVolumeMax = nn::audio::VoiceType::GetVolumeMax();
    const float MixVolumeMin = nn::audio::VoiceType::GetVolumeMin();

    const char* RecordMountName = "record";
    const char* RecordFilePath = "record:/output.wav";

    enum FlagType
    {
        FlagType_Enable4chDelay,       //< 4ch Delay を有効化
        FlagType_Enable4chReverb,      //< 4ch Reverb を有効化
        FlagType_Enable4chAux,         //< 4ch Aux を有効化
        FlagType_Num
    };

    FlagElement g_LocalElements[] =
    {
        { "Enable4chDelay", false },
        { "Enable4chReverb", false },
        { "Enable4chAux", false },
    };

    FlagList g_LocalFlagList(g_LocalElements, sizeof(g_LocalElements) / sizeof(g_LocalElements[0]));
}

void MixModeCheckModule::CommonObjectForMixMode::OnPostStartSound() NN_NOEXCEPT
{
    m_pOwner->SetMixVolume();

    if ( m_pOwner->GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chDelay) ||
         m_pOwner->GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chReverb) ||
         m_pOwner->GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chAux) )
    {
        GetSoundHandle().SetEffectSend(nn::atk::AuxBus_A, 0.6f);
        GetSoundHandle().SetMainSend(-1.0f);
    }
}

void MixModeCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    CommonObject::InitializeParam param;
    param.GetSoundSystemParam().enableRecordingFinalOutputs = true;
    param.GetSoundSystemParam().enableUnusedEffectChannelMuting = true;
    m_CommonObject.Initialize( param );
    m_CommonObject.SetOwner(this);

    nn::atk::SoundSystem::SetOutputMode(nn::atk::OutputMode_Surround);

    // TODO: マルチトラックストリームのテスト
    // TODO: マルチトラックシーケンスのテスト
    //m_MixStreamTrackPiano.fC = 1.0f;
    //m_MixStreamTrackDrum.rL  = 1.0f;
    //m_MixStreamTrackDrum.rR  = 1.0f;
    //m_MixSequenceTrack1.fC   = 1.0f;
    //m_MixSequenceTrack2.rL   = 1.0f;
    //m_MixSequenceTrack2.rR   = 1.0f;
    //m_MixSequenceTrack3.lfe  = 1.0f;

    // delay の設定
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chDelay) )
    {
        m_Delay.SetEnabled( true );
        m_Delay.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
        m_Delay.SetChannelMode(nn::atk::EffectBase::ChannelMode_4Ch);
        auto delayBufferSize = nn::util::align_up(m_Delay.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        m_DelayBuffer = nns::atk::AllocateForMemoryPool(delayBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(m_DelayBuffer);
        bool isDelayAddSuccessed = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &m_Delay, m_DelayBuffer, delayBufferSize);
        NN_LOG( "Delay  %d\n", isDelayAddSuccessed );
    }
    // reverb の設定
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chReverb) )
    {
        m_Reverb.SetEnabled( true );
        m_Reverb.SetSampleRate(nn::atk::EffectBase::SampleRate_48000);
        m_Reverb.SetChannelMode(nn::atk::EffectBase::ChannelMode_4Ch);
        auto reverbBufferSize = nn::util::align_up(m_Reverb.GetRequiredMemSize(), nn::audio::MemoryPoolType::SizeGranularity);
        m_ReverbBuffer = nns::atk::AllocateForMemoryPool(reverbBufferSize, nn::audio::MemoryPoolType::AddressAlignment);
        NN_ABORT_UNLESS_NOT_NULL(m_ReverbBuffer);
        bool isReverbAddSuccessed = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &m_Reverb, m_ReverbBuffer, reverbBufferSize);
        NN_LOG( "Reverb %d\n", isReverbAddSuccessed );
    }
    // aux の設定
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chAux) )
    {
        m_UserEffectAux.SetEnabled( true );
        m_UserEffectAux.SetAudioFrameCount( 5 );
        m_UserEffectAux.SetChannelCount( 4 );
        m_UserEffectAux.SetSampleRate( 48000 );
        size_t auxBufferSize = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize( &m_UserEffectAux );
        m_UserEffectAuxBuffer = nns::atk::AllocateForMemoryPool( auxBufferSize, nn::audio::BufferAlignSize );
        NN_ABORT_UNLESS_NOT_NULL( m_UserEffectAuxBuffer );
        bool isSuccess = nn::atk::SoundSystem::AppendEffect( nn::atk::AuxBus_A, &m_UserEffectAux, m_UserEffectAuxBuffer, auxBufferSize );
        NN_LOG( "Aux  %d\n", isSuccess );
    }

    m_IsAuxBusFadedOut = false;

    SoundRecorderUtil::MountContentsDirectory(RecordMountName);
    m_Recorder.SetOutputFileName(RecordFilePath);

    m_RecorderStackBuffer = nns::atk::Allocate(
        nn::atk::DeviceOutRecorder::RequiredThreadStackSize,
        nn::os::ThreadStackAlignment);
    size_t recorderBufferSize = m_Recorder.GetRequiredMemorySize();
    m_RecorderBuffer = nns::atk::Allocate( recorderBufferSize );
    m_Recorder.Initialize(m_RecorderBuffer, recorderBufferSize, m_RecorderStackBuffer, nn::atk::DeviceOutRecorder::RequiredThreadStackSize);

    for( int i = 0; i < nn::atk::ChannelIndex_Count; i++ )
    {
        m_MixVolume.channel[i] = 0.0f;
    }
    m_MixVolume.frontLeft = 1.0f;
    m_MixVolume.frontRight = 1.0f;
}

void MixModeCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chDelay) )
    {
        m_Delay.SetEnabled(false);
        // エフェクトが削除可能になるまで待つ
        while (!m_Delay.IsRemovable())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(16));
        }
    }
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chReverb) )
    {
        m_Reverb.SetEnabled(false);
        // エフェクトが削除可能になるまで待つ
        while (!m_Reverb.IsRemovable())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(16));
        }
    }
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chAux) )
    {
        m_UserEffectAux.SetEnabled( false );
        while ( !m_UserEffectAux.IsRemovable() )
        {
            nn::os::SleepThread( nn::TimeSpan::FromMicroSeconds( 16 ) );
        }
    }

    nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
    m_Recorder.Finalize();

    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chDelay) )
    {
        nns::atk::FreeForMemoryPool(m_DelayBuffer);
    }
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chReverb) )
    {
        nns::atk::FreeForMemoryPool(m_ReverbBuffer);
    }
    if ( GetLocalFlagList().IsFlagEnabled(FlagType_Enable4chAux) )
    {
        nns::atk::FreeForMemoryPool(m_UserEffectAuxBuffer);
    }

    nns::atk::Free(m_RecorderBuffer);
    nns::atk::Free(m_RecorderStackBuffer);

    SoundRecorderUtil::UnmountContentsDirectory(RecordMountName);

    m_CommonObject.SetOwner(nullptr);
    m_CommonObject.Finalize();
}

void MixModeCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void MixModeCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Left/Right]   Change MixChannel\n");
    NN_LOG("[Up/Down]      Change MixVolume\n");
    NN_LOG("[R + Down]     Start/Stop Recording\n");
    NN_LOG("[R + Up]       Fade AuxBus\n");
}

void MixModeCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
        {
            if (!m_Recorder.IsRecording())
            {
                NN_LOG("StartRecording...\n");
                m_Recorder.StartRecording();
            }
            else
            {
                NN_LOG("StopRecording...\n");
                m_Recorder.StopRecording();
            }
            return;
        }
    }

    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
        {
            FadeAuxBusVolume();
            return;
        }
    }

    m_CommonObject.UpdateInput();

    // ミックスボリュームの調整
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        m_MixChannel -= 1;
        if ( m_MixChannel < 0 )
        {
            m_MixChannel = nn::atk::ChannelIndex_Lfe;
        }
        SetMixVolume();
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        m_MixChannel += 1;
        if ( m_MixChannel > nn::atk::ChannelIndex_Lfe )
        {
            m_MixChannel = 0;
        }
        SetMixVolume();
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        m_MixVolume.channel[m_MixChannel] += MixVolumeDelta;
        if ( m_MixVolume.channel[m_MixChannel] > MixVolumeMax )
        {
            m_MixVolume.channel[m_MixChannel] = MixVolumeMax;
        }
        SetMixVolume();
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Down >() )
    {
        m_MixVolume.channel[m_MixChannel] -= MixVolumeDelta;
        if ( m_MixVolume.channel[m_MixChannel] < MixVolumeMin )
        {
            m_MixVolume.channel[m_MixChannel] = MixVolumeMin;
        }
        SetMixVolume();
    }
}

void MixModeCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void MixModeCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& MixModeCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

// サラウンド振り分け設定
void MixModeCheckModule::SetMixVolume() NN_NOEXCEPT
{
    // ウェーブサウンドのサラウンド振り分け
    nn::atk::WaveSoundHandle waveSoundHandle(&m_CommonObject.GetSoundHandle());
    waveSoundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
    waveSoundHandle.SetMixVolume(0, m_MixVolume);
    waveSoundHandle.SetMixVolume(1, m_MixVolume);

    // ストリームサウンドのトラック毎のサラウンド振り分け
    nn::atk::StreamSoundHandle streamSoundHandle(&m_CommonObject.GetSoundHandle());
    nn::atk::StreamSoundHandle::TrackBitFlagSet streamBitFlag;
    streamBitFlag.Reset();
    streamBitFlag[0] = true;
    streamSoundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
    streamSoundHandle.SetTrackMixVolume(streamBitFlag, 0, m_MixVolume);
    streamSoundHandle.SetTrackMixVolume(streamBitFlag, 1, m_MixVolume);

    // TODO: マルチトラックストリームのテスト
    //streamBitFlag.Reset();
    //streamBitFlag[0] = true;
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 0, m_MixStreamTrackPiano);
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 1, m_MixStreamTrackPiano);
    //streamBitFlag.Reset();
    //streamBitFlag[1] = true;
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 0, m_MixVolume);
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 1, m_MixVolume);
    //sequenceBitFlag.Reset();
    //sequenceBitFlag[2] = true;
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 0, m_MixStreamTrackDrum);
    //streamSoundHandle.SetTrackMixVolume(streamBitFlag, 1, m_MixStreamTrackDrum);

    // シーケンスサウンドのトラック毎のサラウンド振り分け
    nn::atk::SequenceSoundHandle sequenceSoundHandle(&m_CommonObject.GetSoundHandle());
    nn::atk::SequenceSoundHandle::TrackBitFlagSet sequenceBitFlag;
    sequenceBitFlag.Reset();
    sequenceBitFlag[0] = true;
    sequenceSoundHandle.SetMixMode(nn::atk::MixMode_MixVolume);
    sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 0, m_MixVolume);
    sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 1, m_MixVolume);

    // TODO: マルチトラックシーケンスのテスト
    //sequenceBitFlag.Reset();
    //sequenceBitFlag[1] = true;
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 0, m_MixSequenceTrack1);
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 1, m_MixSequenceTrack1);
    //sequenceBitFlag.Reset();
    //sequenceBitFlag[2] = true;
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 0, m_MixSequenceTrack2);
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 1, m_MixSequenceTrack2);
    //sequenceBitFlag.Reset();
    //sequenceBitFlag[3] = true;
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 0, m_MixSequenceTrack3);
    //sequenceSoundHandle.SetTrackMixVolume(sequenceBitFlag, 1, m_MixSequenceTrack3);

    NN_LOG("[MixVolume] (FL, FR, RL, RR, FC, LFE) = (%s%.2f, %s%.2f, %s%.2f, %s%.2f, %s%.2f, %s%.2f )\n"
        , (m_MixChannel == nn::atk::ChannelIndex_FrontLeft)   ? ">" : " " , m_MixVolume.channel[0]
        , (m_MixChannel == nn::atk::ChannelIndex_FrontRight)  ? ">" : " " , m_MixVolume.channel[1]
        , (m_MixChannel == nn::atk::ChannelIndex_RearLeft)    ? ">" : " " , m_MixVolume.channel[2]
        , (m_MixChannel == nn::atk::ChannelIndex_RearRight)   ? ">" : " " , m_MixVolume.channel[3]
        , (m_MixChannel == nn::atk::ChannelIndex_FrontCenter) ? ">" : " " , m_MixVolume.channel[4]
        , (m_MixChannel == nn::atk::ChannelIndex_Lfe)         ? ">" : " " , m_MixVolume.channel[5]);
}

void MixModeCheckModule::FadeAuxBusVolume() NN_NOEXCEPT
{
    if ( m_IsAuxBusFadedOut )
    {
        nn::atk::SoundSystem::SetAuxBusVolume( nn::atk::AuxBus_A, 1.0f, nn::TimeSpan::FromMilliSeconds( 1000 ) );
        NN_LOG( "Fade in AuxBus.\n" );
    }
    else
    {
        nn::atk::SoundSystem::SetAuxBusVolume( nn::atk::AuxBus_A, 0.0f, nn::TimeSpan::FromMilliSeconds( 1000 ) );
        NN_LOG( "Fade out AuxBus.\n" );
    }
    m_IsAuxBusFadedOut = !m_IsAuxBusFadedOut;
}
