﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Effect.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    enum FlagType
    {
        FlagType_EffectForFinalMix,    //< FinalMix 用 AuxEffect を有効化する評価用機能
        FlagType_AdditionalEffectBus,  //< エフェクトバスを追加する(BGM用AuxBusA,B)評価用機能
        FlagType_AdditionalSubMix,     //< サブミックスを追加する(SE用AuxBusA,B,C)評価用機能
        FlagType_Num
    };

    FlagElement g_LocalElements[] =
    {
        { "EffectForFinalMix", false },
        { "AdditionalEffectBus", false },
        { "AdditionalSubMix", false },
    };

    FlagList g_LocalFlagList(g_LocalElements, sizeof(g_LocalElements) / sizeof(g_LocalElements[0]));

    UserEffectAux g_UserEffectAux;
    UserEffectAux g_UserEffectAuxForFinalMix;
    UserEffectAux g_UserEffectAuxForAdditionalSubMix;
}

void EffectCheckModule::CommonObjectForEffect::OnPostStartSound() NN_NOEXCEPT
{
    GetSoundHandle().SetEffectSend(nn::atk::AuxBus_A, 1.0f);
    GetSoundHandle().SetMainSend(-1.0f);
}

void EffectCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    CommonObject::InitializeParam param;
    param.GetSoundSystemParam().rendererSampleRate = ConvertSampleRateToInt(static_cast<SampleRate>(m_SampleRate));

    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
    {
        // SoundSystem の初期化時にオプションを有効化する
        param.GetSoundSystemParam().enableAdditionalEffectBus = true;
    }

    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalSubMix))
    {
        param.GetSoundSystemParam().enableAdditionalSubMix = true;
    }

    m_CommonObject.Initialize(param);

    // パラメータの初期化
    m_SelectEffectType = EffectType_Reverb;
    for( int i = 0; i <= EffectType_Max; i++ )
    {
        m_EffectState[i] = EffectState_Disable;
    }
    m_AuxEffectBufferSize = 0;
    m_IsAuxBusFadedOut = false;

    // delay の設定
    m_Delay.SetDelayTimeMax(nn::TimeSpan::FromMilliSeconds(7000));
    m_Delay.SetDelayTime(nn::TimeSpan::FromMilliSeconds(300));
    m_Delay.SetDryGain(0.0f);
    m_Delay.SetInGain(1.0f);
    m_Delay.SetLowPassAmount(0.95f);
    m_Delay.SetSampleRate(ConvertSampleRateToEffectSampleRate(m_SampleRate));
    m_Delay.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);

    // reverb の設定
    m_Reverb.SetSampleRate(ConvertSampleRateToEffectSampleRate(m_SampleRate));
    m_Reverb.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);

    // aux の設定
    g_UserEffectAux.SetAudioFrameCount(5);
    g_UserEffectAux.SetChannelCount(2);
    g_UserEffectAux.SetSampleRate(ConvertSampleRateToInt(m_SampleRate));

    // FinalMix 用 AuxEffect の設定
    if (GetLocalFlagList().IsFlagEnabled(FlagType_EffectForFinalMix))
    {
        g_UserEffectAuxForFinalMix.SetAudioFrameCount(5);
        g_UserEffectAuxForFinalMix.SetChannelCount(2);
        g_UserEffectAuxForFinalMix.SetSampleRate(ConvertSampleRateToInt(m_SampleRate));
        g_UserEffectAuxForFinalMix.SetEnabled(true);

        m_AuxEffectBufferSizeForFinalMix = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&g_UserEffectAuxForFinalMix);
        m_AuxBufferForFinalMix = nns::atk::AllocateForMemoryPool(m_AuxEffectBufferSizeForFinalMix, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS(nn::atk::SoundSystem::AppendEffectToFinalMix(&g_UserEffectAuxForFinalMix, m_AuxBufferForFinalMix, m_AuxEffectBufferSizeForFinalMix));
    }

    // 追加エフェクト用 reverb の設定
    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
    {
        m_SubMixIndexForAdditionalEffect = 1;
        m_ReverbForAdditionalEffect.SetSampleRate(ConvertSampleRateToEffectSampleRate(m_SampleRate));
        m_ReverbForAdditionalEffect.SetChannelMode(nn::atk::EffectBase::ChannelMode_2Ch);
        m_ReverbForAdditionalEffect.SetEarlyMode(nn::atk::EffectReverb::EarlyMode_LargeRoom);
        m_ReverbForAdditionalEffect.SetLateMode(nn::atk::EffectReverb::LateMode_Room);
        m_ReverbForAdditionalEffect.SetPredelayTime(nn::TimeSpan::FromMilliSeconds(100));
        m_ReverbForAdditionalEffect.SetDecayTime(nn::TimeSpan::FromMilliSeconds(500));
        m_ReverbForAdditionalEffect.SetEnabled(true);

        auto delayBufferSize = m_ReverbForAdditionalEffect.GetRequiredMemSize();
        m_ReverbBufferForAdditionalEffect = nns::atk::AllocateForMemoryPool(delayBufferSize, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(m_ReverbBufferForAdditionalEffect);

        // SoundSystem::AppenEffect 時にサブミックス番号を指定する
        bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &m_ReverbForAdditionalEffect, m_ReverbBufferForAdditionalEffect, delayBufferSize, nn::atk::OutputDevice_Main, m_SubMixIndexForAdditionalEffect);
        NN_LOG( "Delay for additionalEffect  %d\n", isSuccess );

        m_AuxBusChannelVolumeForAdditionalEffect = 0.0f;
    }

    // 追加 SubMix 用 AuxEffect の設定
    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalSubMix))
    {
        g_UserEffectAuxForAdditionalSubMix.SetAudioFrameCount(5);
        g_UserEffectAuxForAdditionalSubMix.SetChannelCount(2);
        g_UserEffectAuxForAdditionalSubMix.SetSampleRate(ConvertSampleRateToInt(m_SampleRate));
        g_UserEffectAuxForAdditionalSubMix.SetEnabled(true);

        m_AuxEffectBufferSizeForAdditionalSubMix = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&g_UserEffectAuxForAdditionalSubMix);
        m_AuxBufferForAdditionalSubMix = nns::atk::AllocateForMemoryPool(m_AuxEffectBufferSizeForAdditionalSubMix, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS(nn::atk::SoundSystem::AppendEffectToAdditionalSubMix(&g_UserEffectAuxForAdditionalSubMix, m_AuxBufferForAdditionalSubMix, m_AuxEffectBufferSizeForAdditionalSubMix));
    }

    NN_LOG("Sample Rate %d\n", ConvertSampleRateToInt(m_SampleRate));
}

void EffectCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    ClearEffect();

    if (GetLocalFlagList().IsFlagEnabled(FlagType_EffectForFinalMix))
    {
        if( g_UserEffectAuxForFinalMix.IsEnabled() )
        {
            g_UserEffectAuxForFinalMix.SetEnabled(false);
            for ( ;; )
            {
                if (g_UserEffectAuxForFinalMix.IsRemovable())
                {
                    nn::atk::SoundSystem::ClearEffectFromFinalMix();
                    nns::atk::FreeForMemoryPool(m_AuxBufferForFinalMix);
                    break;
                }
            }
        }
    }

    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalSubMix))
    {
        if( g_UserEffectAuxForAdditionalSubMix.IsEnabled() )
        {
            g_UserEffectAuxForAdditionalSubMix.SetEnabled(false);
            for ( ;; )
            {
                if (g_UserEffectAuxForAdditionalSubMix.IsRemovable())
                {
                    nn::atk::SoundSystem::ClearEffectFromAdditionalSubMix();
                    nns::atk::FreeForMemoryPool(m_AuxBufferForAdditionalSubMix);
                    break;
                }
            }
        }
    }

    m_CommonObject.Finalize();
}

void EffectCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void EffectCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Left/Right]   Select Effect Type\n");
    NN_LOG("[Down]         Toggle On/Off Effect Type\n");
    NN_LOG("[Up]           Clear All Effect Type\n");

    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
    {
        NN_LOG("[ZR(Z) + Up]   Up   Rear Volume For AdditionalEffectBus\n");
        NN_LOG("[ZR(Z) + Down] Down Rear Volume For AdditionalEffectBus\n");
        NN_LOG("[R + Right]    StartSound SEQ  On Additional Submix (SEQ_TEST_PCM16)\n");
        NN_LOG("[R + Left]     StartSound STRM On Additional Submix (STRM_PIANO16_PCM16)\n");
    }
    NN_LOG("[R + Up]       Change Sample Rate\n");
    NN_LOG("[R + Down]     Fade AuxBus\n");
}

void EffectCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
    {
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            m_SampleRate = static_cast<SampleRate>(m_SampleRate + 1);
            if ( m_SampleRate > SampleRate_Max )
            {
                m_SampleRate = static_cast<SampleRate>(0);
            }

            OnFinalizeAtk();
            OnInitializeAtk();
            OnLoadData();
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            FadeAuxBusVolume();
        }

        if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
        {
            // 再生時に StartInfo でサブミックス番号を指定する
            if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
            {
                nn::atk::SoundStartable::StartInfo info;
                info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_SubMixIndex;
                info.subMixIndex = m_SubMixIndexForAdditionalEffect;

                CommonObject::StartParam startParam;
                startParam.pStartInfo = &info;

                m_CommonObject.PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16", startParam);
            }
            else if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
            {
                nn::atk::SoundStartable::StartInfo info;
                info.enableFlag |= nn::atk::SoundStartable::StartInfo::EnableFlagBit_SubMixIndex;
                info.subMixIndex = m_SubMixIndexForAdditionalEffect;

                CommonObject::StartParam startParam;
                startParam.pStartInfo = &info;

                m_CommonObject.PlayWithStartSound(SEQ_TEST_PCM16, "SEQ_TEST_PCM16", startParam);
            }
        }
    }
    else
    {
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
        {
            m_SelectEffectType -= 1;
            if ( m_SelectEffectType < 0 )
            {
                m_SelectEffectType = EffectType_Max;
            }
            ShowStatus();
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
        {
            m_SelectEffectType += 1;
            if ( m_SelectEffectType > EffectType_Max )
            {
                m_SelectEffectType = 0;
            }
            ShowStatus();
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
        {
            if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
            {
                if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
                {
                    m_AuxBusChannelVolumeForAdditionalEffect = std::max(m_AuxBusChannelVolumeForAdditionalEffect - 0.1f, 0.0f);
                    SetAllBusChannelVolumeForAdditionalEffect();
                }
            }
            else
            {
                if( m_EffectState[m_SelectEffectType] == EffectState_Disable )
                {
                    AppendEffect( m_SelectEffectType );
                    m_EffectState[m_SelectEffectType] = EffectState_Enable;

                }
                else if( m_EffectState[m_SelectEffectType] == EffectState_Enable )
                {
                    PrepareRemoveEffect( m_SelectEffectType );
                    m_EffectState[m_SelectEffectType] = EffectState_Removing;
                }

                ShowStatus();
            }
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
        {
            if (nns::atk::IsHold < ::nn::hid::DebugPadButton::ZR >())
            {
                if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
                {
                    m_AuxBusChannelVolumeForAdditionalEffect = std::min(m_AuxBusChannelVolumeForAdditionalEffect + 0.1f, 1.0f);
                    SetAllBusChannelVolumeForAdditionalEffect();
                }
            }
            else
            {
                ClearEffect();
                ShowStatus();
            }
        }
    }
} // NOLINT(impl/function_size)

void EffectCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();

    for( int i = 0; i <= EffectType_Max; i++ )
    {
        //  削除中のエフェクトがあるため状態を確認
        if( m_EffectState[i] == EffectState_Removing )
        {
            if( CheckEffectRemovable( i ) )
            {
                RemoveEffect( i );
                m_EffectState[i] = EffectState_Disable;
            }
        }
    }
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void EffectCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}
#endif

FlagList& EffectCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}

void EffectCheckModule::ShowStatus() NN_NOEXCEPT
{
    static const char* const EffectTypeName[EffectType_Max + 1] =
    {
        "Reverb", "Delay", "AuxEffect"
    };

    for( int i = 0; i <= EffectType_Max; i++ )
    {
        NN_LOG( "%c %s [%c]  ", m_SelectEffectType == i ? '>' : ' ',  EffectTypeName[i], m_EffectState[i] == EffectState_Enable ? '*' : ' ' );
    }
    NN_LOG( "\n" );
}

void EffectCheckModule::ClearEffect() NN_NOEXCEPT
{
    //  有効なエフェクトを無効化
    for( int i = 0; i <= EffectType_Max; i++ )
    {
        if( m_EffectState[i] == EffectState_Enable )
        {
            PrepareRemoveEffect( i );
            m_EffectState[i] = EffectState_Removing;
        }
    }

    // エフェクトが削除可能になるまで待つ
    for( int i = 0; i <= EffectType_Max; i++ )
    {
        if( m_EffectState[i] == EffectState_Removing )
        {
            if( CheckEffectRemovable( i ) )
            {
                RemoveEffect( i );
                m_EffectState[i] = EffectState_Disable;
            }
            else
            {
                //  まだ削除できないため、少し待ってもう一度確認
                nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 5 ) );
                i--;
            }
        }
    }

    if (GetLocalFlagList().IsFlagEnabled(FlagType_AdditionalEffectBus))
    {

        // 追加エフェクト用 reverb の解放処理
        if( m_ReverbForAdditionalEffect.IsEnabled() )
        {
            m_ReverbForAdditionalEffect.SetEnabled(false);
            for ( ;; )
            {
                if (m_ReverbForAdditionalEffect.IsRemovable())
                {
                    // SoundSystem::ClearEffect でサブミックス番号を指定する
                    nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A, nn::atk::OutputDevice_Main, m_SubMixIndexForAdditionalEffect);
                    nns::atk::FreeForMemoryPool(m_ReverbBufferForAdditionalEffect);
                    break;
                }
            }
        }
    }
}

void EffectCheckModule::AppendEffect( int effectType ) NN_NOEXCEPT
{
    switch(effectType)
    {
    case EffectType_Delay:
        {
            m_Delay.SetEnabled(true);
            size_t delayBufferSize = m_Delay.GetRequiredMemSize();
            m_DelayBuffer = nns::atk::AllocateForMemoryPool(delayBufferSize, nn::audio::BufferAlignSize);
            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &m_Delay, m_DelayBuffer, delayBufferSize);
            if( !isSuccess )
            {
                NN_LOG("Failed to append Delay Effect\n");
            }
        }
        break;
    case EffectType_Reverb:
        {
            m_Reverb.SetEnabled(true);
            size_t reverbBufferSize = m_Reverb.GetRequiredMemSize();
            m_ReverbBuffer = nns::atk::AllocateForMemoryPool(reverbBufferSize, nn::audio::BufferAlignSize);
            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &m_Reverb, m_ReverbBuffer, reverbBufferSize);
            if( !isSuccess )
            {
                NN_LOG("Failed to append Reverb Effect\n");
            }
        }
        break;
    case EffectType_Aux:
        {
            g_UserEffectAux.SetEnabled(true);
            size_t auxBufferSize = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&g_UserEffectAux);
            m_AuxBuffer = nns::atk::AllocateForMemoryPool(auxBufferSize, nn::audio::BufferAlignSize);
            bool isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_UserEffectAux, m_AuxBuffer, auxBufferSize);
            if( !isSuccess )
            {
                NN_LOG("Failed to append Aux Effect\n");
            }
            else
            {
                m_AuxEffectBufferSize = auxBufferSize;
            }
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

void EffectCheckModule::RemoveEffect( int effectType ) NN_NOEXCEPT
{
    switch ( effectType )
    {
    case EffectType_Delay:
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &m_Delay);
        nns::atk::FreeForMemoryPool(m_DelayBuffer);
        break;
    case EffectType_Reverb:
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &m_Reverb);
        nns::atk::FreeForMemoryPool(m_ReverbBuffer);
        break;
    case EffectType_Aux:
        nn::atk::SoundSystem::RemoveEffect(nn::atk::AuxBus_A, &g_UserEffectAux);
        nns::atk::FreeForMemoryPool(m_AuxBuffer);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

bool EffectCheckModule::PrepareRemoveEffect( int effectType ) NN_NOEXCEPT
{
    switch ( effectType )
    {
    case EffectType_Delay:
        m_Delay.SetEnabled(false);
        return true;
    case EffectType_Reverb:
        m_Reverb.SetEnabled(false);
        return true;
    case EffectType_Aux:
        g_UserEffectAux.SetEnabled(false);
        return true;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool EffectCheckModule::CheckEffectRemovable( int effectType ) NN_NOEXCEPT
{
    switch ( effectType )
    {
    case EffectType_Delay:
        return m_Delay.IsRemovable();
    case EffectType_Reverb:
        return m_Reverb.IsRemovable();
    case EffectType_Aux:
        return g_UserEffectAux.IsRemovable();
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EffectCheckModule::FadeAuxBusVolume() NN_NOEXCEPT
{
    if (m_IsAuxBusFadedOut)
    {
        nn::atk::SoundSystem::SetAuxBusVolume(nn::atk::AuxBus_A, 1.0f, nn::TimeSpan::FromMilliSeconds(1000));
        NN_LOG("Fade in AuxBus.\n");
    }
    else
    {
        nn::atk::SoundSystem::SetAuxBusVolume(nn::atk::AuxBus_A, 0.0f, nn::TimeSpan::FromMilliSeconds(1000));
        NN_LOG("Fade out AuxBus.\n");
    }
    m_IsAuxBusFadedOut = !m_IsAuxBusFadedOut;
}

int EffectCheckModule::ConvertSampleRateToInt( SampleRate sampleRate ) NN_NOEXCEPT
{
    switch (sampleRate)
    {
    case SampleRate_32000:
        return 32000;
    case SampleRate_48000:
        return 48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

nn::atk::EffectBase::SampleRate EffectCheckModule::ConvertSampleRateToEffectSampleRate( SampleRate sampleRate ) NN_NOEXCEPT
{
    switch (sampleRate)
    {
    case SampleRate_32000:
        return nn::atk::EffectBase::SampleRate_32000;
    case SampleRate_48000:
        return nn::atk::EffectBase::SampleRate_48000;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void EffectCheckModule::SetAllBusChannelVolumeForAdditionalEffect() const NN_NOEXCEPT
{
    // BGM 用サブミックスからファイナルミックスへ FL→RL, FR→RR へのミックスを設定する
    nn::atk::SoundSystem::SetAllBusChannelVolumeForAdditionalEffect(
        m_AuxBusChannelVolumeForAdditionalEffect,
        nn::atk::ChannelIndex_FrontLeft,
        nn::atk::ChannelIndex_RearLeft);
    nn::atk::SoundSystem::SetAllBusChannelVolumeForAdditionalEffect(
        m_AuxBusChannelVolumeForAdditionalEffect,
        nn::atk::ChannelIndex_FrontRight,
        nn::atk::ChannelIndex_RearRight);
    NN_LOG("AllBusChannelVolume(Front->Rear):%f\n", m_AuxBusChannelVolumeForAdditionalEffect);
}
