﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AtkSandboxCommon.h"

#include "FlagList.h"
#include "GfxCode/DebugViewer.h"

namespace
{
    const char* CommonSoundArchivePath = "content:/test.bfsar";
    const int SoundHeapSize = 4 * 1024 * 1024;
    const int StreamCacheBufferSize = 16 * 1024 * 5;

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
    // モジュール名表示のデフォルトの文字列描画位置
    const int DefaultPrintModuleNamePosX = 800;
    const int DefaultPrintModuleNamePosY = 0;
    // 操作説明のデフォルトの文字列描画位置
    const int DefaultPrintUsagePosX = 800;
    const int DefaultPrintUsagePosY = 20;
    // サウンド共通情報のデフォルトの文字列描画位置
    const int DefaultPrintCommonSoundInfoPosX = 800;
    const int DefaultPrintCommonSoundInfoPosY = 300;
    // サウンド種別情報のデフォルトの文字列描画位置
    const int DefaultPrintSoundInfoPosX = 800;
    const int DefaultPrintSoundInfoPosY = 440;

    void DrawGeneralUsage() NN_NOEXCEPT
    {
        DebugPrint("General_Usage", DefaultPrintUsagePosX, DefaultPrintUsagePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
            "\n"
            "  [A]        :StartSound SEQ  (SEQ_TEST_PCM16)\n"
            "  [R + A]    :StartSound SEQ  (SEQ_TEST_ADPCM)\n"
            "  [X]        :StartSound WSD  (WSD_YOSHI_ADPCM)\n"
            "  [R + X]    :StartSound WSD  (WSD_YOSHI_PCM16)\n"
            "  [Y]        :StartSound STRM (STRM_PIANO16_ADPCM)\n"
            "  [R + Y]    :StartSound STRM (STRM_PIANO16_PCM16)\n"
            "  [B]        :Stop Sound\n"
            "  [R + B]    :Pause Sound\n"
            "  [R + ZR(Z)]:Suspend/Resume AudioRenderer"
        );
    }
#endif

    nn::audio::MemoryPoolType g_MemoryPool;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    NN_OS_ALIGNAS_THREAD_STACK uint8_t g_profilerBuffer[nn::profiler::MinimumBufferSize];
    static void ProfilerSoundFrameUserCallback( uintptr_t /* arg */ )
    {
        // SoundThread が動作する周期に HeartBeat を設定
        nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats_User1);
    }
#endif

    void DumpSoundArchivePlayerInfo(nn::atk::SoundArchive* pSoundArchive) NN_NOEXCEPT
    {
        nn::atk::SoundArchive::SoundArchivePlayerInfo archivePlayerInfo;
        pSoundArchive->ReadSoundArchivePlayerInfo(&archivePlayerInfo);
        NN_LOG("[atk] SoundArchivePlayerInfo\n");
        NN_LOG("      - waveSoundCount:     %2d\n", archivePlayerInfo.waveSoundCount);
        NN_LOG("      - waveTrackCount:     %2d\n", archivePlayerInfo.waveTrackCount);
        NN_LOG("      - sequenceSoundCount: %2d\n", archivePlayerInfo.sequenceSoundCount);
        NN_LOG("      - sequenceTrackCount: %2d\n", archivePlayerInfo.sequenceTrackCount);
        NN_LOG("      - streamSoundCount:   %2d\n", archivePlayerInfo.streamSoundCount);
        NN_LOG("      - streamChannelCount: %2d\n", archivePlayerInfo.streamChannelCount);
        NN_LOG("      - streamTrackCount:   %2d\n", archivePlayerInfo.streamTrackCount);
    }
}


CommonObject::CommonObject() NN_NOEXCEPT
: m_pSoundArchive(nullptr)
, m_IsPause(false)
, m_IsRendererSuspended(false)
#if defined(NN_ATK_CONFIG_ENABLE_DEV)
, m_IsConnected(false)
#endif
{

}

CommonObject::~CommonObject() NN_NOEXCEPT
{

}

void CommonObject::Initialize() NN_NOEXCEPT
{
    CommonObject::InitializeParam param;
    Initialize(param);
}

void CommonObject::Initialize(const CommonObject::InitializeParam& param) NN_NOEXCEPT
{
    bool isSuccess = true;

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableCpuProfiler) )
    {
        nn::Result result = nn::profiler::Initialize(g_profilerBuffer, nn::profiler::MinimumBufferSize);
        if (result.IsFailure())
        {
            NN_LOG("[atk] Failed to enable CPU Profiler:%08x\n", result.GetInnerValueForDebug());
        }
        else
        {
            NN_LOG("[atk] CPU Profiler enabled\n");
        }
    }
#endif

    // SoundSystemParam の設定
    nn::atk::SoundSystem::SoundSystemParam soundSystemParam = param.GetSoundSystemParam();
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
    {
        soundSystemParam.enableMemoryPoolManagement = false;
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) ||
         GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSpy) )
    {
        soundSystemParam.enableCircularBufferSink = true;
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) )
    {
        soundSystemParam.enableCircularBufferSinkBufferManagement = false;
    }

    // SoundSystem 用メモリの確保
    const std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( soundSystemParam );
    m_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );

    std::size_t tempMemSizeForSoundSystemMemoryPool = 0;
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
    {
        tempMemSizeForSoundSystemMemoryPool = nn::atk::SoundSystem::GetRequiredMemSizeForMemoryPool( soundSystemParam );
        m_pMemoryForSoundSystemMemoryPool = nns::atk::AllocateForMemoryPool( tempMemSizeForSoundSystemMemoryPool, nn::audio::BufferAlignSize );
    }
    const std::size_t memSizeForSoundSystemMemoryPool = tempMemSizeForSoundSystemMemoryPool;

    std::size_t tempMemSizeForSoundSystemCircularBufferSink = 0;
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) )
    {
        tempMemSizeForSoundSystemCircularBufferSink = nn::atk::SoundSystem::GetRequiredMemSizeForCircularBufferSink( soundSystemParam );
        m_pMemoryForSoundSystemCircularBufferSink = nns::atk::Allocate( tempMemSizeForSoundSystemCircularBufferSink );
    }
    const std::size_t memSizeForSoundSystemCircularBufferSink = tempMemSizeForSoundSystemCircularBufferSink;

    // SoundSystem の初期化（３パターン）
    {
        nn::Result result;
        if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) )
        {

            nn::atk::SoundSystem::InitializeBufferSet bufferSet;

            bufferSet.workMem = reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystem);
            bufferSet.workMemSize = memSizeForSoundSystem;
            if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
            {
                bufferSet.memoryPoolMem = reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystemMemoryPool);
                bufferSet.memoryPoolMemSize = memSizeForSoundSystemMemoryPool;
            }
            else
            {
                bufferSet.memoryPoolMem = NULL;
                bufferSet.memoryPoolMemSize = 0u;
            }
            bufferSet.circularBufferSinkMem = reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystemCircularBufferSink);
            bufferSet.circularBufferSinkMemSize = memSizeForSoundSystemCircularBufferSink;

            isSuccess = nn::atk::SoundSystem::Initialize(
                &result,
                soundSystemParam,
                bufferSet
            );
        }
        else
        {
            if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
            {
                isSuccess = nn::atk::SoundSystem::Initialize(
                    &result,
                    soundSystemParam,
                    reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystem),
                    memSizeForSoundSystem,
                    reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystemMemoryPool),
                    memSizeForSoundSystemMemoryPool
                );
            }
            else
            {
                isSuccess = nn::atk::SoundSystem::Initialize(
                    &result,
                    soundSystemParam,
                    reinterpret_cast<uintptr_t>(m_pMemoryForSoundSystem),
                    memSizeForSoundSystem
                );
            }
        }

        if ( !isSuccess )
        {
            if ( nn::audio::ResultNoAudioDeviceFound::Includes(result) )
            {
                NN_LOG("cannot find audio devices.\n");
            }
        }
    }
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // Spy 対応モジュールの初期化
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSpy) )
    {
        size_t spyToolProfilerBufferSize = m_SpyToolProfiler.GetRequiredMemorySize();
        m_pMemoryForSpyToolProfiler = nns::atk::Allocate( spyToolProfilerBufferSize );
        m_SpyToolProfiler.Initialize( m_pMemoryForSpyToolProfiler, spyToolProfilerBufferSize );
    }

    // atk プロファイラの初期化
    size_t atkProfilerBufferSize = m_AtkProfiler.GetRequiredMemorySize(TaskThreadProfileCount, SoundThreadUpdateProfileCount);
    m_pMemoryForAtkProfiler = nns::atk::Allocate(atkProfilerBufferSize);
    m_AtkProfiler.Initialize(m_pMemoryForAtkProfiler, atkProfilerBufferSize, TaskThreadProfileCount, SoundThreadUpdateProfileCount);

    // SoundHeap の初期化
    m_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = m_SoundHeap.Create( m_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    //  要求したメモリを書き出す
    size_t requiredMemSize = 0u;
    nn::atk::SoundSystem::DumpMemory();
    NN_LOG( "\n" );
    NN_LOG( "    required:\n" );
    NN_LOG( "    work buffer                    %8zu bytes\n", memSizeForSoundSystem );
    requiredMemSize += memSizeForSoundSystem;
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
    {
        NN_LOG( "    memory pool                    %8zu bytes\n", memSizeForSoundSystemMemoryPool );
        requiredMemSize += memSizeForSoundSystemMemoryPool;
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) )
    {
        NN_LOG( "    circularBufferSink             %8zu bytes\n", memSizeForSoundSystemCircularBufferSink );
        requiredMemSize += memSizeForSoundSystemCircularBufferSink;
    }
    NN_LOG( "    total:                         %8zu bytes\n", requiredMemSize );

    // 2016/12/22 現在要求メモリと消費メモリで以下のずれがあります。(8256 or 15168)
    // = nn::atk::detail::fnd::Thread::StackAlignment (4096)
    // + nn::os::MemoryPageSize (4096)
    // + nn::audio::BufferAlignSize (64) (enableProfiler が true の場合、AtkSandbox ではデフォルト true）
    // if (enableMemoryPoolManagement == true)
    //     + nn::util::align_up( memoryPoolMemSize, nn::audio::MemoryPoolType::SizeGranularity ); (2816増)
    //     + nn::audio::MemoryPoolType::AddressAlignment (4096)

    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseMemorySoundArchive) )
    {
        // メモリ上にアーカイブをロード
        nn::fs::FileHandle fileHandle;
        nn::Result result = nn::fs::OpenFile(&fileHandle, CommonSoundArchivePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        int64_t fileSize = 0;
        result = nn::fs::GetFileSize(&fileSize, fileHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        m_pMemoryForMemorySoundArchive = nns::atk::AllocateForMemoryPool( static_cast<size_t>(fileSize), nn::atk::SoundDataManager::BufferAlignSize  );
        size_t readSize = 0;
        result = nn::fs::ReadFile(&readSize, fileHandle, 0, m_pMemoryForMemorySoundArchive, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        nn::fs::CloseFile(fileHandle);

        m_MemorySoundArchive.Initialize(m_pMemoryForMemorySoundArchive);
        m_pSoundArchive = &m_MemorySoundArchive;
    }
    else
    {
        // SoundArchive の初期化
        isSuccess = m_FsSoundArchive.Open(CommonSoundArchivePath);
        NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", CommonSoundArchivePath );

        // SoundArchive のパラメータ情報をメモリにロード
        std::size_t infoBlockSize = m_FsSoundArchive.GetHeaderSize();
        m_pMemoryForInfoBlock = nns::atk::Allocate( infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
        isSuccess = m_FsSoundArchive.LoadHeader( m_pMemoryForInfoBlock, infoBlockSize );
        NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

        m_pSoundArchive = &m_FsSoundArchive;
    }

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = m_SoundDataManager.GetRequiredMemSize( m_pSoundArchive );
    m_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = m_SoundDataManager.Initialize(
        m_pSoundArchive,
        m_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    nn::atk::SoundArchivePlayer::InitializeParam initializeParam;
    initializeParam.pSoundArchive = m_pSoundArchive;
    initializeParam.pSoundDataManager = &m_SoundDataManager;

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = m_SoundArchivePlayer.GetRequiredStreamBufferSize(m_pSoundArchive);
    m_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);
    initializeParam.pStreamBuffer = m_pMemoryForStreamBuffer;
    initializeParam.streamBufferSize = memSizeForStreamBuffer;

    std::size_t memSizeForSoundArchivePlayer = 0;
    if (GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSoundArchivePlayerAppliedStreamInstanceBuffer))
    {
        initializeParam.enablePreparingStreamInstanceBufferFromSetupBuffer = false;

        std::size_t memSizeForStreamInstance = m_SoundArchivePlayer.GetRequiredStreamInstanceSize( m_pSoundArchive );
        m_pMemoryForStreamInstanceBuffer = nns::atk::AllocateForMemoryPool( memSizeForStreamInstance, nn::audio::BufferAlignSize );
        initializeParam.pStreamInstanceBuffer = m_pMemoryForStreamInstanceBuffer;
        initializeParam.streamInstanceBufferSize = memSizeForStreamInstance;

        memSizeForSoundArchivePlayer = m_SoundArchivePlayer.GetRequiredMemSize( initializeParam );
    }
    else
    {
        memSizeForSoundArchivePlayer = m_SoundArchivePlayer.GetRequiredMemSize( m_pSoundArchive );
    }

    m_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    initializeParam.pSetupBuffer = m_pMemoryForSoundArchivePlayer;
    initializeParam.setupBufferSize = memSizeForSoundArchivePlayer;

    if (GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseStreamCache))
    {
        std::size_t memSizeForStreamCacheBuffer = m_SoundArchivePlayer.GetRequiredStreamCacheSize( m_pSoundArchive, StreamCacheBufferSize );
        m_pMemoryForStreamCacheBuffer = nns::atk::Allocate( memSizeForStreamCacheBuffer, nn::audio::BufferAlignSize );
        initializeParam.pStreamCacheBuffer = m_pMemoryForStreamCacheBuffer;
        initializeParam.streamCacheSize = memSizeForStreamCacheBuffer;
    }

    isSuccess = m_SoundArchivePlayer.Initialize( initializeParam );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    // SoundArchive のラベル文字列情報をメモリにロード
    if ( !GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseMemorySoundArchive) )
    {
        std::size_t memSizeForLabelString = m_FsSoundArchive.GetLabelStringDataSize();
        m_pMemoryForLabelData = nns::atk::Allocate( memSizeForLabelString, nn::atk::FsSoundArchive::BufferAlignSize );
        m_FsSoundArchive.LoadLabelStringData( m_pMemoryForLabelData, memSizeForLabelString );
    }

    // インゲーム編集機能の初期化
    nn::atk::viewer::SoundEdit::Option option;
    std::size_t memSizeForSoundEdit = m_SoundEdit.GetRequiredMemorySize( option );
    m_pMemoryForSoundEdit = nns::atk::Allocate( memSizeForSoundEdit );

    nn::atk::viewer::SoundEdit::InitializeArg args;
    args.buffer             = m_pMemoryForSoundEdit;
    args.bufferSize         = memSizeForSoundEdit;
    args.soundArchive       = m_pSoundArchive;
    args.soundArchivePlayer = &m_SoundArchivePlayer;
    m_SoundEdit.Initialize( args, option );

    // インゲーム編集を開始
    m_IsConnected = false;
    m_SoundEdit.Start();
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableCpuProfiler) )
    {
        nn::atk::SoundSystem::SetSoundFrameUserCallback(&ProfilerSoundFrameUserCallback, 0);
    }
#endif

    // サウンドアーカイブプレイヤーの情報をダンプ
    DumpSoundArchivePlayerInfo(m_pSoundArchive);
} // NOLINT(impl/function_size)

void CommonObject::Finalize() NN_NOEXCEPT
{
#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    // インゲーム編集機能の終了
    m_SoundEdit.Stop();
    m_SoundEdit.Finalize();
    nns::atk::Free( m_pMemoryForSoundEdit );

    if ( !GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseMemorySoundArchive) )
    {
        nns::atk::Free( m_pMemoryForLabelData );
    }
#endif // NN_ATK_CONFIG_ENABLE_DEV

    m_AtkProfiler.Finalize();
    nns::atk::Free(m_pMemoryForAtkProfiler);
    m_pMemoryForAtkProfiler = nullptr;

    // Spy 対応モジュールの終了処理
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSpy) )
    {
        m_SpyToolProfiler.Finalize();
        nns::atk::Free( m_pMemoryForSpyToolProfiler );
        m_pMemoryForSpyToolProfiler = nullptr;
    }

    m_SoundArchivePlayer.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool( &g_MemoryPool );

    m_SoundDataManager.Finalize();
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseMemorySoundArchive) )
    {
        m_MemorySoundArchive.Finalize();
    }
    else
    {
        m_FsSoundArchive.Close();
    }
    m_pSoundArchive = nullptr;
    m_SoundHeap.Destroy();

    OnPreFinalizeSoundSystem();
    nn::atk::SoundSystem::Finalize();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableCpuProfiler) )
    {
        nn::profiler::Finalize();
    }
#endif

    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationCircularBufferSinkBuffer) )
    {
        nns::atk::Free( m_pMemoryForSoundSystemCircularBufferSink );
    }
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_SoundSystemUsingApplicationMemoryPool) )
    {
        nns::atk::FreeForMemoryPool( m_pMemoryForSoundSystemMemoryPool );
    }
    nns::atk::FreeForMemoryPool( m_pMemoryForStreamBuffer );
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSoundArchivePlayerAppliedStreamInstanceBuffer) )
    {
        nns::atk::FreeForMemoryPool( m_pMemoryForStreamInstanceBuffer );
    }
    if (GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseStreamCache))
    {
        nns::atk::Free( m_pMemoryForStreamCacheBuffer );
    }

    nns::atk::Free( m_pMemoryForSoundArchivePlayer );
    nns::atk::Free( m_pMemoryForSoundDataManager );
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_UseMemorySoundArchive) )
    {
        nns::atk::FreeForMemoryPool(m_pMemoryForMemorySoundArchive);
    }
    else
    {
        nns::atk::Free( m_pMemoryForInfoBlock );
    }
    nns::atk::Free( m_pMemoryForSoundHeap );
    nns::atk::Free( m_pMemoryForSoundSystem );
}

void CommonObject::LoadData() NN_NOEXCEPT
{
    bool isSuccess = true;

    isSuccess = m_SoundDataManager.LoadData( SEQ_TEST_PCM16, &m_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_TEST_PCM16) failed." );

    isSuccess = m_SoundDataManager.LoadData( SEQ_TEST_ADPCM, &m_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_TEST_ADPCM) failed." );

    isSuccess = m_SoundDataManager.LoadData( SEQ_MARIOKART, &m_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = m_SoundDataManager.LoadData( WSD_YOSHI_PCM16, &m_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(WSD_YOSHI_PCM16) failed." );

    isSuccess = m_SoundDataManager.LoadData( WSD_YOSHI_ADPCM, &m_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(WSD_YOSHI_ADPCM) failed." );
}

void CommonObject::Update() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    // メインのループが動作する周期に HeartBeat を設定
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableCpuProfiler) )
    {
        nn::profiler::RecordHeartbeat(nn::profiler::Heartbeats::Heartbeats_Main);
    }
#endif

    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_EnableSpy) )
    {
        m_SpyToolProfiler.Update( m_SoundArchivePlayer, m_SoundHeap );
    }

    if ( m_SoundHandle.IsAttachedSound() && m_SoundHandle.GetId() == STRM_PIANO16_PCM16 )
    {
        nn::atk::StreamSoundHandle streamSoundHandle(&m_SoundHandle);

        int64_t playSamplePosition = streamSoundHandle.GetPlaySamplePosition();
        NN_LOG("[STRM] playSamplePosition = %lld\n", playSamplePosition);
    }
    else if ( m_SoundHandle.IsAttachedSound() && m_SoundHandle.GetId() == WSD_YOSHI_PCM16 )
    {
        nn::atk::WaveSoundHandle waveSoundHandle(&m_SoundHandle);

        int64_t playSamplePosition = waveSoundHandle.GetPlaySamplePosition();
        NN_LOG("[WSD ] playSamplePosition = %lld\n", playSamplePosition);
    }

    m_SoundArchivePlayer.Update();

#if defined(NN_ATK_ENABLE_VOICE_COMMAND_SAMPLE)
    nn::atk::SoundSystem::VoiceCommandProcess(6);
    if ( GetGlobalFlagList().IsFlagEnabled(GlobalFlagType_DisableSoundThread) )
    {
        nn::atk::SoundSystem::VoiceCommandUpdate();
    }
#endif

#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    m_SoundEdit.Update();

    if ( m_IsConnected != m_SoundEdit.IsConnected() )
    {
        if ( m_IsConnected )
        {
            m_IsConnected = false;
            NN_LOG("disconnected.\n");
        }
        else
        {
            m_IsConnected = true;
            NN_LOG("connected.\n");
        }
    }
#endif // NN_ATK_CONFIG_ENABLE_DEV
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void CommonObject::UpdateDraw(const char* moduleName) NN_NOEXCEPT
{
    // モジュール名の表示
    DebugPrint("ModuleName", DefaultPrintModuleNamePosX, DefaultPrintModuleNamePosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, moduleName);

    // 操作説明の表示
    DrawGeneralUsage();

    // 再生中のサウンド情報の表示
    DrawSoundCommonInfo();
    DrawSoundHandleInfo();
}
#endif

void CommonObject::PrintUsage() NN_NOEXCEPT
{
    NN_LOG("[A]            StartSound SEQ  (SEQ_TEST_PCM16)\n");
    NN_LOG("[R + A]        StartSound SEQ  (SEQ_TEST_ADPCM)\n");
    NN_LOG("[X]            StartSound WSD  (WSD_YOSHI_ADPCM)\n");
    NN_LOG("[R + X]        StartSound WSD  (WSD_YOSHI_PCM16)\n");
    NN_LOG("[Y]            StartSound STRM (STRM_PIANO16_ADPCM)\n");
    NN_LOG("[R + Y]        StartSound STRM (STRM_PIANO16_PCM16)\n");
    NN_LOG("[B]            Stop Sound\n");
    NN_LOG("[R + B]        Pause Sound\n");
    NN_LOG("[R + ZR(Z)]    Suspend/Resume AudioRenderer\n");
    m_AtkProfiler.PrintUsage();
}

int CommonObject::UpdateInput() NN_NOEXCEPT
{
    int processCount = m_AtkProfiler.UpdateInput();

    // StartSound / StopSound / Profile
    if (processCount == 0)
    {
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                PlayWithStartSound(SEQ_TEST_ADPCM, "SEQ_TEST_ADPCM");
                processCount++;
            }
            else
            {
                PlayWithStartSound(SEQ_TEST_PCM16, "SEQ_TEST_PCM16");
                processCount++;
            }
        }
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                PlayWithStartSound(WSD_YOSHI_ADPCM, "WSD_YOSHI_ADPCM");
                processCount++;
            }
            else
            {
                PlayWithStartSound(WSD_YOSHI_PCM16, "WSD_YOSHI_PCM16");
                processCount++;
            }
        }
        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                PlayWithStartSound(STRM_PIANO16_PCM16, "STRM_PIANO16_PCM16");
                processCount++;
            }
            else
            {
                PlayWithStartSound(STRM_PIANO16_ADPCM, "STRM_PIANO16_ADPCM");
                processCount++;
            }
        }

        if (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B>())
        {
            if (nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
            {
                m_IsPause = !m_IsPause;
                m_SoundHandle.Pause(m_IsPause, 0);
                processCount++;
            }
            else
            {
                m_SoundHandle.Stop(0);
                processCount++;
            }
        }

        if ((nns::atk::IsTrigger< ::nn::hid::DebugPadButton::ZR >() && nns::atk::IsHold< ::nn::hid::DebugPadButton::R >())
        ||  (nns::atk::IsTrigger< ::nn::hid::DebugPadButton::R >() && nns::atk::IsHold< ::nn::hid::DebugPadButton::ZR >()))
        {
            if (m_IsRendererSuspended)
            {
                NN_LOG("Wait RendererEvent %d msec.\n", nn::atk::detail::driver::SoundThread::GetInstance().GetRendererEventWaitTimeMilliSeconds());
                nn::atk::SoundSystem::ResumeAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
            }
            else
            {
                nn::atk::SoundSystem::SuspendAudioRenderer(nn::TimeSpan::FromMilliSeconds(0));
            }
            m_IsRendererSuspended = !m_IsRendererSuspended;
        }
    }

    m_AtkProfiler.Show(m_SoundHandle);
    return processCount;
}

void CommonObject::PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName) NN_NOEXCEPT
{
    PlayWithStartSound(soundId, debugLabelName, StartParam());
}

void CommonObject::PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName, const StartParam& startParam) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pSoundArchive);

    char filePath[nn::fs::EntryNameLengthMax + 1];
    if ( m_pSoundArchive->ReadStreamSoundFilePath( filePath, nn::fs::EntryNameLengthMax + 1, soundId ) )
    {
        NN_LOG( "StreamFilePath = %s\n", filePath );
    }

    if( startParam.isSoundStopEnabled )
    {
        m_SoundHandle.Stop( startParam.stopFadeFrames );
    }

    bool result = m_SoundArchivePlayer.StartSound( &m_SoundHandle, soundId, startParam.pStartInfo ).IsSuccess();
    NN_LOG("StartSound(%s)%s ... (%d)\n", debugLabelName, CheckCacheState(soundId) ? " with edit" : "", result);

    OnPostStartSound();
}

bool CommonObject::CheckCacheState(nn::atk::SoundArchive::ItemId soundId) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(m_pSoundArchive);

    bool isEdited = false;
#if defined(NN_ATK_CONFIG_ENABLE_DEV)
    isEdited = m_SoundEdit.GetSoundArchiveEditor().GetItemCacheState(
        m_pSoundArchive->GetItemLabel(soundId)) == nn::atk::viewer::CacheState_Cached;
#else
    NN_UNUSED(soundId);
#endif

    return isEdited;
}

void CommonObject::OnPostStartSound() NN_NOEXCEPT
{
}

void CommonObject::OnPreFinalizeSoundSystem() NN_NOEXCEPT
{
}

#if defined( NN_ATK_ENABLE_GFX_VIEWING )
void CommonObject::DrawSoundCommonInfo() NN_NOEXCEPT
{
    // サウンド共通の情報
    if ( m_SoundHandle.IsAttachedSound() )
    {
        DebugPrint("CommonSoundInfo", DefaultPrintCommonSoundInfoPosX, DefaultPrintCommonSoundInfoPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
            "\n"
            "  Name      :%s\n"
            "  ID        :0x%08x\n"
            "  FrameCount:  %8u%s"
            , GetSoundArchive().GetItemLabel(m_SoundHandle.GetId())
            , m_SoundHandle.GetId()
            , m_SoundHandle.GetPlayFrameCount()
            , m_SoundHandle.IsPause() ? " [Pause]" : ""
        );
    }
}
void CommonObject::DrawSoundHandleInfo() NN_NOEXCEPT
{
    // サウンド種別の情報
    if ( m_SoundHandle.IsAttachedSound() )
    {
        const int64_t InvalidSamplePosition = -1;
        nn::atk::WaveSoundHandle waveSoundHandle(&m_SoundHandle);
        nn::atk::SequenceSoundHandle sequenceSoundHandle(&m_SoundHandle);
        nn::atk::StreamSoundHandle streamSoundHandle(&m_SoundHandle);
        int64_t playSamplePostion = InvalidSamplePosition;
        int64_t loopStartPostion = InvalidSamplePosition;
        int64_t loopEndPostion = InvalidSamplePosition;
        bool loopFlag = false;
        if ( waveSoundHandle.IsAttachedSound() )
        {
            playSamplePostion = waveSoundHandle.GetPlaySamplePosition();
            DebugPrint("SoundHandleInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                "[WSD]\n"
                "  ChannelCount :%8u\n"
                "  PlaySamplePos:%8d"
                , waveSoundHandle.GetChannelCount()
                , waveSoundHandle.GetPlaySamplePosition()
            );
            DebugViewer::GetInstance().ErasePrimitive("Filled");
            DebugViewer::GetInstance().ErasePrimitive("Loop");
            DebugViewer::GetInstance().ErasePrimitive("Played");
            nn::atk::WaveSoundDataInfo dataInfo;
            if ( waveSoundHandle.ReadWaveSoundDataInfo(&dataInfo) )
            {
                loopStartPostion = dataInfo.loopStart;
                loopEndPostion = dataInfo.loopEnd;
                loopFlag = dataInfo.loopFlag;
                DebugPrint("SoundDataInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY + 50, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                    "\n"
                    "  SampleRate   :%8d\n"
                    "  Loop         :%s\n"
                    "  LoopStart    :%8d\n"
                    "  LoopEnd      :%8d\n"
                    "  CompLoopStart:%8d\n"
                    "  CompLoopEnd  :%8d"
                    , dataInfo.sampleRate
                    , dataInfo.loopFlag ? "      ON" : "     OFF"
                    , dataInfo.loopStart
                    , dataInfo.loopEnd
                    , dataInfo.compatibleLoopStart
                    , dataInfo.compatibleLoopEnd
                );
            }
            else
            {
                DebugPrint("SoundDataInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY + 50, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "");
            }
        }
        else if ( sequenceSoundHandle.IsAttachedSound() )
        {
            DebugPrint("SoundHandleInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                "[SEQ]\n"
                "  Tick :%8u"
                , sequenceSoundHandle.GetTick()
            );
            DebugPrint("SoundDataInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY + 50, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "");
            DebugViewer::GetInstance().ErasePrimitive("Filled");
            DebugViewer::GetInstance().ErasePrimitive("Loop");
            DebugViewer::GetInstance().ErasePrimitive("Played");
        }
        else if ( streamSoundHandle.IsAttachedSound() )
        {
            playSamplePostion = streamSoundHandle.GetPlaySamplePosition();
            DebugPrint("SoundHandleInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                "[STRM]\n"
                "  PlayPos     :%8d\n"
                "  LoopCount   :%8d\n"
                "  Block(Play) :%8d\n"
                "  Block(Wait) :%8d\n"
                "  Block(Done) :%8d\n"
                "  Block(Free) :%8d\n"
                "  TotalBlock  :%8d\n"
                "  FilledBuf   :%8.2f"
                , streamSoundHandle.GetPlaySamplePosition()
                , streamSoundHandle.GetPlayLoopCount()
                , streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Play)
                , streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Wait)
                , streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Done)
                , streamSoundHandle.GetBufferBlockCount(nn::atk::WaveBuffer::Status_Free)
                , streamSoundHandle.GetTotalBufferBlockCount()
                , streamSoundHandle.GetFilledBufferPercentage()
            );

            const int FilledBufferBarOffsetY = -25;
            const int FilledBufferBarHeight = 8;
            int filledBufferBarOffsetX = static_cast<int>(DebugViewer::DefaultDrawBarWidth * streamSoundHandle.GetFilledBufferPercentage() / 100.f);
            DebugDrawQuad(
                "Filled",
                DefaultPrintSoundInfoPosX,
                DefaultPrintSoundInfoPosY + FilledBufferBarOffsetY,
                DefaultPrintSoundInfoPosX + filledBufferBarOffsetX,
                DefaultPrintSoundInfoPosY + FilledBufferBarOffsetY + FilledBufferBarHeight,
                GetDebugColor( 0, 0, 230 )
            );

            nn::atk::StreamSoundDataInfo dataInfo;
            if ( streamSoundHandle.IsPrepared() && streamSoundHandle.ReadStreamSoundDataInfo(&dataInfo) )
            {
                loopStartPostion = dataInfo.loopStart;
                loopEndPostion = dataInfo.loopEnd;
                loopFlag = dataInfo.loopFlag;
                DebugPrint("SoundDataInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY + 170, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY,
                    "\n"
                    "  SampleRate   :%8d\n"
                    "  Loop         :%s\n"
                    "  LoopStart    :%8d\n"
                    "  LoopEnd      :%8d\n"
                    "  CompLoopStart:%8d\n"
                    "  CompLoopEnd  :%8d"
                    , dataInfo.sampleRate
                    , dataInfo.loopFlag ? "      ON" : "     OFF"
                    , dataInfo.loopStart
                    , dataInfo.loopEnd
                    , dataInfo.compatibleLoopStart
                    , dataInfo.compatibleLoopEnd
                );
            }
            else
            {
                DebugPrint("SoundDataInfo", DefaultPrintSoundInfoPosX, DefaultPrintSoundInfoPosY + 50, DebugViewer::DefaultPrintScaleX, DebugViewer::DefaultPrintScaleY, "");
                DebugViewer::GetInstance().ErasePrimitive("Loop");
                DebugViewer::GetInstance().ErasePrimitive("Played");
            }
        }

        if (playSamplePostion != InvalidSamplePosition && loopEndPostion != InvalidSamplePosition)
        {
            const int PlaySamplePositionBarOffsetY = -15;
            const int PlaySamplePositionBarHeight = 8;
            int playSamplePositionBarOffsetX = static_cast<int>(static_cast<float>(DebugViewer::DefaultDrawBarWidth) * playSamplePostion / loopEndPostion);
            DebugDrawQuad(
                "Played",
                DefaultPrintSoundInfoPosX,
                DefaultPrintSoundInfoPosY + PlaySamplePositionBarOffsetY,
                DefaultPrintSoundInfoPosX + playSamplePositionBarOffsetX,
                DefaultPrintSoundInfoPosY + PlaySamplePositionBarOffsetY + PlaySamplePositionBarHeight,
                GetDebugColor( 0, 80, 0 )
            );
            if (loopStartPostion != InvalidSamplePosition && loopFlag)
            {
                const int LoopSamplePositionBarOffsetY = PlaySamplePositionBarOffsetY + PlaySamplePositionBarHeight;
                const int LoopSamplePositionBarHeight = 4;
                int loopSamplePositionBarOffsetX = static_cast<int>(static_cast<float>(DebugViewer::DefaultDrawBarWidth) * loopStartPostion / loopEndPostion);
                DebugDrawQuad(
                    "Loop",
                    DefaultPrintSoundInfoPosX + loopSamplePositionBarOffsetX,
                    DefaultPrintSoundInfoPosY + LoopSamplePositionBarOffsetY,
                    DefaultPrintSoundInfoPosX + DebugViewer::DefaultDrawBarWidth,
                    DefaultPrintSoundInfoPosY + LoopSamplePositionBarOffsetY + LoopSamplePositionBarHeight,
                    GetDebugColor( 80, 0, 0 )
                );
            }
        }
    }
} // NOLINT(impl/function_size)
#endif

nn::atk::SoundSystem::SoundSystemParam& CommonObject::InitializeParam::GetSoundSystemParam() NN_NOEXCEPT
{
    return m_SoundSystemParam;
}

const nn::atk::SoundSystem::SoundSystemParam& CommonObject::InitializeParam::GetSoundSystemParam() const NN_NOEXCEPT
{
    return m_SoundSystemParam;
}

void CommonObject::InitializeParam::SetFlagList(FlagList* pFlagList) NN_NOEXCEPT
{
    m_pFlagList = pFlagList;
}

FlagList* CommonObject::InitializeParam::GetFlagList() NN_NOEXCEPT
{
    return m_pFlagList;
}

const FlagList* CommonObject::InitializeParam::GetFlagList() const NN_NOEXCEPT
{
    return m_pFlagList;
}
