﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RendererVoice.h"

#include "../FlagList.h"
#include "../GfxCode/DebugViewer.h"

namespace
{
    FlagList g_LocalFlagList(nullptr, 0);
}

void RendererVoiceCheckModule::OnInitializeAtk() NN_NOEXCEPT
{
    m_CommonObject.Initialize();
}

void RendererVoiceCheckModule::OnFinalizeAtk() NN_NOEXCEPT
{
    m_CommonObject.Finalize();
}

void RendererVoiceCheckModule::OnLoadData() NN_NOEXCEPT
{
    m_CommonObject.LoadData();
}

void RendererVoiceCheckModule::OnPrintUsage() NN_NOEXCEPT
{
    m_CommonObject.PrintUsage();

    NN_LOG("[Up]           Play Voice\n");
    NN_LOG("[Down]         Stop Voice\n");
    NN_LOG("[Left]         Free Voice\n");
    NN_LOG("[Right]        Alloc Voice\n");
}

void RendererVoiceCheckModule::OnUpdateInput() NN_NOEXCEPT
{
    m_CommonObject.UpdateInput();

    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Down >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_pRendererVoice->SetVoiceState(nn::atk2::detail::VoiceState_Stop);
            NN_LOG("Set stop state[0x%08x].\n", m_pRendererVoice);
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Up >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_CommonObject.GetAudioEngine().GetRendererManager().Lock();
            m_CommonObject.GetAudioEngine().GetRendererVoiceManager().Lock();
            m_pRendererVoice->AppendWaveBufferList(&m_WaveBuffer);
            m_pRendererVoice->SetVoiceState(nn::atk2::detail::VoiceState_Play);
            m_CommonObject.GetAudioEngine().GetRendererVoiceManager().Unlock();
            m_CommonObject.GetAudioEngine().GetRendererManager().Unlock();
            NN_LOG("Set play state[0x%08x].\n", m_pRendererVoice);
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Left >() )
    {
        if (m_pRendererVoice != nullptr)
        {
            m_CommonObject.GetAudioEngine().GetRendererVoiceManager().FreeVoice(m_pRendererVoice);
            NN_LOG("Free renderer voice.\n");
        }
    }
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Right >() )
    {
        if (m_pRendererVoice == nullptr)
        {
            m_CommonObject.GetAudioEngine().GetRendererManager().Lock();
            m_CommonObject.GetAudioEngine().GetRendererVoiceManager().Lock();

            m_pRendererVoice = m_CommonObject.GetAudioEngine().GetRendererVoiceManager().AllocVoice();
            m_pRendererVoice->SetSampleRate(48000);
            m_pRendererVoice->SetSampleFormat(nn::atk2::detail::SampleFormat_PcmS16);
            m_pRendererVoice->AcquireVoiceSlot();
            m_pRendererVoice->SetVoicePriority(0);
            m_pRendererVoice->SetVoiceVolume(0.3f);
            m_pRendererVoice->SetVoicePitch(1.0f);
            m_pRendererVoice->SetVoiceBiquadFilterParameter(0);
            m_pRendererVoice->SetOutputReceiver(&m_CommonObject.GetAudioEngine().GetRendererManager().GetFinalMix());
            m_pRendererVoice->SetVoiceMixVolume(0.5f, 0, 0);
            m_pRendererVoice->SetVoiceMixVolume(0.5f, 0, 1);

            int sampleCount = 48000;
            size_t bufferSize = sampleCount * sizeof(int16_t);
            void* buffer = nns::atk::AllocateForMemoryPool(bufferSize, nn::atk2::detail::RendererManager::BufferAlignSize);
            int16_t* sampleArray = reinterpret_cast<int16_t*>(buffer);
            const float Pi = 3.1415926535897932384626433f;
            for (auto i = 0; i < sampleCount; ++i)
            {
                sampleArray[i] = static_cast<int16_t>(std::numeric_limits<int16_t>::max() * sinf(2 * Pi * 440 * i / 48000));
            }

            m_WaveBuffer.bufferAddress = buffer;
            m_WaveBuffer.bufferSize = bufferSize;
            m_WaveBuffer.loopFlag = true;
            m_WaveBuffer.sampleLength = sampleCount;
            m_WaveBuffer.sampleOffset = 0;

            NN_LOG("Alloc renderer voice[0x%08x].\n", m_pRendererVoice);

            m_CommonObject.GetAudioEngine().GetRendererVoiceManager().Unlock();
            m_CommonObject.GetAudioEngine().GetRendererManager().Unlock();
        }
    }
}

void RendererVoiceCheckModule::OnUpdateAtk() NN_NOEXCEPT
{
    m_CommonObject.Update();
}

void RendererVoiceCheckModule::OnUpdateDraw() NN_NOEXCEPT
{
    m_CommonObject.UpdateDraw(GetModuleName());
}

const char* RendererVoiceCheckModule::GetModuleName() const NN_NOEXCEPT
{
    return "RendererVoice";
}

FlagList& RendererVoiceCheckModule::GetLocalFlagList() NN_NOEXCEPT
{
    return g_LocalFlagList;
}
