﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AocSeasonPassDelivery.cpp,PageSampleAocSeasonPassDelivery}
 *
 * @brief
 *  シーズンパスを利用するコンテンツ配信時のサンプルプログラム
 */

/**
 * @page PageSampleAocSeasonPassDelivery コンテンツ配信時
 * @tableofcontents
 *
 * @brief
 *  シーズンパスを利用するコンテンツ配信時のサンプルプログラムの解説です。
 *
 * @section PageSampleAocSeasonPassDelivery_SectionBrief 概要
 *  ここでは、シーズンパスのコンテンツ配信時のサンプルプログラムの説明を行います。
 *
 *  シーズンパスは以下の流れで行い、本サンプルは 2 に該当します。
 *  1.シーズンパスの販売開始時には、将来配信するコンテンツにはダミーのデータを作成する。(01_SalesStart のサンプル)
 *  2.シーズンパスとしてのデータの追加配信は、ダミーとして登録していたコンテンツを本物のデータに更新することで実現する。(本サンプル)
 *
 *  パッチ更新を行うタイミングで、関連する追加コンテンツに対しても自動更新が行われます。
 *  そのため、本サンプルのようにパッチ更新とシーズンパス更新を同時に行うケースでは基本的には、新しいパッチと新しいシーズンパスが同時にインストールされている状態になります。
 *  但し、SDカード挿抜などを組み合わせた例外ケースでは、パッチだけ新しく追加コンテンツが古いというケースが起こり得ます。そのような場合はアプリ側でハンドルする必要があります。
 *  アプリ側で追加コンテンツのバージョンが低いことを検出した場合は nn::err::ShowUnacceptableAddOnContentVersionError を使うことで、ユーザに明示的な追加コンテンツの更新を促すことが可能です。
 *  上記エラー判定をするためにも、追加コンテンツにはアプリから読み取れるバージョン値を含めることを推奨します。
 *
 *  詳しくは、「Nintendo Switch eコマースガイド」の「5.2.1. シーズンパスの実現方法」をご参照ください。
 *
 *  追加コンテンツを使用する際には、追加コンテンツライブラリを使用します。
 *  追加コンテンツライブラリでは本体にインストールされている追加コンテンツの管理を行います。
 *  実際に追加コンテンツの内容を読み取る際は、ファイルシステムライブラリを併せて使用する必要があることに注意してください。
 *
 *  @ref nn::aoc "追加コンテンツライブラリの関数リファレンス"
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス"
 *
 * @section PageSampleAocSeasonPassDelivery_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AocSeasonPass/02_Delivery Samples/Sources/Applications/AocSeasonPass/02_Delivery @endlink 以下にあります。
 *
 * @section PageSampleAocSeasonPassDelivery_SectionNecessaryEnvironment 必要な環境
 *  開発機 に ApplicationId が 0x0100cb3000062000 の追加コンテンツがインストールされている必要があります。
 *  バージョン 1 の追加コンテンツをインストールするには AocSeasonPass/02_Delivery のディレクトリ内の InstallAoc.bat を実行してください。
 *  (フォルダ 01_SalesStart にあるバージョン 0 の追加コンテンツがインストールされている場合はエラーを表示し、バージョン 1 の場合はエラーを表示しない作りになっています。)
 *
 * @section PageSampleAocSeasonPassDelivery_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleAocSeasonPassDelivery_SectionPrecaution 注意事項
 *  このデモは、Aoc のバージョン 1 がインストール済みの場合、画面上に何も表示されません。実行結果はログに出力されます。
 *  Aocのバージョンが 0 のみがインストールされている場合は、画面上にコンテンツの更新を促すメッセージを表示し、ログにエラーを表示します。（追加コンテンツがアプリケーションの必須バージョンを満たしていないため）
 *
 *  ApplicationId が 0x0100cb3000062000 の追加コンテンツが一つも(バージョン 0,1 両方)インストールされていない場合、
 *  @ref nn::aoc::CountAddOnContent の出力は 0 となり、期待する出力と一致しなくなることに注意してください。
 *  SDEV を接続したうえで、InstallAoc.bat を呼び出して、追加コンテンツをインストールしてください。
 *  InstallAoc.bat では追加コンテンツのビルドとインストールが行われます。
 *  ビルド時には Aoc/SampleAoc.nmeta が参照されます。このファイルで設定されている各項目の詳細については
 *  ドキュメントの「機能／AOC ライブラリ」を参照してください。
 *
 *  プログラムと追加コンテンツの ApplicationId は一致していなければなりません。
 *  フォルダに同梱されている VisualStudio プロジェクトを使用する場合は自動的に ApplicationId が 0x0100cb3000062000 となります。
 *  独自のビルドシステムを使用する場合は、フォルダに同梱されている .nmeta ファイルを参照してプログラムの ApplicationId を明示的に指定するようにしてください。
 *  nn::err::ShowUnacceptableAddOnContentVersionError で、追加コンテンツのバージョンが古い場合に更新を求めるメッセージを表示させるには、
 *  追加コンテンツの .nmeta ファイルの RequiredApplicationReleaseVersion を、パッチファイル(本サンプルのこと)の .nmeta ファイルの ReleaseVersion 以上の値(同値推奨)に
 *  設定する必要があります。
 *  パッチの作成は「Nintendo Switch パッチガイド 」をご参照ください。
 *
 * @section PageSampleAocSeasonPassDelivery_SectionHowToExecute 実行手順
 *  開発機を接続した状態で AocSeasonPass/02_Delivery のディレクトリ内の InstallAoc.bat を実行又は、
 *  AocSeasonPass/01_SalesStart のディレクトリ内の InstallAoc.bat を実行してください。
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAocSeasonPassDelivery_SectionDetail 解説
 *
 * @subsection PageSampleAocSeasonPassDelivery_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AocSeasonPassDelivery.cpp
 *  @includelineno AocSeasonPassDelivery.cpp
 *
 * @subsection PageSampleAocSeasonPassDelivery_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - シーズンパスを利用するコンテンツ配信時のサンプルプログラム
 *
 *  このプログラムでは、インストール済みの追加コンテンツを列挙してその内容を読み取る操作を行っています。
 *  バージョン1をインストールした場合のサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  AocSeasonPass_OutputExample.txt
 *
 *  概ね以下の順に処理を行い、追加コンテンツ内のファイルの内容を読み取っています。
 *
 *
 *  - 追加コンテンツ数の取得(@ref nn::aoc::CountAddOnContent "nn::aoc::CountAddOnContent")
 *  - 追加コンテンツ数のリストアップ(@ref nn::aoc::ListAddOnContent "nn::aoc::ListAddOnContent")
 *  - 追加コンテンツのマウント(@ref nn::fs::MountAddOnContent "nn::fs::MountAddOnContent")
 *  - 追加コンテンツ内のファイルの読み取り(@ref nn::fs::OpenFile "nn::fs::OpenFile", @ref nn::fs::ReadFile "nn::fs::ReadFile")
 *  - 追加コンテンツのバージョンエラー(更新促し)表示(@ref nn::err::ShowUnacceptableAddOnContentVersionError "nn::err::ShowUnacceptableAddOnContentVersionError")
 *  - 追加コンテンツのアンマウント(@ref nn::fs::Unmount "nn::fs::Unmount")
 *
 *  それぞれの関数の詳細については、関数リファレンスを参照してください。
 *  このサンプルプログラムでは処理を単純化するために、追加コンテンツマウント後に決め打ちのファイル名で
 *  データにアクセスしていることに注意してください。
 *  実際のアプリケーションでは追加コンテンツマウント後は @ref nn::fs::OpenDirectory や @ref nn::fs::ReadDirectory を併用して
 *  データアクセスすることが一般的です。
 *
 */

#include <new>
#include <nn/aoc.h>
#include <nn/fs.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/err/err_ShowUnacceptableAddOnContentVersionErrorApi.h>

namespace {
    // 追加コンテンツインデックスを列挙するためのバッファ
    const int MaxListupCount = 256;
    nn::aoc::AddOnContentIndex g_AocListupBuffer[MaxListupCount];

    // 追加コンテンツ内のファイルの内容を読み取るためのバッファ
    const int ReadBufferSize = 1024;
    nn::Bit8 g_ReadBuffer[ReadBufferSize] = {};

    // 確認するAocのインデックス
    const int VersionAocIndex = 1;

    // Aoc Version1の内容
    const char* TextAocVersion = "This content is official data of season pass.";
}


extern "C" void nnMain()
{
    nn::Result result;
    NN_LOG("\n\nAocSeasonPassDelivery\n");

    // 追加コンテンツ数の取得
    int aocCount = nn::aoc::CountAddOnContent();
    NN_LOG("CountAddOnCountent -> %d\n", aocCount);

    // 追加コンテンツのリストアップ
    int listupCount = nn::aoc::ListAddOnContent(g_AocListupBuffer, 0, MaxListupCount);
    NN_LOG("ListAddOnCountent  -> %d\n", listupCount);
    for (int i = 0; i < listupCount; ++i)
    {
        NN_LOG("  Index[%d]: %d\n", i, g_AocListupBuffer[i]);
    }
    NN_ASSERT((listupCount == 1) && (g_AocListupBuffer[0] == VersionAocIndex));

    // リストアップされた追加コンテンツを読み取り、表示する
    // 追加コンテンツのマウント
    size_t cacheSize = 0;
    result = nn::fs::QueryMountAddOnContentCacheSize(&cacheSize, VersionAocIndex);
    NN_ASSERT(result.IsSuccess());

    char* mountCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountCacheBuffer);

    const char MountName[] = "aoc";
    result = nn::fs::MountAddOnContent(
        MountName, VersionAocIndex, mountCacheBuffer, cacheSize
    );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // マウントされた追加コンテンツのファイルへのアクセス
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "aoc:/versionData", nn::fs::OpenMode::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, g_ReadBuffer, sizeof(g_ReadBuffer));
    NN_ASSERT(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);

    g_ReadBuffer[readSize] = '\0';
    NN_LOG("%s\n\n", g_ReadBuffer);

    const char* const aocVersion = reinterpret_cast<const char*>(g_ReadBuffer);
    // Aoc のバージョンが必須バージョン(本サンプルでは 1) を満たしていない場合はエラー
    if (strncmp(TextAocVersion, aocVersion, readSize) != 0)
    {
        NN_LOG("The version of Aoc is old.\n\n\n");
        nn::err::ShowUnacceptableAddOnContentVersionError();
    }
    else
    {
        NN_LOG("Available Aoc.\n\n\n");
    }

    // 追加コンテンツのアンマウント
    nn::fs::Unmount(MountName);
    delete[] mountCacheBuffer;
}
