﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AocSeasonPassSalesStart.cpp,PageSampleAocSeasonPassSalesStart}
 *
 * @brief
 *  シーズンパスを利用するソフト販売開始時のサンプルプログラム
 */

/**
 * @page PageSampleAocSeasonPassSalesStart ソフト販売開始時
 * @tableofcontents
 *
 * @brief
 *  シーズンパスを利用するソフト販売開始時のサンプルプログラムの解説です。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionBrief 概要
 *  ここでは、シーズンパスを利用する販売開始時のサンプルプログラムの説明を行います。
 *
 *  シーズンパスは以下の流れで行い、本サンプルは 1 に該当します。
 *  1.シーズンパスの販売開始時には、将来配信するコンテンツにはダミーのデータを作成する。(本サンプル)
 *  2.シーズンパスとしてのデータの追加配信は、ダミーとして登録していたコンテンツを本物のデータに更新することで実現する。(02_Delivery のサンプル)
 *
 *  シーズンパスアイテムを始めに販売する際には、
 *  今後配信する追加コンテンツの数だけ正式版とは別のダミーの追加コンテンツを含んだ状態を作成してください。
 *
 *  詳しくは、「Nintendo Switch eコマースガイド」の「5.2.1. シーズンパスの実現方法」をご参照ください。
 *
 *  追加コンテンツを使用する際には、追加コンテンツライブラリを使用します。
 *  追加コンテンツライブラリでは本体にインストールされている追加コンテンツの管理を行います。
 *  実際に追加コンテンツの内容を読み取る際は、ファイルシステムライブラリを併せて使用する必要があることに注意してください。
 *
 *  @ref nn::aoc "追加コンテンツライブラリの関数リファレンス"
 *  @ref nn::fs "ファイルシステムライブラリの関数リファレンス"
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AocSeasonPass/01_SalesStart Samples/Sources/Applications/AocSeasonPass/01_SalesStart @endlink 以下にあります。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionNecessaryEnvironment 必要な環境
 *  開発機 に ApplicationId が 0x0100cb3000062000 の追加コンテンツがインストールされている必要があります。
 *  この追加コンテンツをインストールするには AocSeasonPass/01_SalesStart のディレクトリ内の InstallAoc.bat を実行してください。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 *  ApplicationId が 0x0100cb3000062000 の追加コンテンツがインストールされていない場合、
 *  @ref nn::aoc::CountAddOnContent の出力は 0 となり、期待する出力と一致しなくなることに注意してください。
 *  SDEV を接続したうえで、InstallAoc.bat を呼び出して、追加コンテンツをインストールしてください。
 *  InstallAoc.bat では追加コンテンツのビルドとインストールが行われます。
 *  ビルド時には Aoc/SampleAoc.nmeta が参照されます。このファイルで設定されている各項目の詳細については
 *  ドキュメントの「機能／AOC ライブラリ」を参照してください。
 *
 *  プログラムと追加コンテンツの ApplicationId は一致していなければなりません。
 *  フォルダに同梱されている VisualStudio プロジェクトを使用する場合は自動的に ApplicationId が 0x0100cb3000062000 となります。
 *  独自のビルドシステムを使用する場合は、フォルダに同梱されている .nmeta ファイルを参照してプログラムの ApplicationId を明示的に指定するようにしてください。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionHowToExecute 実行手順
 *  開発機を接続した状態でAocSeasonPass/01_SalesStart のディレクトリ内の InstallAoc.bat を実行してください。
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAocSeasonPassSalesStart_SectionDetail 解説
 *
 * @subsection PageSampleAocSeasonPassSalesStart_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AocSeasonPassSalesStart.cpp
 *  @includelineno AocSeasonPassSalesStart.cpp
 *
 * @subsection PageSampleAocSeasonPassSalesStart_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 *  - シーズンパスを利用するソフト販売開始時のサンプルプログラム
 *
 *  このプログラムでは、インストール済みの追加コンテンツを列挙してその内容を読み取る操作を行っています。
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  AocSeasonPass_OutputExample.txt
 *
 *  概ね以下の順に処理を行い、追加コンテンツ内のファイルの内容を読み取っています。
 *
 *
 *  - 追加コンテンツ数の取得(@ref nn::aoc::CountAddOnContent "nn::aoc::CountAddOnContent")
 *  - 追加コンテンツ数のリストアップ(@ref nn::aoc::ListAddOnContent "nn::aoc::ListAddOnContent")
 *  - 追加コンテンツのマウント(@ref nn::fs::MountAddOnContent "nn::fs::MountAddOnContent")
 *  - 追加コンテンツ内のファイルの読み取り(@ref nn::fs::OpenFile "nn::fs::OpenFile", @ref nn::fs::ReadFile "nn::fs::ReadFile")
 *  - 追加コンテンツのアンマウント(@ref nn::fs::Unmount "nn::fs::Unmount")
 *
 *  それぞれの関数の詳細については、関数リファレンスを参照してください。
 *  このサンプルプログラムでは処理を単純化するために、追加コンテンツマウント後に決め打ちのファイル名で
 *  データにアクセスしていることに注意してください。
 *  実際のアプリケーションでは追加コンテンツマウント後は @ref nn::fs::OpenDirectory や @ref nn::fs::ReadDirectory を併用して
 *  データアクセスすることが一般的です。
 *
 */

#include <new>
#include <nn/aoc.h>
#include <nn/fs.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

namespace {
    // 追加コンテンツインデックスを列挙するためのバッファ
    const int MaxListupCount = 256;
    nn::aoc::AddOnContentIndex g_AocListupBuffer[MaxListupCount];

    // 追加コンテンツ内のファイルの内容を読み取るためのバッファ
    const int ReadBufferSize = 1024;
    nn::Bit8 g_ReadBuffer[ReadBufferSize] = {};

    // 確認するAocのインデックス
    const int VersionAocIndex = 1;
}


extern "C" void nnMain()
{
    nn::Result result;
    NN_LOG("\n\nAocSeasonPassSalesStart\n");

    // 追加コンテンツ数の取得
    int aocCount = nn::aoc::CountAddOnContent();
    NN_LOG("CountAddOnCountent -> %d\n", aocCount);

    // 追加コンテンツのリストアップ
    int listupCount = nn::aoc::ListAddOnContent(g_AocListupBuffer, 0, MaxListupCount);
    NN_LOG("ListAddOnCountent  -> %d\n", listupCount);
    for (int i = 0; i < listupCount; ++i)
    {
        NN_LOG("  Index[%d]: %d\n", i, g_AocListupBuffer[i]);
    }
    NN_ASSERT((listupCount == 1) && (g_AocListupBuffer[0] == VersionAocIndex));

    // リストアップされた追加コンテンツを読み取り、表示する
    // 追加コンテンツのマウント
    size_t cacheSize = 0;
    result = nn::fs::QueryMountAddOnContentCacheSize(&cacheSize, VersionAocIndex);
    NN_ASSERT(result.IsSuccess());

    char* mountCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountCacheBuffer);

    const char MountName[] = "aoc";
    result = nn::fs::MountAddOnContent(
        MountName, VersionAocIndex, mountCacheBuffer, cacheSize
    );
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // マウントされた追加コンテンツのファイルへのアクセス
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "aoc:/versionData", nn::fs::OpenMode::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, g_ReadBuffer, sizeof(g_ReadBuffer));
    NN_ASSERT(result.IsSuccess());
    nn::fs::CloseFile(fileHandle);

    g_ReadBuffer[readSize] = '\0';
    NN_LOG("%s\n\n", g_ReadBuffer);

    // 追加コンテンツのアンマウント
    nn::fs::Unmount(MountName);
    delete[] mountCacheBuffer;
}
