﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AlbumSaveAndShare_Main.cpp,PageSampleAlbumSaveAndShare}

    @brief @copybrief PageSampleAlbumSaveAndShare
 */

/**
    @page PageSampleAlbumSaveAndShare 生成した画像を画面写真として保存・投稿する

    @tableofcontents

    @brief アプリケーションが生成した画像を画面写真としてアルバムに保存・ SNS へ投稿する方法を示します。

    @section PageSampleAlbumSaveAndShare_SectionNotification 注意
    本サンプルプログラムでは SNS への投稿画面が表示されます。
    投稿画面から投稿を完了した場合、実際の SNS へ投稿されることに注意してください。

    X ボタンや B ボタンにより投稿画面を終了することで SNS へ投稿せずにサンプルプログラムを終了することができます。

    @section PageSampleAlbumSaveAndShare_SectionBrief 概要
    アルバムライブラリをアプリケーションから使用する方法を示します。

    @section PageSampleAlbumSaveAndShare_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AlbumSaveAndShare
    Samples/Sources/Applications/AlbumSaveAndShare @endlink 以下にあります。

    @section PageSampleAlbumSaveAndShare_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機向けにのみビルドし実行することができます。
    Windows 環境はサポートされていません。

    - 投稿画面を表示するためにインターネット接続が必要です。
    - ユーザーアカウントおよびニンテンドーアカウントが必要です。

    @section PageSampleAlbumSaveAndShare_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。

    サンプルプログラムを起動すると最初にユーザーアカウント選択画面が表示されます。
    ここで SNS 投稿に使用するユーザーアカウントを選択してください。

    ユーザーアカウントの選択が完了すると SNS への投稿画面が表示されます。
    本サンプルプログラム内で保存した画面写真の投稿を行うことができます。
    この画面から投稿操作を完了させた場合、実際の SNS に投稿されることに注意してください。
    X ボタンや B ボタンによって投稿画面を中断することで SNS への投稿を行わずに終了します。

    @section PageSampleAlbumSaveAndShare_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - アルバムライブラリおよび必要なライブラリの初期化
    - ユーザーアカウントの選択画面の表示
    - 保存する画像データの生成
    - 画像データを画面写真として保存して投稿画面を表示する
    - アルバムライブラリの終了

    下記ソースコードの随所に補足説明を記載していますので、詳細はそちらを参照してください。

    AlbumSaveAndShare_Main.cpp
    @includelineno AlbumSaveAndShare_Main.cpp

    本サンプルで使用している nn::album::SaveAndShareScreenshot() では
    記録メディアへの静止画像ファイルの書き込みが行なわれます。

    記録メディアへの書き込み回数や書き込み容量に関しては、
    別途ガイドラインが規定されていますので、
    そちらに注意して API を不必要に乱発しないようにして下さい。

    なお、書き込み頻度を調査するためのデバッグ機能も別途提供されています。
    詳細は「@confluencelink{174228221,FS ライブラリ／デバッグ機能／アクセス頻度確認}」を参照して下さい。

 */

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/album/album_Result.h>
#include <nn/album/album_SaveAndShareScreenshot.h>
#include <nn/album/album_GetSharePostedService.h>
#include <nn/album/album_GetSharePostedLastUrl.h>
#include <nn/nifm.h>
#include <nn/account.h>

// 保存する画面写真のサイズ。
// サポートされる画面写真のサイズはファームウェアにより決められています。
const nn::album::ImageSize ScreenShotSize = nn::album::ImageSize_1280x720;

// 保存結果の通知設定。
// 画像をアルバムに保存した結果をユーザーに自動的に通知する条件を選ぶことができます。
// 自動的に通知しなかった場合でも API の戻り値によってアプリケーションが独自の通知を行うことができます。
const nn::album::AlbumReportOption ReportOption = nn::album::AlbumReportOption_ReportAlways;

// 保存する画像のサイズ。
// アプリケーションは画面写真と同じ大きさの画像を用意する必要があります。
const int PixelSize = 4;
const int ImageWidth = 1280;
const int ImageHeight = 720;
uint8_t g_ImageData[PixelSize * ImageWidth * ImageHeight] = {};

namespace {
    // 画面写真として保存する画像データを生成します。
    // このサンプルでは上半分が白、下半分が赤の画像を作成します。
    // 画像データの色フォーマットは R8G8B8A8 ですが、アルファチャンネルは無視されます。
    void SetupImageData() NN_NOEXCEPT
    {
        uint8_t* pImage = g_ImageData;
        for(int y = 0; y < ImageHeight; y++)
        {
            uint8_t* pLine = pImage + PixelSize * ImageWidth * y;
            for(int x = 0; x < ImageWidth; x++)
            {
                uint8_t* pPixel = pLine + PixelSize * x;
                if(y < ImageHeight / 2)
                {
                    pPixel[0] = 255; // R
                    pPixel[1] = 255; // G
                    pPixel[2] = 255; // B
                    pPixel[3] = 255; // A （無視される）
                }
                else
                {
                    pPixel[0] = 255; // R
                    pPixel[1] = 0;   // G
                    pPixel[2] = 0;   // B
                    pPixel[3] = 255; // A （無視される）
                }

            }
        }
    }
}

extern "C" void nnMain() NN_NOEXCEPT
{
    // アルバムライブラリと必要なライブラリを初期化します。
    nn::nifm::Initialize();
    nn::account::Initialize();
    nn::album::Initialize();

    // 投稿に使用する UserId を選択するためのアカウント選択画面を表示します。
    // UserId を指定せずに (nn::account::InvalidUid を指定する) 投稿 API を呼び出すこともできます。
    // この場合、投稿 API 中でアカウント選択画面が表示されます。
    NN_LOG("Select user\n");
    nn::account::Uid uid = nn::account::InvalidUid;
    {
        auto result = nn::account::ShowUserSelector(&uid);
        if(result.IsFailure())
        {
            NN_LOG("  -> UserSelect Canceled(%d-%d)\n", result.GetModule(), result.GetDescription());
            return;
        }
    }
    NN_LOG("  -> OK\n");

    // 保存する画像を生成します。
    // g_ImageData に値が設定されます
    SetupImageData();

    // 生成した画像を画面写真として保存して投稿画面を表示します。
    nn::album::ShareId shareId = {};
    {
        NN_LOG("SaveAndShareScreenshot\n");
        auto result = nn::album::SaveAndShareScreenshot(
            &shareId,
            g_ImageData,
            sizeof(g_ImageData),
            ScreenShotSize,
            ReportOption,
            uid
        );
        if(result.IsFailure())
        {
            if(nn::album::ResultAlbumUnknown::Includes(result))
            {
                if(nn::album::ResultAlbumError::Includes(result))
                {
                    // 画面写真の作成や保存に失敗した場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumError)\n");
                }
                else if(nn::album::ResultAlbumFull::Includes(result))
                {
                    // 画面写真を保存するストレージが満杯だった場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFull)\n");
                }
                else if(nn::album::ResultAlbumFileCountLimit::Includes(result))
                {
                    // 画面写真の数がアルバムに保存できる最大数に達している場合にこのエラーが返ります。
                    NN_LOG("  -> Screenshot Failed(AlbumFileCountLimit)\n");
                }
                else
                {
                    NN_LOG("  -> Screenshot Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultShareUnknown::Includes(result))
            {
                if(nn::album::ResultShareCanceled::Includes(result))
                {
                    // ユーザー操作によって投稿がキャンセルされた場合にこのエラーが返ります。
                    NN_LOG("  -> Share Failed(ShareCanceled)\n");
                }
                else if(nn::album::ResultShareFailed::Includes(result))
                {
                    // 投稿中にエラーが発生した場合にこのエラーが返ります。
                    NN_LOG("  -> Share Failed(ShareFailed)\n");
                }
                else
                {
                    NN_LOG("  -> Share Failed(Unknown)\n");
                }
            }
            else if(nn::album::ResultNotSupported::Includes(result))
            {
                NN_LOG("  -> NotSupported\n");
            }
            else
            {
                NN_LOG("  -> Error(Unknown)\n");
            }
            return;
        }
    }
    NN_LOG("  -> OK\n");

    // 投稿先情報を取得
    {
        char serviceName[nn::album::SharePostedServiceNameSizeMax] = {};
        char entryId[nn::album::SharePostedEntryIdSizeMax] = {};
        char url[nn::album::SharePostedLastUrlSizeMax] = {};
        int serviceNameLength = nn::album::GetSharePostedServiceName(serviceName, sizeof(serviceName), shareId);
        int entryIdLength = nn::album::GetSharePostedEntryId(entryId, sizeof(entryId), shareId);
        int urlLength = nn::album::GetSharePostedLastUrl(url, sizeof(url), shareId);
        NN_LOG("ServiceName: %s (%d)\n", serviceName, serviceNameLength);
        NN_LOG("EntryId    : %s (%d)\n", entryId, entryIdLength);
        NN_LOG("Url        : %s (%d)\n", url, urlLength);
    }

    // アルバムライブラリを終了します。
    nn::album::Finalize();
}
