﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{AccountApplicationExternalNsa.cpp,PageSampleAccountApplicationExternalNsa}

    @brief @copybrief PageSampleAccountApplicationExternalNsa
 */

/**
    @page PageSampleAccountApplicationExternalNsa 本体システム外部のネットワークサービスアカウントのアプリケーションでの利用

    @tableofcontents

    @brief アカウントライブラリを使用して、本体システム外部のネットワークサービスアカウントをアプリケーションで利用する取得する方法を示します。

    @section PageSampleAccountApplicationExternalNsa_SectionBrief 概要
    アカウントライブラリを使用して、本体システム外部のネットワークサービスアカウントをアプリケーションで利用する取得する方法を示します。

    このサンプルプログラムで示す処理は、下記の例にあるような、外部のネットワークサービスアカウントをアプリケーションで利用する場合にのみ必要です。

    - アプリケーションに一時的にユーザーを参加させる
    - 大会などで不特定のユーザーを参加させる

    @section PageSampleAccountApplicationExternalNsa_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/AccountApplicationExternalNsa
    Samples/Sources/Applications/AccountApplicationExternalNsa @endlink 以下にあります。

    @section PageSampleAccountApplicationExternalNsa_SectionNecessaryEnvironment 必要な環境
    本サンプルプログラムは開発機向けにのみビルドし実行することができます。
    Windows 環境をサポートしていないことに注意してください。

    @section PageSampleAccountApplicationExternalNsa_SectionHowToOperate 操作方法
    一般的なサンプルプログラムと同様に本プログラムをビルドし、実行してください。
    このプログラムの実行の経過は画面への文字の描画、及び実行ログとして出力されます。

    @section PageSampleAccountApplicationExternalNsa_SectionDetail 解説
    このサンプルプログラムは次の順序で処理を行います。

    - 本体システム外部のネットワークサービスアカウントの導入
    - ネットワークサービスアカウントの使用

    下記ソースコードの随所に補足説明を記載していますので、詳細はそちらを参照してください。

    AccountApplicationExternalNsa.cpp
    @includelineno AccountApplicationExternalNsa.cpp

    実行の結果、次のようなログが出力されます。

    AccountApplicationExternalNsa_ExampleOutput.txt
    @verbinclude AccountApplicationExternalNsa_ExampleOutput.txt
 */

#include "AccountApplicationExternalNsa.h"

#include <cstdlib>

#include <nn/account/account_Api.h>
#include <nn/account/account_ApiForApplications.h>
#include <nn/account/account_ExternalNetworkServiceAccountInfo.h>
#include <nn/account/account_Result.h>

#include <nn/nn_Assert.h>
#include <nn/err.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiNetworkConnection.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_ScopeExit.h>

namespace
{
NN_ALIGNAS(4096) char g_ExternalNsaInfoBuffer[nn::account::RequiredBufferSizeForExternalNetworkServiceAccountInfo];

} // 無名名前空間終わり

// アプリケーションのメイン
void Program::Execute() NN_NOEXCEPT
{
    // 依存するモジュールの初期化
    nn::nifm::Initialize();

    // アカウントライブラリの初期化
    nn::account::Initialize();

    // 本体システムに登録されていないユーザー向けシーンを開始
    SceneExternalUser();
}

// 本体システムに登録されていないユーザー向けシーン
void Program::SceneExternalUser() NN_NOEXCEPT
{
    // 本体システムに登録されていないユーザーの利用はオンライン状態でのみ可能
    while (NN_STATIC_CONDITION(true))
    {
        nn::nifm::SubmitNetworkRequestAndWait();
        if (!nn::nifm::IsNetworkAvailable())
        {
            auto r = nn::nifm::HandleNetworkRequestResult();
            if (!r.IsSuccess())
            {
                if (!nn::nifm::ResultErrorHandlingCompleted::Includes(r))
                {
                    // エラーかつ、それを解消できなかった場合はシーンを終了する。
                    return;
                }
                // エラーかつ、それを解消できた場合はリトライする。
                continue;
            }
            // break
        }
        // ネットワーク接続を利用できる。
        break;
    }

    // 本体システム外部のネットワークサービスアカウントの導入
    //  - 本体システムの UI が表示され、ユーザー操作が行われる。
    //  - ユーザーはキャンセルすることができる。
    nn::account::ExternalNetworkServiceAccountInfo externalNsaInfo;
    auto r = nn::account::IntroduceExternalNetworkServiceAccount(&externalNsaInfo, g_ExternalNsaInfoBuffer, sizeof(g_ExternalNsaInfoBuffer));
    if (!r.IsSuccess())
    {
        if (nn::account::ResultCancelledByUser::Includes(r))
        {
            // 本体システムのUI上でユーザーがキャンセル操作を行った。
            Printfln("- IntroduceExternalNetworkServiceAccount cancelled");
            return;
        }
        // 上記以外で特にハンドルすべきエラーはない。
        NN_ABORT_UNLESS_RESULT_SUCCESS(r);
    }
    Printfln("- External Network Service Account logged-in");

    // ユーザーのニックネームを取得する
    nn::account::Nickname nickname;
    NN_ABORT_UNLESS_RESULT_SUCCESS(externalNsaInfo.GetNickname(&nickname));
    Printfln("- Nickname: \"%s\"", nickname.name);

    // ユーザーのプロフィール画像を取得する
    void* profileImage = nullptr; // nullptr を指定するとサイズのみ取得できる
    size_t profileImageSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(externalNsaInfo.GetProfileImage(&profileImageSize, profileImage, 0));
    Printfln("- Profile image: %zu bytes", profileImageSize);

    // ネットワークサービスアカウントIDを取得する
    nn::account::NetworkServiceAccountId nsaId;
    NN_ABORT_UNLESS_RESULT_SUCCESS(externalNsaInfo.GetNetworkServiceAccountId(&nsaId));
    Printfln("- Network Service Account ID: %016llx", nsaId.id);

    // ネットワークサービスアカウントのIDトークンキャッシュを取得する
    auto* idToken = reinterpret_cast<char*>(std::malloc(nn::account::NetworkServiceAccountIdTokenLengthMax));
    NN_UTIL_SCOPE_EXIT
    {
        std::free(idToken);
    };
    size_t idTokenLength;
    NN_ABORT_UNLESS_RESULT_SUCCESS(externalNsaInfo.LoadNetworkServiceAccountIdTokenCache(&idTokenLength, idToken, nn::account::NetworkServiceAccountIdTokenLengthMax));
    Printfln("- Network Service Account ID token: %.*s", idTokenLength, idToken);

    // :
    // その他オンライン状態で実行可能なアプリケーション実装
    // :
}
