﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/pad/demo_PadUtil.h>
#include <nw/math.h>

namespace nw
{
namespace demo
{
namespace detail
{

static const s32 STICK_VALUE_MIDDLE = 0x8000; //!< アナログスティックの値の中間値です。

//------------------------------------------------------------------------------
void
SetStickValueWithClamp( float* stickValue, int rawValue, bool isReverse, int stickClampMin, int stickClampMax )
{
    NW_ASSERT_NOT_NULL( stickValue );

    // アナログスティックの生の値は中間値が0x8000となっているが、最小値である0x0と
    // 最大値である0xffffで動く範囲が異なる(マイナス側は0x8000であるのに対し、
    // プラス側は0x7fffである)。このため、ここではプラス側とマイナス側で同一の
    // クランプ値（1～0x7fff）を使用した上で、マイナス側は有効な値域をマイナス方向に
    // 1広げることで対応している。
    float direction = isReverse ? -1.f : 1.f;

    if ( rawValue <= STICK_VALUE_MIDDLE - stickClampMin )
    {
        if (rawValue <= STICK_VALUE_MIDDLE - stickClampMax - 1)
        {
            *stickValue = -1.f * direction;
        }
        else
        {
            int total_size = stickClampMax - stickClampMin + 1;
            int clamped_value = STICK_VALUE_MIDDLE - rawValue - stickClampMin;
            *stickValue = - static_cast<float>(clamped_value) / total_size * direction;
        }
    }
    else if ( STICK_VALUE_MIDDLE + stickClampMin <= rawValue )
    {
        if (STICK_VALUE_MIDDLE + stickClampMax <= rawValue)
        {
            *stickValue = 1.f * direction;
        }
        else
        {
            int total_size = stickClampMax - stickClampMin;
            int clamped_value = rawValue - STICK_VALUE_MIDDLE - stickClampMin;
            *stickValue = static_cast<float>(clamped_value) / total_size * direction;
        }
    }
    else
    {
        *stickValue = 0.f;
    }
}

//------------------------------------------------------------------------------
void
NormalizeAnalogStickAsCircle( nw::math::VEC2* vec )
{
    // xかyの何れかが0の場合は正規化の必要はない(xとyが-1以上1以下になっているのは前提)
    if (vec->x == 0.f || vec->y == 0.f) return;

    /**
     *  正方形になっている入力を円形に正規化するためのアルゴリズム。
     *  出力のlengthは1以下になる。
     *
     *  計算方法としては、原点とvecを結んだ点をvec側に延長し、(-1,-1)-(1,1)を囲む
     *  正方形の何れかの辺に当たった点pと原点の距離dを求め、1/dでvecをスケールすれ
     *  ばよい。
     *
     *  ここでは、pからx軸もしくはy軸の近い方に垂直な線を引き、それとx軸もしくはy軸
     *  の近い方が交わる点をqとして、原点、q、pがなす直角三角形の斜辺がdになるので、
     *  三平方の定理で求めている。
     *
     *  変数は、直線 pq の長さが extendedXYMin に、1/d の値が rate に対応している。
     */
    float absX = nw::math::FAbs( vec->x );
    float absY = nw::math::FAbs( vec->y );

    float extendedXYMin;
    if ( absX <= absY )
    {
        extendedXYMin = absX * ( 1.f / absY );
    }
    else
    {
        extendedXYMin = absY * ( 1.f / absX );
    }

    float rate = 1.f / nw::math::FSqrt( 1.f + extendedXYMin * extendedXYMin );
    vec->x *= rate;
    vec->y *= rate;
}

//------------------------------------------------------------------------------
void
NormalizeStickValueCafe( nw::math::VEC2* out, s8 x, s8 y )
{
    static const float max_val = 56.f;

    out->Set(static_cast<float>(x) / max_val, static_cast<float>(y) / max_val);
    {
        float length = out->Length();
        if (length > 1.f)
        {
            *out /= length;
        }
    }
}

} // namespace detail
} // namespace demo
} // namespace nw

