﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo/gl/demo_SystemWinGL.h>
#include <nw/demo/gl/demo_TextureFrameBufferGL.h>
#include <nw/demo/demo_Define.h>
#include <nw/gfnd/gfnd_Graphics.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/dev/dev_Resource.h>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <windows.h>
#include <mmsystem.h>
#include <winext/cafe/os/win32/os_WindowMessage.h>

namespace nw
{
    namespace internal
    {
        namespace winext
        {
        // PcSDK で作成されるウィンドウのハンドルです。
        extern HWND ghCommandWnd;
        extern HWND ghFrameWnd;
            }
    }
}

namespace nw
{
namespace demo
{

SystemWinGL* SystemWinGL::s_Instance = NULL;

//------------------------------------------------------------------------------
SystemWinGL::SystemWinGL( const CreateArg& arg )
: m_Arg( arg ),
  m_HWnd( NULL ),
  m_Exit( false ),
  m_MsgProcCallback( NULL ),
  m_DefaultFrameBuffer( NULL ),
  m_CopyReservedFrameBuffer( NULL ),
  m_Pad( NULL ),
  m_WPad( NULL ),
  m_Mouse( NULL ),
  m_MeterDrawer( &m_MeterFrame ),
  m_TextWriterInitialized( false ),
  m_PointerTextureImage( NULL ),
  m_DisplayState( HIDE )
{
    NW_ASSERT( !s_Instance );
    s_Instance = this;
}

//------------------------------------------------------------------------------
void
SystemWinGL::Initialize()
{
    SetupWindow();

    // デモファイルシステムの初期化を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
        fileDeviceArg.allocator = m_Arg.allocator;
        fileSystem->Initialize( fileDeviceArg );
    }

    // Pad の初期化処理を行います。
    {
        nw::demo::JoyPadDeviceWin::GetInstance()->Initialize();
        nw::demo::KeyboardMouseDeviceWin::GetInstance()->Initialize();
        nw::demo::KeyboardMouseDeviceWin::GetInstance()->SetMainWindowHandle( m_HWnd );

        m_Pad = new( m_Arg.allocator->Alloc( sizeof( nw::demo::PadWin ) ) ) nw::demo::PadWin( 0 );
        m_Pad->SetKeyboardMouseDevice( nw::demo::KeyboardMouseDeviceWin::GetInstance() );
        m_Pad->SetJoyPadDevice( nw::demo::JoyPadDeviceWin::GetInstance() );

        m_WPad = new( m_Arg.allocator->Alloc( sizeof( nw::demo::WPadWin ) ) ) nw::demo::WPadWin();
        m_WPad->SetKeyboardMouseDevice( nw::demo::KeyboardMouseDeviceWin::GetInstance() );

        m_Mouse = new( m_Arg.allocator->Alloc( sizeof( nw::demo::MouseWin ) ) ) nw::demo::MouseWin();
        m_Mouse->SetKeyboardMouseDevice( nw::demo::KeyboardMouseDeviceWin::GetInstance() );
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::Finalize()
{
    // Pad の終了処理を行います。
    {
        nw::demo::JoyPadDeviceWin::GetInstance()->Finalize();
        nw::demo::KeyboardMouseDeviceWin::GetInstance()->Finalize();
        nw::ut::SafeFree( m_Pad, m_Arg.allocator );
        nw::ut::SafeFree( m_WPad, m_Arg.allocator );
        nw::ut::SafeFree( m_Mouse, m_Arg.allocator );
    }

    // デモファイルシステムの終了処理を行います。
    if ( m_Arg.fileDeviceManagerInitialize )
    {
        nw::dev::FileDeviceManager::GetInstance()->Finalize();
    }

    OS_SetWindowMessageCallback( NULL );
}

//------------------------------------------------------------------------------
void
SystemWinGL::InitializeGraphicsSystem( int, char** )
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    // グラフィックスデバイス初期化.
    {
        HDC hdc = GetDC( reinterpret_cast<HWND>( m_HWnd ) );

        // PIXELFORMATDESCRIPTOR準備.
        s32 pixelformat;
        PIXELFORMATDESCRIPTOR pfd;

        ::std::memset( &pfd, 0, sizeof( PIXELFORMATDESCRIPTOR ) );
        pfd.nSize           = sizeof( PIXELFORMATDESCRIPTOR );
        pfd.nVersion        = 1;
        pfd.dwFlags         = PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER | PFD_GENERIC_ACCELERATED;
        pfd.iPixelType      = PFD_TYPE_RGBA;
        pfd.cColorBits      = 32;
        pfd.cDepthBits      = 24;
        pfd.cStencilBits    = 8;

        // ピクセルフォーマット取得.
        if ( ( pixelformat = ChoosePixelFormat( hdc, &pfd ) ) == 0 )
        {
            NW_ERR( "Failed ChoosePixelFormat." );
            ReleaseDC( reinterpret_cast<HWND>( m_HWnd ), hdc );
            return;
        }

        DescribePixelFormat( hdc, pixelformat, pfd.nSize, &pfd );
        if ( !SetPixelFormat( hdc, pixelformat, &pfd ) )
        {
            NW_ERR( "Failed SetPixelFormat." );
            ReleaseDC( reinterpret_cast<HWND>( m_HWnd ), hdc );
            return;
        }

        // gl生成.
        HGLRC hglrc = wglCreateContext( hdc );

        {
            wglMakeCurrent( hdc, hglrc );
            GLenum error = glewInit();
            NW_ASSERT( error == GLEW_OK );
            NW_GL_ASSERT();
            wglMakeCurrent( hdc, NULL );
        }

        //ReleaseDC( m_HWnd, hdc );

        // グラフィックスインスタンス生成.
        {
            nw::gfnd::Graphics::CreateArg arg;
            arg.hdc = hdc;
            arg.hglrc = hglrc;

            nw::gfnd::Graphics::SetInstance( new( m_Arg.allocator->Alloc( sizeof(nw::gfnd::Graphics) ) ) nw::gfnd::Graphics( arg ) );
            nw::gfnd::Graphics::GetInstance()->Initialize();

            m_HDC   = hdc;
            m_HGLRC = hglrc;

            // Swap専用のContextを生成。
            m_HGLRCForSwap = wglCreateContext(hdc);
        }

        nw::math::VEC2 frameBufferSize( static_cast<f32>( m_Arg.width ), static_cast<f32>( m_Arg.height ) );
        CreateFrameBuffer( frameBufferSize );

        if (m_Arg.isSRGBWrite)
        {
            wglMakeCurrent( hdc, hglrc );
            glEnable(GL_FRAMEBUFFER_SRGB_EXT);
            wglMakeCurrent( NULL, NULL );
        }
    }

    // PrimitiveRenderer の初期化を行います。
    if ( m_Arg.primitiveRendererInitialize )
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        nw::dev::PrimitiveRenderer::GetInstance()->Initialize( m_Arg.allocator );
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }

    // デバッグ文字列描画用行列を計算します。
    {
        f32 l     = 0.f;
        f32 r     = static_cast<f32>( m_Arg.width );
        f32 t     = 0.f;
        f32 b     = static_cast<f32>( m_Arg.height );
        f32 zNear = 0.0f;
        f32 zFar  = 1.f;
        nw::math::MTX44Ortho( &m_TextWriterProjMtx, l, r, b, t, zNear, zFar );
        nw::math::MTX34Identity( &m_TextWriterViewMtx );
    }

    // デバッグ文字描画のための DevTextWriter を初期化します。
    if ( m_Arg.fontBinary )
    {
        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            NULL,
            0,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }
    else if ( m_Arg.fontPath )
    {
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = m_Arg.fontPath;
        loadArg.allocator = m_Arg.allocator;
        loadArg.alignment = nw::font::RESOURCE_ALIGNMENT;
        m_Arg.fontBinary = fileSystem->Load( loadArg );
        m_Arg.fontBinarySize = loadArg.readSize;

        m_DevTextWriter.Initialize(
            m_Arg.fontBinary,
            m_Arg.fontBinarySize,
            NULL,
            0,
            m_Arg.allocator
        );

        m_DevTextWriter.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_TextWriterInitialized = true;
    }

    // 負荷メーターの初期化を行います。
    if ( m_Arg.drawMeter )
    {
        m_MeterFrame.SetName( "Frame" );
        m_MeterFrame.SetColor( nw::ut::Color4u8( DEMO_COLOR_CPU_FRAME ) );
        m_MeterCalc.SetName( "Calc" );
        m_MeterCalc.SetColor( ut::Color4u8( DEMO_COLOR_CPU_CALC ) );
        m_MeterDraw.SetName( "Draw" );
        m_MeterDraw.SetColor( ut::Color4u8( DEMO_COLOR_CPU_DRAW ) );
        m_MeterGPU.SetName( "GPU" );
        m_MeterGPU.SetColor( ut::Color4u8( DEMO_COLOR_GPU ) );
        m_MeterGPU.Initialize();

        m_MeterDrawer.AttachLoadMeter( &m_MeterCalc );
        m_MeterDrawer.AttachLoadMeter( &m_MeterDraw );
        m_MeterDrawer.AttachLoadMeter( &m_MeterGPU );

        m_MeterDrawer.SetMatrix( m_TextWriterProjMtx, m_TextWriterViewMtx );
        m_MeterDrawer.SetTextWriter( m_TextWriterInitialized ? &m_DevTextWriter : NULL );
        m_MeterDrawer.SetFrameRate( m_Arg.waitVBlank > 0 ? 60.f / m_Arg.waitVBlank : m_Arg.fps );
        m_MeterDrawer.SetMinimizeBarSize( 0.f );

        // 負荷メーターの位置、大きさを調整します。
        {
            const f32 METER_WIDTH = m_Arg.width - 10.f;
            const f32 METER_HEIGHT = m_MeterDrawer.GetAdjustedHeight();

            m_MeterDrawer.SetWidth( METER_WIDTH );
            m_MeterDrawer.SetHeight( METER_HEIGHT );
            m_MeterDrawer.SetPosition( nw::math::VEC2( 5.f, m_Arg.height - ( METER_HEIGHT + 5.f ) ) );
        }
    }

    // ポインタカーソル用テクスチャ画像を生成します。
    if ( m_Arg.usePointerCursor )
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        m_PointerTextureImage = nw::dev::CreatePointerTexture( m_Arg.allocator, nw::dev::POINTER_ARROW, &m_PointerTexture );
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }

    if ( GetDisplayState() == HIDE )
    {
        m_DisplayState = READY;
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::FinalizeGraphicsSystem()
{
    NW_ASSERT_NOT_NULL( m_Arg.allocator );

    if ( m_Arg.usePointerCursor )
    {
        m_Arg.allocator->Free( m_PointerTextureImage );
    }

    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Finalize();
    }

    if ( m_Arg.fontPath && m_Arg.fontBinary )
    {
        m_Arg.allocator->Free( m_Arg.fontBinary );
    }

    if ( m_Arg.primitiveRendererInitialize )
    {
        nw::dev::PrimitiveRenderer::GetInstance()->Finalize( m_Arg.allocator );
    }

    if ( m_DefaultFrameBuffer )
    {
        m_DefaultFrameBuffer->~TextureFrameBufferGL();
        nw::ut::SafeFree( m_DefaultFrameBuffer, m_Arg.allocator );
    }

    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();
    nw::ut::SafeFree( gfx, m_Arg.allocator );
}

//------------------------------------------------------------------------------
SystemWinGL&
SystemWinGL::SetWindowTitle(const char *title)
{
    ::SetWindowTextA(nw::internal::winext::ghFrameWnd, title);
    return *this;
}

//------------------------------------------------------------------------------
void
SystemWinGL::SetFps( f32 fps )
{
    // 1フレーム相当の時間をあらかじめ計算しておく.
    m_FrameTime = nw::ut::TimeSpan::FromMicroSeconds( static_cast<s64>( 1000000 / fps ) );
    m_Arg.fps = fps;
}

//------------------------------------------------------------------------------
bool
SystemWinGL::SetWaitVBlankInterval(u32 interval)
{
    bool vsync_ok = nw::gfnd::Graphics::GetInstance()->SetVBlankWaitInterval( interval );

    if ( !vsync_ok )
    {
        return false;
    }
    else
    {
        m_Arg.waitVBlank = interval;
        return true;
    }
}

//------------------------------------------------------------------------------
f32
SystemWinGL::CalcFps()
{
    if ( m_LastDiffTime.GetNanoSeconds() == 0 )
    {
        return 0.f;
    }
    else
    {
        return m_FrameTime.GetNanoSeconds() * m_Arg.fps / m_LastDiffTime.GetNanoSeconds();
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::SetCaption( const char16* caption )
{
    SetWindowText( reinterpret_cast<HWND>( m_HWnd ), reinterpret_cast<const TCHAR*>( caption ) );
}

//------------------------------------------------------------------------------
bool
SystemWinGL::SetProcessPriority( ProcessPriority priority )
{
    switch( priority )
    {
    case IDLE:
        SetPriorityClass( GetCurrentProcess(), IDLE_PRIORITY_CLASS );
        break;

    case NORMAL:
        SetPriorityClass( GetCurrentProcess(), NORMAL_PRIORITY_CLASS );
        break;

    case HIGH:
        SetPriorityClass( GetCurrentProcess(), HIGH_PRIORITY_CLASS );
        break;

    case REAL_TIME:
        SetPriorityClass( GetCurrentProcess(), REALTIME_PRIORITY_CLASS );
        break;

    default:
        NW_ERR( "Undefined ProcessPriority.\n" );
        return false;
    }

    return true;
}

//------------------------------------------------------------------------------
void
SystemWinGL::SetViewport( f32 x, f32 y, f32 width, f32 height, f32 nearZ, f32 farZ )
{
    NW_UNUSED_VARIABLE(x);
    NW_UNUSED_VARIABLE(y);

    const FrameBuffer* frameBuffer = FrameBuffer::GetBoundFrameBuffer();
    NW_ASSERT_NOT_NULL( frameBuffer );

    nw::gfnd::Graphics::GetInstance()->SetViewport(
        x,
        y,
        width,
        height,
        nearZ,
        farZ,
        frameBuffer->GetSize().x,
        frameBuffer->GetSize().y
    );
}

//------------------------------------------------------------------------------
void
SystemWinGL::SetScissor( f32 x, f32 y, f32 width, f32 height )
{
    NW_UNUSED_VARIABLE(x);
    NW_UNUSED_VARIABLE(y);

    const FrameBuffer* frameBuffer = FrameBuffer::GetBoundFrameBuffer();
    NW_ASSERT_NOT_NULL( frameBuffer );

    nw::gfnd::Graphics::GetInstance()->SetScissor(
        x,
        y,
        width,
        height,
        frameBuffer->GetSize().x,
        frameBuffer->GetSize().y
    );
}

//------------------------------------------------------------------------------
void
SystemWinGL::ClearFrameBuffers()
{
    // デフォルトFBクリア.
    if ( m_DefaultFrameBuffer != NULL )
    {
        m_DefaultFrameBuffer->Bind();

        SetScissor( 0.f, 0.f, m_DefaultFrameBuffer->GetSize().x, m_DefaultFrameBuffer->GetSize().x );

        ClearFrameBuffersDetail( CLEAR_FLAG_COLOR | CLEAR_FLAG_DEPTH | CLEAR_FLAG_STENCIL, m_Arg.clearColor, m_Arg.clearDepth, 0 );
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::ClearFrameBuffersDetail( u8 flag, const nw::ut::Color4u8& color, f32 depth, u32 stencil )
{
    GLbitfield mask = 0;
    if ( flag & CLEAR_FLAG_COLOR )
    {
        mask |= GL_COLOR_BUFFER_BIT;
    }
    if ( flag & CLEAR_FLAG_DEPTH )
    {
        mask |= GL_DEPTH_BUFFER_BIT;
        glDepthMask( GL_TRUE );
    }
    if ( flag & CLEAR_FLAG_STENCIL )
    {
        mask |= GL_STENCIL_BUFFER_BIT;
    }

    nw::ut::Color4f fColor = static_cast<nw::ut::Color4f>( color );
    glClearColor( fColor.r, fColor.g, fColor.b, fColor.a );

    NW_GL_ASSERT();
    glClearDepth( depth );
    NW_GL_ASSERT();
    glClearStencil( stencil );
    NW_GL_ASSERT();
    glClear( mask );
    NW_GL_ASSERT();
}

//------------------------------------------------------------------------------
void
SystemWinGL::SwapBuffer()
{
    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

    gfx->LockDrawContext();
    {
        const TextureFrameBufferGL* frameBuffer = m_DefaultFrameBuffer;

        // 差し込まれたものがあれば、優先でそれを使う
        if ( m_CopyReservedFrameBuffer != NULL )
        {
            frameBuffer = m_CopyReservedFrameBuffer;
        }

        if ( frameBuffer != NULL )
        {
            frameBuffer->CopyToDisplayBuffer( &m_DisplayBuffer );
        }

        m_DisplayBuffer.CopyToDisplay();
    }

    gfx->UnlockDrawContext();

    // 転送.
    // 特定のGPU(ATI等)でContextを指定しておかないとSwapできないときがあるため,
    // Swap専用のContextをActiveにする。
    wglMakeCurrent(reinterpret_cast<HDC>(m_HDC),reinterpret_cast<HGLRC>(m_HGLRCForSwap));
        SwapBuffers( reinterpret_cast<HDC>( m_HDC ) );
    wglMakeCurrent(reinterpret_cast<HDC>(m_HDC),NULL);

    // 表示開始.
    if ( GetDisplayState() == READY )
    {
        RECT workAreaRect;
        SystemParametersInfo( SPI_GETWORKAREA, 0, &workAreaRect, 0 );

        RECT commandWindowRect;
        GetWindowRect( nw::internal::winext::ghCommandWnd, &commandWindowRect );

        // ウィンドウ位置を調整
        SetWindowPos(
            nw::internal::winext::ghCommandWnd,
            NULL,
            workAreaRect.right - ( commandWindowRect.right - commandWindowRect.left ),
            workAreaRect.bottom - ( commandWindowRect.bottom - commandWindowRect.top ),
            0,
            0,
            SWP_NOSIZE);

        SetWindowPos(
            reinterpret_cast<HWND>( m_HWnd ),
            HWND_TOP,
            workAreaRect.left,
            workAreaRect.top,
            0,
            0,
            SWP_NOSIZE | SWP_SHOWWINDOW);

        m_DisplayState = SHOW;
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::WaitForVBlank()
{
    // VBlank 待ちの間、メッセージ処理を行います。
    for(;;)
    {
        MSG msg;
        if( PeekMessage(&msg, NULL, 0, 0, PM_REMOVE) )
        {
            // 処理するメッセージがあるときはそれを優先.
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
        else
        {
            // メッセージがないときは、経過時間を見て判断.
            // vsyncが有効な時は時間を無視してフレーム処理.
            nw::ut::TimeSpan dt = nw::ut::Tick::GetSystemCurrent() - m_LastUpdateTime;
            if ( m_Arg.waitVBlank > 0 || dt > m_FrameTime )
            {
                m_LastDiffTime = dt;
                m_LastUpdateTime = nw::ut::Tick::GetSystemCurrent();

                nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

                // Windowsでは、次のlockDrawContextで初めてVBlank待ちが行われる模様.
                gfx->LockDrawContext();
                gfx->UnlockDrawContext();

                Sleep( 0 );

                break;
            }
            else if ( dt + nw::ut::TimeSpan::FromMilliSeconds( 10 ) > m_FrameTime )
            {
                // 残り10.0msec以下のときはSleep(0).
                Sleep( 0 );
            }
            else
            {
                // 残り10.0msec以上ならSleep(1)で下位スレッドに処理を回す.
                timeBeginPeriod( 1 );
                Sleep( 1 );
                timeEndPeriod( 1 );
            }
        }
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::BeginFrame()
{
    m_MeterFrame.BeginMeasure();
}

//------------------------------------------------------------------------------
void
SystemWinGL::EndFrame()
{
    ClearTextWriter();
    m_MeterFrame.EndMeasure();
    m_MeterDrawer.EndFrame();
}

//------------------------------------------------------------------------------
void
SystemWinGL::UpdatePad()
{
    nw::demo::KeyboardMouseDeviceWin::GetInstance()->Update();
    nw::demo::JoyPadDeviceWin::GetInstance()->Update();

    m_Pad->Update();
    m_WPad->Update();
    m_Mouse->Update();
}

//------------------------------------------------------------------------------
void
SystemWinGL::DrawLoadMeters()
{
    if( m_Arg.drawMeter )
    {
        m_MeterDrawer.Draw();
    }
}

//------------------------------------------------------------------------------
void
SystemWinGL::ClearTextWriter()
{
    if ( m_TextWriterInitialized )
    {
        m_DevTextWriter.Clear();
    }

}

//------------------------------------------------------------------------------
void
SystemWinGL::DrawPointerCursor( f32 posX, f32 posY, f32 width, f32 height )
{
    if ( m_Arg.usePointerCursor )
    {
        static nw::math::MTX44 projMtx;
        static nw::math::MTX34 viewMtx;

        const f32 NEAR_CLIP = 0.0f;
        const f32 FAR_CLIP  = 1.0f;

        nw::math::MTX44Ortho(
            &projMtx,
            -width / 2.0f,  // left
             width / 2.0f,  // right
             height / 2.0f, // bottom
            -height / 2.0f, // top
            NEAR_CLIP,      // near
            FAR_CLIP        // far
        );
        nw::math::MTX34Identity( &viewMtx );


        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->SetBlendEnable( true );
        graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );

        nw::dev::PrimitiveRenderer* renderer = nw::dev::PrimitiveRenderer::GetInstance();
        renderer->SetProjectionMtx( &projMtx );
        renderer->SetViewMtx( &viewMtx );

        renderer->Begin();
        {
            renderer->SetModelMatrix( nw::math::MTX34::Identity() );

            nw::dev::PrimitiveRenderer::QuadArg arg;
            arg.SetCornerAndSize(
                posX * width / 2.0f,
                posY * height / 2.0f,
                0.0f,
                static_cast<f32>( nw::dev::POINTER_TEXTURE_SIZE ),
                static_cast<f32>( nw::dev::POINTER_TEXTURE_SIZE )
            );
            arg.SetColor( nw::ut::Color4u8(nw::ut::Color4u8::WHITE) );
            renderer->DrawQuad( m_PointerTexture, arg );
        }
        renderer->End();
    }
}


//------------------------------------------------------------------------------
void
SystemWinGL::SetupWindow()
{
    // PcSDK で作成されるウィンドウへの設定です。
    {
        m_HWnd = nw::internal::winext::ghFrameWnd;
        OS_SetWindowMessageCallback( reinterpret_cast<WindowMessageCallback>( SystemWinGL::MsgProc ) );

        DWORD style = WS_OVERLAPPED | WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX | WS_CLIPCHILDREN;
        RECT rect;
        s32 width;
        s32 height;

        SetRect( &rect, 0, 0, m_Arg.width, m_Arg.height );
        AdjustWindowRect( &rect, style, FALSE );

        width = rect.right - rect.left;
        height = rect.bottom - rect.top;

        SetWindowPos( reinterpret_cast<HWND>( m_HWnd ), HWND_TOP, 0, 0, width, height, SWP_NOMOVE );
    }

    // フレーム管理初期化。
    {
        SetFps( m_Arg.fps );

        m_LastUpdateTime = nw::ut::Tick::GetSystemCurrent();
        m_LastUpdateTime -= nw::ut::Tick(m_FrameTime);
    }
}

//------------------------------------------------------------------------------
_W64 long
SystemWinGL::MsgProc( void* hWnd, unsigned int msg, _W64 unsigned wParam, _W64 long lParam, BOOL *handled )
{
    NW_ASSERT_NOT_NULL( s_Instance );

    return s_Instance->MsgProcImpl( hWnd, msg, wParam, lParam, handled );
}

//------------------------------------------------------------------------------
_W64 long
SystemWinGL::MsgProcImpl( void* hWnd, unsigned int msg, _W64 unsigned int wParam, _W64 long lParam, BOOL *handled )
{
    if ( m_MsgProcCallback )
    {
        LRESULT result = reinterpret_cast<WNDPROC>( m_MsgProcCallback )(
                            reinterpret_cast<HWND>( hWnd ),
                            static_cast<UINT>( msg ),
                            static_cast<WPARAM>( wParam ),
                            static_cast<LPARAM>( lParam )
                      );
        if ( result != 0 )
        {
            *handled = TRUE;
            return result;
        }
    }

    switch ( msg )
    {
    case WM_CREATE:
        break;

    case WM_CLOSE:
        // ×ボタン押下.
        m_Exit = true;
        *handled = TRUE;
        return 1;

    case WM_CHAR:
        if ( wParam == 0x11 )
        {
            // Ctrl + Q キー押下.
            m_Exit = true;
            *handled = TRUE;
            return 1;
        }
        break;

    case WM_SYSKEYDOWN:
        *handled = TRUE;
        return 1;
    }

    return 0;
}

//------------------------------------------------------------------------------
void
SystemWinGL::CreateFrameBuffer( const nw::math::VEC2& frameBufferSize )
{
    // デフォルトのフレームバッファを設定します.
    nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();

    gfx->LockDrawContext();
    {
        int width = m_Arg.width;
        int height = m_Arg.height;

        if ( m_Arg.createDefaultFrameBuffer )
        {
            m_DefaultFrameBuffer = new( m_Arg.allocator->Alloc( sizeof( TextureFrameBufferGL ) ) ) TextureFrameBufferGL( frameBufferSize );

            m_Color.CreateTextureImage2D(
                0,
                GL_RGBA8,
                width,
                height,
                0,
                GL_RGBA,
                GL_UNSIGNED_BYTE,
                NULL
            );

            m_Depth.CreateTextureImage2D(
                0,
                GL_DEPTH_COMPONENT32,
                static_cast<GLsizei>( width ),
                static_cast<GLsizei>( height ),
                0,
                GL_DEPTH_COMPONENT,
                GL_UNSIGNED_BYTE,
                NULL
            );

            m_DefaultFrameBuffer->Setup( &m_Color, &m_Depth );

            NW_ASSERT( m_DefaultFrameBuffer );
            m_DefaultFrameBuffer->Bind();
        }

        m_DisplayBuffer.Initialize( static_cast< f32 >( m_Arg.width ), static_cast< f32 >( m_Arg.height ) );

        // 初期設定.
        bool vsync_ok = SetWaitVBlankInterval( m_Arg.waitVBlank );
        if ( !vsync_ok )
        {
            m_Arg.waitVBlank = 0;
        }
        NW_GL_ASSERT();

    }
    gfx->UnlockDrawContext();
}

} // namespace nw::demo
} // namespace nw
