﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEMO_TEST_UTILITY_H_
#define NW_DEMO_TEST_UTILITY_H_

#include <nw/demo/demo_HeapAllocator.h>
#include <nw/dev/dev_Pad.h>

namespace nw
{
namespace demo
{

//---------------------------------------------------------------------------
//! @brief   デモのデバッグに使用するユーティリティ関数をまとめたクラスです。
//---------------------------------------------------------------------------
class DebugUtility
{
public:
    //---------------------------------------------------------------------------
    //! @brief          メモリリーク検出を初期化します。
    //!
    //! @param[in]      allocator 対象のアロケータです。
    //---------------------------------------------------------------------------
    static void InitializeMemoryLeakTester(
        nw::demo::IHeapAllocator* allocator
    )
    {
        s_Allocator = allocator;
    }

    //---------------------------------------------------------------------------
    //! @brief          メモリリークの検出機能をリセットします。
    //!
    //!                 この関数が実行されると、その時点でのメモリ使用量を記憶します。
    //!                 メモリリークを検出するためには TestMemoryLeak() 関数を実行してください。
    //!
    //!                 この関数を複数回実行すると、最後の実行のみが有効になります。
    //---------------------------------------------------------------------------
    static void ResetMemoryLeakTester()
    {
        s_MemoryFreeSize = s_Allocator->GetFreeSize();
        // NW_LOG( "######### Begining Memory State\n" );
        // s_Allocator->Dump();
    }

    //---------------------------------------------------------------------------
    //! @brief          メモリリークを検出します。
    //!
    //!                 最後に ResetMemoryLeakTester() が実行されたときとメモリ使用量を比べます。
    //!                 メモリ使用量が ResetMemoryLeakTester() 実行時と一致しない場合はヒープのダンプを出力します。
    //!
    //! @return         メモリ使用量が ResetMemoryLeakTester() 実行時と一致する場合は true を、そうでない場合は false を返します。
    //---------------------------------------------------------------------------
    static bool TestMemoryLeak()
    {
        bool result = true;

        s32 mainLeakSize = s_MemoryFreeSize - s_Allocator->GetFreeSize();

        if ( mainLeakSize > 0 )
        {
            NW_LOG( "######### Memory Leak Detected !! (%d bytes)\n", mainLeakSize );
            s_Allocator->Dump();
            result = false;
        }

        return result;
    }

    //! @brief 再入テストを無効化します。
    static void DisableReentryTest(bool disable)
    {
        s_DisableReentryTest = disable;
    }

    //! @brief 再入テストが無効化されているかどうかをチェックします。
    static bool IsReentryTestDisabled()
    {
        return s_DisableReentryTest;
    }

private:
    static u32 s_MemoryFreeSize;
    static nw::demo::IHeapAllocator* s_Allocator;
    static bool s_DisableReentryTest;

};

namespace internal
{

//! @details :private
class DemoTestLoop
{
public:
    //! @brief コンストラクタです。
    explicit DemoTestLoop(nw::demo::IHeapAllocator* allocator)
      : m_Allocator( allocator ),
        m_ContinueFlag(true)
    {
#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        nw::demo::DebugUtility::InitializeMemoryLeakTester( allocator );
        nw::demo::DebugUtility::ResetMemoryLeakTester();
#endif
    }

    //! @brief コンストラクタです。
    explicit DemoTestLoop(nw::ut::MemoryAllocator* allocator)
      : m_ContinueFlag(true)
    {
        nw::demo::HeapAllocator* heapAllocator = new(m_AllocatorBuffer) nw::demo::HeapAllocator();
        heapAllocator->Initialize( allocator );
        m_Allocator = heapAllocator;

#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        nw::demo::DebugUtility::InitializeMemoryLeakTester( m_Allocator );
        nw::demo::DebugUtility::ResetMemoryLeakTester();
#endif
    }

    //! @brief ループ継続時の処理です。
    void Continue()
    {
#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        nw::demo::DebugUtility::TestMemoryLeak();
#else
        m_ContinueFlag = false;
#endif
    }

    //! @brief ループ継続判定です。
    bool IsContinuing()
    {
#if defined(NW_DEBUG_CHECK_MEMORY_LEAK)
        return ! DebugUtility::IsReentryTestDisabled();
#else
        return m_ContinueFlag;
#endif
    }

private:
    nw::demo::IHeapAllocator* m_Allocator;
    u32  m_AllocatorBuffer[ sizeof(nw::demo::HeapAllocator) / sizeof(u32) ];
    bool m_ContinueFlag;

};

} // namespace internal


//============================================================================
//! @name テストループマクロ
//@{

#define NW_DEMO_TEST_BLOCK(allocator)                              \
    for (                                                          \
        nw::demo::internal::DemoTestLoop demoTestLoop(allocator);  \
        demoTestLoop.IsContinuing();                               \
        demoTestLoop.Continue() )

//@}

} // namespace nw::demo
} // namespace nw

#endif // NW_DEMO_TEST_UTILITY_H_
