﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TutorialBaseClass.h,PageSampleNvnTutorialLibrary}
 *
 * @brief
 *  This class is shared among all tutorials in order to abstract away per-operating system
 *  details related to NVN initialization and operating system integration. Each tutorial
 *  derives from the TutorialBaseClass, and provides an implementation of the t()
 *  global singleton accessor function.
 */

#pragma once

#include <cstdint>
#include <nvn/nvn.h>
#include <nn/nn_SdkAssert.h>

class TutorialBaseClass
{
    public:
            /*
             * Each tutorial gets a initialization call once the loader entry point is available.
             */
        virtual void Init(PFNNVNBOOTSTRAPLOADERPROC loader, NVNnativeWindow nativeWindow) = 0;

            /*
             * Each tutorial gets a shutdown call when the user indicates the program should exit.
             * The input scheme to indictae the tutorial should exit is operating system dependent.
             */
        virtual void Shutdown() = 0;

            /*
             * Each tutorial receives a call each time the tutorial should redraw. The base class
             * implementation attempts to draw 60 frames per second.
             */
        virtual void Draw(uint64_t millisec) = 0;

            /*
             * Each tutorial receives a call each time the tutorial should prepare for a change in
             * the display window size.
             */
        virtual void Resize(int width, int height) = 0;
};

/*
 * Each tutorial should implement this extern to return an instance of the class it
 * derives from TutorialBaseClass. The expected pattern of implementation is:
 *
 * TutorialBaseClass* t()
 * {
 *     static DerivedTutorialClass c;
 *     return (&c);
 * }
 */
extern TutorialBaseClass* t();

/*
 * Each tutorial should call this function in nnMain() to run tutorial.
 */
extern void TutorialRun();
