﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>

#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
#include <cafe/pad.h>

// 一時利用しているNWのためにビルド構成定義をリレーする
#ifdef NN_SDK_BUILD_DEBUG
#ifndef NW_DEBUG
#define NW_DEBUG
#endif
#elif NN_SDK_BUILD_DEVELOP
#ifndef NW_DEVELOP
#define NW_DEVELOP
#endif
#elif NN_SDK_BUILD_RELEASE
#ifndef NW_RELEASE
#define NW_RELEASE
#endif
#endif
#include <nw/math.h>

#else
#include <nn/hid.h>
#include <nn/hid/hid_Keyboard.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/nn_Log.h>
#include <nn/util/util_BitPack.h>
#include <nns/nac/nac_Types.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>
#endif

namespace nns {
namespace nac {

class Pad
{
public:

    //! @brief パッドのビット番号による定義です。
    enum PadIdx
    {
        IDX_A = 0,
        IDX_B,
        IDX_X,
        IDX_Y,
        IDX_Z,
        IDX_HOME,
        IDX_START,
        IDX_SELECT,
        IDX_L,
        IDX_R,
        IDX_UP,
        IDX_DOWN,
        IDX_LEFT,
        IDX_RIGHT,
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
        IDX_LS_SHIFT,
        IDX_RS_SHIFT,
        IDX_0,
        IDX_1,
        IDX_2,
        IDX_3,
        IDX_4,
        IDX_5,
        IDX_6,
        IDX_C,
        IDX_E,
        IDX_F,
        IDX_G,
#endif
        PAD_IDX_MAX
    };

    //! @brief パッドのマスクによる定義です。
    enum PadMask
    {
        MASK_A          = 1 << IDX_A,
        MASK_B          = 1 << IDX_B,
        MASK_X          = 1 << IDX_X,
        MASK_Y          = 1 << IDX_Y,
        MASK_Z          = 1 << IDX_Z,
        MASK_HOME       = 1 << IDX_HOME,
        MASK_START      = 1 << IDX_START,
        MASK_SELECT     = 1 << IDX_SELECT,
        MASK_L          = 1 << IDX_L,
        MASK_R          = 1 << IDX_R,
        MASK_UP         = 1 << IDX_UP,
        MASK_DOWN       = 1 << IDX_DOWN,
        MASK_LEFT       = 1 << IDX_LEFT,
        MASK_RIGHT      = 1 << IDX_RIGHT,
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
        MASK_LS_SHIFT   = 1 << IDX_LS_SHIFT,
        MASK_RS_SHIFT   = 1 << IDX_RS_SHIFT,
        MASK_0          = 1 << IDX_0,
        MASK_1          = 1 << IDX_1,
        MASK_2          = 1 << IDX_2,
        MASK_3          = 1 << IDX_3,
        MASK_4          = 1 << IDX_4,
        MASK_5          = 1 << IDX_5,
        MASK_6          = 1 << IDX_6,
        MASK_C          = 1 << IDX_C,
        MASK_E          = 1 << IDX_E,
        MASK_F          = 1 << IDX_F,
        MASK_G          = 1 << IDX_G,
#endif
    };

    //! @param マスクの定義を複合させた定義です。
    enum
    {
        //! 十字キー全てのマスク
        MASK_CROSS = MASK_UP | MASK_DOWN | MASK_LEFT | MASK_RIGHT,
        //! 全てのボタンのマスク
        MASK_ALL = 0x0fffffff
    };

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Pad();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Pad() {}

    //---------------------------------------------------------------------------
    //! @brief        初期化
    //---------------------------------------------------------------------------
    bool Initialize( void* hWnd = NULL );

    //---------------------------------------------------------------------------
    //! @brief        解放処理
    //---------------------------------------------------------------------------
    bool Finalize();

    //---------------------------------------------------------------------------
    //! @brief        毎フレームごとのアップデート
    //---------------------------------------------------------------------------
    bool Update( bool isSetFocus = false );

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかが押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsHold(uint32_t mask) const;

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームに押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsTrigger(uint32_t mask) const;

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームで離されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       離されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsRelease(uint32_t mask) const;

    //---------------------------------------------------------------------------
    //! @brief        左アナログスティックのX値を取得します。
    //!
    //! @return       左アナログスティックのX値を返します。
    //---------------------------------------------------------------------------
    float GetLeftStickX();

    //---------------------------------------------------------------------------
    //! @brief        左アナログスティックのY値を取得します。
    //!
    //! @return       左アナログスティックのY値を返します。
    //---------------------------------------------------------------------------
    float GetLeftStickY();

    //---------------------------------------------------------------------------
    //! @brief        右アナログスティックのX値を取得します。
    //!
    //! @return       右アナログスティックのX値を返します。
    //---------------------------------------------------------------------------
    float GetRightStickX();

    //---------------------------------------------------------------------------
    //! @brief        右アナログスティックのY値を取得します。
    //!
    //! @return       右アナログスティックのY値を返します。
    //---------------------------------------------------------------------------
    float GetRightStickY();
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    //---------------------------------------------------------------------------
    //! @brief        パッド値を取得するかどうかの状態を設定します。
    //!
    //! @param[in]    isEnabled   パッド時を取得するかしないかを設定します。
    //---------------------------------------------------------------------------
    void SetPadState( bool isEnabled )
    {
        m_IsPadEnabled = isEnabled;
    }

    //---------------------------------------------------------------------------
    //! @brief        キーボード/パッド値を取得するかどうかをアプリケーション側からの設定します。
    //!
    //! @param[in]    isEnabled   todo
    //---------------------------------------------------------------------------
    void SetUseKeyboardForcus( bool isEnabled )
    {
        m_IsUseKeyboardForcus = isEnabled;
    }
#endif
    //---------------------------------------------------------------------------
    //! @brief        Npadからデータが来ているかどうかの状態を取得します。
    //!
    //! @return       Npadからデータが来ている時は、trueを返します。
    //---------------------------------------------------------------------------
    bool IsNpadDataAvailable()
    {
#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
        return false;
#else
        return m_IsNpadDataAvailable;
#endif
    }

private:

#if defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    //---------------------------------------------------------------------------
    //! @brief maskで指定されたbitをbの値にします。
    //!
    //! @param mask bの値にするbitを1にした値
    //! @param b    セットしたい値
    //---------------------------------------------------------------------------
    void ChangeMask(u32 mask, bool b)
    {
        if ( b )
        {
            m_Press |= mask;
        }
        else
        {
            m_Press &= ~mask;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        PADRead で取得できるスティック値を正規化します。
    //!
    //! @param[out]    out      出力するスティック値です。
    //! @param[in]     x        入力のスティック X 値です。
    //! @param[in]     y        入力のスティック Y 値です。
    //---------------------------------------------------------------------------
    void NormalizeStickValueCafe(f32* outX, f32* outY, s8 x, s8 y)
    {
        static const float max_val = 56.f;

        f32 fx, fy;

        fx = static_cast<float>(x) / max_val;
        fy = static_cast<float>(y) / max_val;

        f32 length = ::std::sqrtf(fx*fx + fy*fy);
        if (length > 1.f)
        {
            fx /= length;
            fy /= length;
        }

        *outX = fx;
        *outY = fy;
    }

    u32 m_Press;        //!< 入力値
    u32 m_PressPrev;    //!< 前回の入力値
    u32 m_Trigger;      //!< 押しトリガ入力
    u32 m_Release;      //!< 離しトリガ入力
    s32 m_Channel;      //!< 接続されているチャンネルです。

    f32 leftX, leftY;
    f32 rightX, rightY;
    PADStatus m_PADStatuses[ PAD_MAX_CONTROLLERS ];
    bool m_IsConnected; //!< 接続されているかを表します。
#else

    //! @param Npadのサポート数です。。
    enum NpadId
    {
        NPADID_NUM = 4
    };

    //---------------------------------------------------------------------------
    //! @brief        NPadの初期化
    //---------------------------------------------------------------------------
    void InitializeNPad();

    //---------------------------------------------------------------------------
    //! @brief        Npadの有効な操作形態(NpadStyleSet)を取得します。
    //---------------------------------------------------------------------------
    void GetNpadState();

    //---------------------------------------------------------------------------
    //! @brief        NPadの指定されたキーの有無を取得します。
    //!
    //! @param[in]    index       取得するキーのインデックス。
    //!
    //! @return       キーが取得されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool TestNapdStyleSet(int index );

    nn::hid::DebugPadState     m_PadState;     //!< DebugPadのステータス
    nn::util::BitPack32        m_Press;        //!< 入力値
    nn::util::BitPack32        m_PressPrev;    //!< 前回の入力値
    nn::util::BitPack32        m_Trigger;      //!< 押しトリガ入力
    nn::util::BitPack32        m_Release;      //!< 離しトリガ入力
    bool                       m_IsNpadDataAvailable;

    nn::hid::AnalogStickState  m_AnalogStickL;      //!< Npad の左コンのアナログスティックの状態です。
    nn::hid::AnalogStickState  m_AnalogStickR;      //!< Npad の右コンのアナログスティックの状態です。

    nn::hid::NpadIdType        m_NpadIds[NPADID_NUM];
    nn::hid::NpadButtonSet     m_NpadState[NPADID_NUM];
    nn::hid::NpadJoyDualState  m_NpadJoyDualState[NPADID_NUM];
    nn::hid::NpadHandheldState m_NpadHandheldState;
#endif
#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    void*                      m_HWnd;         //!< ウィンドウハンドル
    bool                       m_IsPadEnabled; //!< デバックパッド値の取得/取得しない
    bool                       m_IsUseKeyboardForcus;  //!< GetForegroundWindow()でクライアント領域の上にあるか否かを調べる。
#endif

};

}
}
