﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
/**
 * @file mm_NvnVideoRenderer.h
 * @brief Render video data using NVN
 */
#include <nn/vi.h>
#include <nvn/nvn.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/mem/mem_StandardAllocator.h>

#include "mm_MoviePlayerUtils.h"

unsigned Align(unsigned value, unsigned alignment);

struct DataSection
{
    const void *data;
    size_t size;
};

/**
 * @brief
 * GlslCompiler class for generating shader data.
 *
 */
class GlslCompiler
{
public:
    GlslCompiler();
    ~GlslCompiler();
    bool Init();
    void Destroy();
    bool CompileShader(const char *src, NVNshaderStage stage);

    //the pointers returned in DataSection are invalid after new compilations
    DataSection GetOutput() const;

private:
    GLSLCcompileObject m_Compiler;
    DataSection m_OutputData;
    bool m_Init;
};

/**
 * @brief
 * NvnShaderSerialize helper class for NVN rendering
 *
 */
class NvnShaderSerialize
{
public:
    NvnShaderSerialize();
    ~NvnShaderSerialize();
    void LoadNvnShaderMemory(const uint8_t *shaderBinary);

    DataSection GetDataSection();
    DataSection GetControlSection();
    unsigned GetScratchSize();

private:
    void *m_ShaderData;
    DataSection m_DataSection;
    DataSection m_ControlSection;
    unsigned m_ScratchSize;
};

/**
 * @brief
 * NvnMemoryPool helper class for NVN rendering
 *
 */
class NvnMemoryPool
{
public:
    NvnMemoryPool();
    ~NvnMemoryPool();
    bool Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory);
    void Destroy();
    ptrdiff_t GetNewMemoryChunkOffset(unsigned size, unsigned alignment);
    NVNmemoryPool &GetMemoryPool();

private:
    NVNmemoryPool m_MemoryPool;
    unsigned m_Size;
    void *m_pMemory;
    unsigned m_Offset;
};

/**
 * @brief
 * NvnBuffer helper class for NVN rendering
 *
 */
class NvnBuffer
{
public:
    NvnBuffer();
    ~NvnBuffer();
    bool Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size);
    void Destroy();
    NVNbufferAddress GetAddress() const;
    NVNbuffer &GetBuffer();
    void *Map();

private:
    NVNbuffer m_Buffer;
    NVNdevice *m_pDevice;
    void *m_pMappedData;
};

/**
 * @brief
 * NvnCommandBuffer helper class for NVN rendering
 *
 */
class NvnCommandBuffer
{
public:
    NvnCommandBuffer();
    ~NvnCommandBuffer();
    bool Init(class NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool);
    void Destroy();
    void Reset();
    size_t GetCommandMemoryUsed() const;
    size_t GetControlMemoryUsed() const;
    NVNcommandBuffer &GetCommandBuffer();

private:
    NVNcommandBuffer m_CommandBuffer;
    NvnDevice *m_pDevice;
    nn::mem::StandardAllocator *m_pAllocator;
    NvnMemoryPool *m_pMemoryPool;
    ptrdiff_t m_CommandOffset;
    void *m_pControlPool;

    static const size_t m_CommandSize = 1024;
    static const size_t m_ControlSize = 512;
};

/**
 * @brief
 * NvnDevice helper class for NVN rendering
 *
 */
class NvnDevice
{
public:
    NvnDevice();
    ~NvnDevice();
    bool Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator);
    void Destroy();
    void ResizeRenderTargets(unsigned width, unsigned height);
    NVNdevice &GetDevice();
    NVNwindow &GetWindow();
    NVNtexture &GetRenderTarget(int index);
    int GetDeviceInfo(NVNdeviceInfo name);

private:
    static void DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char * message,
        void* pUser);

    NVNdevice m_Device;

    nn::mem::StandardAllocator *m_pAllocator;

    NVNtextureBuilder m_RenderTargetBuilder;
    NVNtexture m_RenderTargets[2];

    NvnMemoryPool m_RenderTargetMemoryPool;
    void *m_pRenderTargetMemory;

    NVNwindow m_pWindow;
    NVNwindowBuilder m_WindowBuilder;
    size_t m_ColorTargetSize;

    bool m_FirstResize;

    static const int m_NumColorBuffers = 2;
};

/**
 * @brief
 * NvnProgram helper class for NVN rendering
 *
 */
class NvnProgram
{
public:
    NvnProgram();
    ~NvnProgram();
    bool Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount);
    void Destroy();
    NVNprogram &GetProgram();

private:
    NVNprogram m_Program;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnQueue helper class for NVN rendering
 *
 */
class NvnQueue
{
public:
    NvnQueue();
    ~NvnQueue();
    bool Init(NVNdevice &device);
    void Destroy();
    NVNqueue &GetQueue();
    NVNsync &GetNvnSync();

private:
    NVNqueue m_Queue;
    NVNsync m_CommandBufferSync;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnSampler helper class for NVN rendering
 *
 */
class NvnSampler
{
public:
    NvnSampler();
    ~NvnSampler();
    bool Init(NVNdevice &device);
    void Destroy();
    NVNsampler &GetSampler();

private:
    NVNsampler m_Sampler;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnTexture helper class for NVN rendering
 *
 */
class NvnTexture
{
public:
    NvnTexture();
    ~NvnTexture();
    bool Init(NVNdevice &device,
        NvnMemoryPool &memoryPool,
        unsigned width,
        unsigned height,
        NVNformat format,
        int32_t flags,
        unsigned stride,
        bool nvnVideoInteropMode = false);
    void Destroy();
    NVNtexture &GetTexture();

private:
    NVNtexture m_Texture;
    NVNdevice *m_pDevice;
};

/**
 * @brief
 * NvnTextureSamplerPool helper class for NVN rendering
 *
 */
class NvnTextureSamplerPool
{
public:
    NvnTextureSamplerPool();
    ~NvnTextureSamplerPool();
    bool Init(NVNdevice &device, nn::mem::StandardAllocator &allocator);
    void Destroy();
    NVNtexturePool &GetTexturePool();
    NVNsamplerPool &GetSamplerPool();
    unsigned Register(NVNtexture &pTexture, NVNsampler &pSampler);

private:
    NVNtexturePool m_TexturePool;
    NVNsamplerPool m_SamplerPool;
    NvnMemoryPool m_MemoryPool;
    void *m_pMemory;
    nn::mem::StandardAllocator *m_pAllocator;
    NVNdevice *m_pDevice;
    unsigned m_Id;
};

/**
 * @brief
 * NvnWindow helper class for NVN rendering
 *
 */
class NvnWindow
{
public:
    NvnWindow();
    ~NvnWindow();
    bool Init(nn::vi::NativeWindowHandle nativeWindow);
    void Destroy();
    nn::vi::NativeWindowHandle GetNativeWindow();
    int GetWidth() const;
    int GetHeight() const;

private:
    void InitMemory();
    bool InitWindow();

    nn::vi::NativeWindowHandle  m_NativeWindow;
    nn::vi::Display*            m_pDisplay;
    nn::vi::Layer*              m_pLayer;
    int                         m_WindowWidth;
    int                         m_WindowHeight;
    bool m_NativeWindowCreated;
};

/**
 * @brief
 * NvnVideoRenderer class for renderig of video using NVN
 *
 */
class NvnVideoRenderer
{
public:

   /**
    * @brief
    * Enumeration of Output formats used by NVN renderer.
    *
    * @details
    */
    enum NvnOutputFormat
    {
        NvnOutputFormat_YuvNv12NvnBuffer  = 0,  //!< Video output format - NV12.
        NvnOutputFormat_AbgrNvnTexture    = 1,  //!< Video output format - ABGR (8:8:8:8 bit).
                                                //!< NVN texture direct access by decoder.
    };

   /**
    * @brief
    * NvnVideoRenderer constructor.
    *
    * @param[in] nativeWindow                   VI NativeWindowHandle to use.
    * @param[in] nvnOutputFormat                NvnOutputFormat, one of the following:
    *                                           NvnOutputFormat_YuvNv12NvnBuffer or NvnOutputFormat_AbgrNvnTexture.
    *
    */
    NN_IMPLICIT NvnVideoRenderer(nn::vi::NativeWindowHandle nativeWindow,
        NvnOutputFormat nvnOutputFormat = NvnOutputFormat_YuvNv12NvnBuffer) NN_IMPLICIT;

   /**
    * @brief
    * Initialize NvnVideoRenderer.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Generic error.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    * @retval ::Status_NotSupported             Unsupported NvnOutputFormat.
    *
    * @details
    *
    */
    movie::Status Initialize();

   /**
    * @brief
    * ResizeTextures.
    *
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Generic error.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * Resizes NVN texture if video height and width are changed.
    *
    */
    movie::Status ResizeTextures(int32_t width, int32_t height);

   /**
    * @brief
    * GetNvnVideoBuffer.
    *
    * @param[in] index                          Video buffer index.
    * @param[out] bufferMemory                  Video buffer data pointer.
    * @param[out] yuvBufferSize                 Video buffer size in bytes.
    *
    * @return None
    *
    * @details
    * Returns a buffer, which can be used to copy video decoder output data.
    *
    */
    void GetNvnVideoBuffer(int index, char **bufferMemory, size_t *yuvBufferSize);

   /**
    * @brief
    * GetNvnVideoTexture.
    *
    * @param[in] index                          Texture index.
    * @param[out] textureHandle                 NVN texture handle.
    * @param[out] size                          NVN texture size.
    *
    * @return None
    *
    * @details
    * Returns NVN texture handle, which can be used by video decoder. This API can be used only if
    * output format is set to NvnOutputFormat_AbgrNvnTexture.
    *
    */
    void GetNvnVideoTexture(int index, void **textureHandle, size_t *size);

   /**
    * @brief GetNvnVideoSync.
    *
    * @param[out] deviceHandle                  NVN device handle.
    * @param[out] syncHandle                    NVN sync handle.
    *
    * @return None
    *
    * @details
    * Returns NVN device & sync handle, which can be used by video decoder for synchronization between modules.
    *
    */
    void GetNvnVideoSync(void **deviceHandle, void **syncHandle);
   /**
    * @brief Draw.
    *
    * @param[in] width                          Video width (Number of distinct vertical lines).
    * @param[in] height                         Video height (Number of distinct horizontal lines).
    * @param[in] yOffset                        Offset for Y-data from start of the output buffer address, when output format is NV12
    * @param[in] yStride                        Stride for Y-rows, when output format is NV12.
    * @param[in] uvOffset                       Offset for UV-data from start of the output buffer address, when output format is NV12.
    * @param[in] colorSpace                     Color space for UV-data, when output format is NV12.
    *
    * @return None
    *
    * @details
    * Draw current video frame using NVN.
    *
    */
    void Draw(int32_t width, int32_t height, int32_t yOffset, int32_t yStride, int32_t uvOffset, int32_t colorSpace);

   /**
    * @brief
    * Finalize NvnVideoRenderer.
    *
    * @return None
    *
    * @details
    * All resources are freed.
    *
    */
    void Finalize();

   /**
    * @brief
    * GetDevice.
    *
    * @return NvnDevice
    *
    * @details
    * All resources are freed.
    *
    */
    NvnDevice* GetDevice() { return &m_device; }

private:
    nn::mem::StandardAllocator m_allocator;
    void *m_pAllocatorMemory;

    NvnWindow m_window;
    NvnDevice m_device;
    NvnQueue m_queue;

    void *m_pShaderPoolMemory;
    void *m_pVertexPoolMemory;
    void *m_pCommandMemory;
    void *m_pTextureMemory;
    void *m_pBufferMemory;
    void *m_pShaderScratchMemory;
    void *m_pUniformMemory;
    NvnMemoryPool m_shaderPool;
    NvnMemoryPool m_vertexPool;
    NvnMemoryPool m_commandPool;
    NvnMemoryPool m_textureMemoryPool;
    NvnMemoryPool m_bufferMemory;
    NvnMemoryPool m_shaderScratchPool;
    NvnMemoryPool m_uniformPool;

    NvnProgram m_program;
    NvnShaderSerialize m_vertexShaderFile;
    NvnShaderSerialize m_fragmentShaderFile;

    NvnCommandBuffer m_drawCommands;
    NvnCommandBuffer m_renderTargetCommand;

    NvnTextureSamplerPool m_texturePool;
    NvnTexture m_videoTextureY;
    NvnTexture m_videoTextureUV;
    NvnSampler m_videoSampler;
    NVNtextureHandle m_videoTextureHandleY;
    NVNtextureHandle m_videoTextureHandleUV;
    unsigned m_videoTextureIdY;
    unsigned m_videoTextureIdUV;
    NvnBuffer m_vertexDataBuffer;
    NvnBuffer m_fragmentDataBuffer;
    NvnBuffer m_squareMesh;
    NvnBuffer m_uniformBuffer;
    unsigned m_uniformUsefulSize;

    NvnBuffer m_videoBufferYUV;
    char *m_VideoOutputBuffer;

    NVNblendState m_blendState;
    NVNchannelMaskState m_channelMaskState;
    NVNcolorState m_colorState;
    NVNmultisampleState m_multisampleState;
    NVNpolygonState m_polygonState;
    NVNdepthStencilState m_depthStencilState;

    DataSection m_vertexDataSection;
    DataSection m_vertexControlSection;
    DataSection m_fragmentDataSection;
    DataSection m_fragmentControlSection;

    NVNcommandHandle m_commandHandleDraw;
    size_t m_YuvBufferSize;
    nn::vi::NativeWindowHandle m_NativeWindow;
    NvnOutputFormat m_NvnOutputFormat;
    NvnTexture m_RgbTexture;
    NVNtextureHandle m_RgbTextureHandle;
    unsigned m_RgbTextureId;
};
