﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file mm_MovieAudioOutputHandler.h
 * @brief Handle audio decoded data from movie::decoder
 */
#include "mm_MoviePlayerUtils.h"
#include "mm_AudioOutRenderer.h"

/**
 * @brief
 * MovieAudioOutputHandler class for processing audio data from movie::decoder
 *
 */
void MovieAudioOutputHandlerThreadFunction(void *arg);
class MovieAudioOutputHandler
{
public:
   /**
    * @brief
    * MovieAudioOutputHandler constructor.
    */
    MovieAudioOutputHandler();

   /**
    * @brief
    * MovieAudioOutputHandler destructor.
    *
    */
    ~MovieAudioOutputHandler();

   /**
    * @brief
    * Initialize MovieAudioOutputHandler.
    *
    * @param[in] mediaClock                     Clock component to do audio-video sync.
    * @param[in] audioRenderComplete            Client event to be signaled when rendering is complete.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to create thread.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * An instance of AudioOutRenderer is created and is initialized. MovieAudioOutputHandler main processing
    * thread is created, resources are initialized.
    *
    */
    movie::Status Initialize(MediaClock *mediaClock, nn::os::EventType *audioRenderComplete=nullptr);

   /**
    * @brief
    * Finalize MovieAudioOutputHandler.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * When Finalize() API is called, MovieAudioOutputHandler can be initialized again using Initialize().
    * MovieAudioOutputHandler main thread will be destroyed and all lists are cleared. AudioOutRenderer
    * instance is released.
    *
    */
    movie::Status Finalize();

   /**
    * @brief
    * Open MovieAudioOutputHandler.
    *
    * @param[in] audioDecoder                   movie::Decoder (Audio) to use
    * @param[in] sampleRate                     Audio sample rate in hertz.
    * @param[in] channelCount                   Number of audio channels.
    * @param[in] variableSizeAudioFrames        Does decoder produces variable size audio frames.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    * @retval ::Status_OutOfMemory              Memory allocation failed.
    *
    * @details
    * When Open() API is called, AudioOutRenderer is opened. Resources including buffers are allocated.
    *
    */
    movie::Status Open(movie::Decoder* audioDecoder, uint32_t sampleRate, int channelCount, bool variableSizeAudioFrames);

   /**
    * @brief
    * Close MovieAudioOutputHandler and release resources.
    *
    * @return none
    *
    * @details
    * When Close() API is called, AudioOutRenderer is closed. Audio decoder buffers are returned back to the decoder.
    * Resources allocated by MovieAudioOutputHandler are released.
    *
    */
    void Close();

   /**
    * @brief
    * Start processing data from movie::Decoder.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             Failed to start playback.
    *
    * @details
    * When Start() API is called, AudioOutRenderer is started. MovieAudioOutputHandler main thread is started.
    * The thread loop will call Render() API. This will result in getting audio data from decoder, preparing a fixed
    * size frame if needed, getting an empty buffer from  AudioOutRenderer and submitting audio buffer for rendering.
    *
    */
    movie::Status Start();

   /**
    * @brief
    * Stop processing audio data from movie::Decoder
    *
    * @return none
    *
    * @details
    * When Stop() API is called, AudioOutRenderer is stopped. Any decoder buffers with MovieAudioOutputHandler are
    * released back to the decoder.
    *
    */
    void Stop();

   /**
    * @brief
    * Pause processing audio data from movie::Decoder
    *
    * @return none
    *
    * @details
    * When Pause() API is called, MovieAudioOutputHandler pauses processing of audio. The main thread will stop calling
    * Render() API. This will result in AudioOutRenderer to render zero (silence) samples. Since output buffers are not
    * pulled from decoder, it will pause. Client can call Resume() API to starting processing audio data again. This
    * API can be used by client to implement trick mode playback.
    *
    */
    void Pause();

   /**
    * @brief
    * Resume processing audio data from movie::Decoder
    *
    * @return none
    *
    * @details
    * When Resume() API is called, MovieAudioOutputHandler restarts processing of audio. The main thread will start
    * calling Render() API. This API can be used by client to implement trick mode playback.
    *
    */
    void Resume();

   /**
    * @brief
    * Flush movie::Decoder (audio)
    *
    * @return none
    *
    * @details
    * When Flush() API is called, MovieAudioOutputHandler clears output buffer lists. All output buffers received from
    * the movie decoder are returned back. This API can be used to implement trick mode playback like seeking a media
    * file. While implementing seek, flush is needed to discard any old audio buffers.
    *
    */
    void Flush();

   /**
    * @brief
    * Render audio
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_UnknownError             No data to render.
    *
    * @details
    * Render() API is called from MovieAudioOutputHandler thread loop. This will check for availability of filled PCM
    * buffers. If there is a filled PCM buffer, it will try to get an empty audio buffer from AudioRenderer.
    * If an empty buffer is available and if needed, mono to dual channel conversion is done. Resampling may be
    * done if sample rate is not supported. Media clock anchor time is updated. The filled buffer is submitted for
    * rendering. The buffer is moved back to empty list. If the buffer has end of stream flags and if an event is
    * registered, it will be signaled.
    *
    */
    movie::Status Render();

   /**
    * @brief
    * Generate fixed size audio frame
    *
    * @return None
    *
    * @details
    * Generates fixed size audio frame by combing multiple frames. Some decoders like Vorbis generate audio frames
    * with varying sizes. If decoder buffer has more data than the defined fixed frame size, it will use data needed
    * to generate the fixed frame. The decoder output buffer is updated with remaining data size. The time stamp of the
    * audio buffer is updated. The buffer is not released back to the decoder until all data is consumed.
    *
    */
    void GenerateFixedSizeAudioFrame();

   /**
    * @brief
    * Audio output buffer available call back.
    *
    * @param[in] index                          Output buffer index from decoder
    * @param[in] presentationTimeUs             Presentation time for this audio buffer
    * @param[in] flags                          movie:: buffer flags
    *
    * @return None
    *
    * @details
    * When audio decoder completes decoding an audio frame, it will signal output buffer available event. @class
    * MovieDecoderEventHandler receives all decoder events. This event will be routed to @class MovieAudioOutputHandler
    * through an API call. The buffer index is added to @class MovieAudioOutputHandler list. If possible, a fixed size
    * audio frame is generated.
    *
    */
    void OnOutputAvailable(int index, int64_t presentationTimeUs, uint32_t flags);

   /**
    * @brief
    * Exit thread
    *
    * @return boolean
    * @retval true  - Threads need to be terminated.
    * @retval false - Threads need to be running.
    *
    * @details
    * API to get thread loop exit condition.
    *
    */
    bool ExitThread();

   /**
    * @brief
    * IsAudioOutStarted
    *
    * @return boolean
    * @retval true  - Rendering of audio is started.
    * @retval false - Rendering of audio not started.
    *
    * @details
    * API to get AudioOut rendering start state.
    *
    */
    bool IsAudioOutStarted();

   /**
    * @brief
    * GetThreadSleepTimeMs
    *
    * @return sleepTimeMs - Sleep time in milliseconds.
    *
    * @details
    * API to get rendering threads sleep time.
    *
    */
    int64_t GetThreadSleepTimeMs();

   /**
    * @brief
    * AllFramesRendered
    *
    * @return boolean
    * @retval true  - All audio frames are rendered.
    * @retval false - All frames not rendered, more frames available for rendering.
    *
    * @details
    * API to get rendering completion condition.
    *
    */
    bool AllFramesRendered() { return m_AllFramesRendered; }

   /**
    * @brief
    * GetPlayedOutAudioDurationUs
    *
    * @param[in] nowUs                          Current time in micro seconds.
    *
    * @return playout - Playout duration in micro seconds.
    *
    * @details
    * API to get playout duration.
    *
    */
    int64_t GetPlayedOutAudioDurationUs(int64_t nowUs);

   /**
    * @brief
    * Thread function for MovieAudioOutputHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * This is the thread loop function. This thread loop will call Render() API and then will sleep.
    * The sleep duration is determined by the audio consumption rate.
    *
    */
    friend void MovieAudioOutputHandlerThreadFunction(void *arg);

   /**
    * @brief
    * IsPaused
    *
    * @return boolean
    * @retval true  - If audio rendering is paused.
    * @retval false - If audio rendering is not paused.
    *
    * @details
    * API to get pause state. In paused state @class MovieAudioOutputHandler thread loop will skip sending audio
    * data to AudioRenderer.
    *
    */
    bool IsPaused() { return m_IsPaused; }

   /**
    * @brief
    * GetPlaybackPosition
    * @param[out] playbackPositionUs             Playback position in micro seconds.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_ErrorBadValue            Invalid input parameters.
    *
    * @details
    * Returns current audio playback position in micro seconds.
    *
    */
    movie::Status GetPlaybackPosition(int64_t *playbackPositionUs) const;

   /**
    * @brief
    * CanStartRendering
    *
    * @return boolean
    * @retval true  - If audio rendering can be started.
    * @retval false - If audio rendering cannot be started(not enough audio data).
    *
    * @details
    * API to check whether audio rendering can be started. This API will return true if sufficient
    * buffers are available so that a smooth audio playback can be started.
    *
    */
    bool CanStartRendering();

private:
    nn::os::EventType *m_AudioRenderComplete;
    movie::Decoder   *m_AudioDecoder;
    AudioOutRenderer *m_AudioOutRenderer;
    bool m_ThreadStarted;
    bool m_ThreadDone;
    bool m_ThreadCreated;
    bool m_AudioOutStarted;
    bool m_AudioOutOpened;
    int m_SampleRate;
    int m_Channels;
    int m_InputSampleRate;
    int m_InputChannels;
    int m_FrameSize;
    int64_t m_PresentationTimeUs;
    size_t m_Threadstacksize;
    void *m_Threadstack;
    nn::os::ThreadType  m_ThreadType;
    size_t m_SampleSize;
    char *m_InAudioBuffer;
    int32_t m_InAudioBufferSize;
    int m_AudioDecoderBuffers;
    int m_AudioDecoderBufferSize;
    int m_FixedAudioFrameSize;
    bool m_GenerateFixedSizeAudioFrames;

    struct audioData
    {
        int index;
        int64_t presentationTimeUs;
        uint32_t flags;
    };
    std::vector<audioData> m_AudioCodecBuffers;
    std::vector<movie::Buffer> m_AudioBufferRenderEmptyList;
    std::vector<movie::Buffer> m_AudioBufferRenderFilledList;

    movie::Buffer* m_DecoderBuffer;

    int64_t m_ThreadSleepTimeMs;
    size_t m_BufferSize;
    nn::os::MutexType m_AudioBufferVectorMutex;
    bool m_AllFramesRendered;

    MediaClock *m_MediaClock;
    int64_t m_AudioFirstAnchorTimeMediaUs;
    int64_t m_AudioStartAnchorTimeUs;
    int64_t m_FramesWritten;
    bool m_IsPaused;
    bool m_ResampleAudio;
    bool m_MonoToDualChannel;
    void *m_MonoToDualChannelBuffer;
    size_t m_MonoToDualChannelBufferSize;
    int64_t m_PlaybackPositionUs;
    int64_t m_AudioOutputBuffersReceived;
};

