﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       マウスのデバイスアセットに関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/hid/hid_Mouse.h>
#include <nns/hid/hid_DeviceAsset.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

/**
 * @brief   マウスのデバイスアセットクラスです。
 */
class MouseAsset : public DeviceAsset
{
    NN_DISALLOW_COPY(MouseAsset);
    NN_DISALLOW_MOVE(MouseAsset);

public:
    /**
     * @brief       MouseAsset のコンストラクタです。
     *
     * @pre
     *              - 事前に 1 回以上 Initialize() が呼び出されている
     *
     * @param[in]   pManager                    マウスのデバイスアセットの管理者です。
     */
    explicit MouseAsset(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       MouseAsset のデストラクタです。
     */
    virtual ~MouseAsset() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       マウスを初期化します。
     */
    static void Initialize() NN_NOEXCEPT;

    /**
     * @brief       マウスのデバイスアセット識別子を返します。
     *
     * @return      マウスのデバイスアセット識別子です。
     */
    virtual DeviceAssetId GetDeviceAssetId() NN_NOEXCEPT NN_OVERRIDE
    {
        return DeviceAssetId_Mouse;
    }

    /**
     * @brief       マウスのデバイスアセットを更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       マウスの入力状態を返します。
     *
     * @return      マウスの入力状態です。
     */
    const std::vector<nn::hid::MouseState>& GetMouseStates() const NN_NOEXCEPT
    {
        return m_States;
    }

private:
    int64_t m_SamplingNumber;                   //!< マウスのサンプリング番号です。

    std::vector<nn::hid::MouseState> m_States;  //!< マウスの入力状態です。

    nn::hid::MouseState m_TemporaryStates[
        nn::hid::MouseStateCountMax];           //!< 入力状態読み出し用のバッファです。
};

}} // namespace nns::hid
