﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラのデバイスアセットに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nns/hid/hid_DeviceAsset.h>
#include <nns/hid/hid_DeviceAssetId.h>

namespace nns { namespace hid {

/**
 * @brief   コントローラのデバイスアセットクラスです。
 */
class GamePadNxAsset : public nns::hid::DeviceAsset
{
    NN_DISALLOW_COPY(GamePadNxAsset);
    NN_DISALLOW_MOVE(GamePadNxAsset);

public:
    //!< サポートするコントローラの数です
    const int ControllerCountMax = 5;

    /**
     * @brief       コントローラの入力状態の属性定義です。
     */
    typedef nn::hid::NpadJoyAttribute GamePadNxAttribute;

    /**
     * @brief       コントローラの入力状態の属性定義の集合です
     */
    typedef nn::hid::NpadAttributesSet GamePadNxAttributesSet;

    /**
     * @brief       コントローラのデジタルボタンのデジタルボタン定義です。
     */
    typedef nn::hid::NpadButton GamePadNxButton;

    /**
     * @brief       コントローラの入力状態を表す型です。
     */
    typedef nn::hid::NpadFullKeyState GamePadNxState;

    /**
     * @brief       GamePadNxAsset のコンストラクタです。
     *
     * @pre
     *              - 事前に 1 回以上 Initialize() が呼び出されている
     *
     * @param[in]   pManager                    コントローラのデバイスアセットの管理者です。
     */
    explicit GamePadNxAsset(nns::hid::ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       GamePadNxAsset のデストラクタです。
     */
    virtual ~GamePadNxAsset() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       コントローラを初期化します。
     */
    static void Initialize() NN_NOEXCEPT;

    /**
     * @brief       コントローラのデバイスアセット識別子を返します。
     *
     * @return      コントローラのデバイスアセット識別子です。
     */
    virtual nns::hid::DeviceAssetId GetDeviceAssetId() NN_NOEXCEPT NN_OVERRIDE
    {
        return nns::hid::DeviceAssetId_GamePad;
    }

    /**
     * @brief       コントローラのデバイスアセットを更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       コントローラの入力状態を取得します。
     *
     * @details     指定のコントローラ番号と対応するコントローラから入力状態を取得します
     *
     * @param[out]  pOutValue                   入力状態を読み出すバッファです。
     * @param[in]   controllerNumber            コントローラ番号です。
     *
     * @return      コントローラの入力状態の取得に成功したか否かを表す値です。
     */
    bool GetGamePadNxState(GamePadNxState* pOutValue, int controllerNumber
                         ) const NN_NOEXCEPT;

    //! @}

    //! @name   コントローラの状態の取得
    //! @{

    /**
    * @brief       コントローラの入力状態の属性定義の集合を返します。
    *
    * @return      コントローラの入力状態の属性定義の集合です。
    */
    const GamePadNxAttributesSet GetGamePadNxAttributesSet() const NN_NOEXCEPT;

private:
    GamePadNxState* m_States;  //!< 入力状態読み出し用のバッファです。
    int* m_StatesTable;             //!< controllerNumber に対応する入力状態のバッファの index を示したテーブルです
};

}} // namespace nns::hid
