﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       ゲームパッドの機能を持つコントローラに関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nns/hid/hid_Controller.h>
#include <nns/hid/hid_ControllerId.h>
#include <nns/hid/hid_ControllerManager.h>
#include <nns/hid/hid_GamePadAsset.h>

namespace nns { namespace hid {

/**
 * @brief   ゲームパッドの機能を持つコントローラクラスです。
 */
class GamePad : public Controller
{
    NN_DISALLOW_COPY(GamePad);
    NN_DISALLOW_MOVE(GamePad);

public:
    /**
     * @brief       GamePad のコンストラクタです。
     *
     * @param[in]   pManager                    ゲームパッドの管理者です。
     *
     * @pre
     *              - pManager != NULL
     */
    explicit GamePad(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       GamePad のデストラクタです。
     */
    virtual ~GamePad() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ゲームパッドのプレイヤー番号を返します。
     *
     * @return      ゲームパッドのプレイヤー番号です。
     */
    int GetPlayerNumber() const NN_NOEXCEPT
    {
        return m_PlayerNumber;
    }

    /**
     * @brief       ゲームパッドのプレイヤー番号を設定します。
     *
     * @param[in]   playerNumber                ゲームパッドのプレイヤー番号です。
     */
    void SetPlayerNumber(int playerNumber) NN_NOEXCEPT
    {
        m_PlayerNumber = playerNumber;
    }

    /**
     * @brief       ゲームパッドの状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ゲームパッドのコントローラ識別子を返します。
     *
     * @return      ゲームパッドのコントローラ識別子です。
     */
    virtual ControllerId GetControllerId() NN_NOEXCEPT NN_OVERRIDE
    {
        return ControllerId_GamePad;
    }

    /**
     * @brief       ゲームパッドが接続状態にあるか否かを返します。
     *
     * @return      ゲームパッドが接続状態にあるか否かを表す値です。
     */
    virtual bool IsConnected() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsConnected;
    }

private:
    GamePadAsset* m_pGamePadAsset;  //!< 対応するゲームパッドのデバイスアセットです。

    int m_PlayerNumber;             //!< ゲームパッドのプレイヤー番号です。

    bool m_IsConnected;             //!< ゲームパッドが接続状態にあるか否かを表す値です。
};

}} // namespace nns::hid
