﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       コントローラに関する API の宣言
 */

#pragma once

#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/util/util_MathTypes.h>
#include <nns/hid/hid_Button.h>
#include <nns/hid/hid_ControllerAddonId.h>
#include <nns/hid/hid_ControllerId.h>

namespace nns { namespace hid {

// 依存クラスの前方宣言です。
class ControllerAddon;
class ControllerManager;

/**
 * @brief   コントローラの基底クラスです。
 *
 * @details 28 個のデジタルボタン、左右のスティック、1 つのポインティングデバイスを扱うことができます。
 */
class Controller
{
    NN_DISALLOW_COPY(Controller);
    NN_DISALLOW_MOVE(Controller);

public:
    /**
     * @brief       Controller のコンストラクタです。
     *
     * @param[in]   pManager                    コントローラの管理者です。
     *
     * @pre
     *              - pManager != NULL
     */
    explicit Controller(ControllerManager* pManager) NN_NOEXCEPT;

    /**
     * @brief       Controller のデストラクタです。
     */
    virtual ~Controller() NN_NOEXCEPT;

    /**
     * @brief       コントローラの状態を更新します。
     *
     * @details     更新はフレーム毎に行う必要があります。
     */
    virtual void Update() NN_NOEXCEPT = 0;

    /**
     * @brief       コントローラ識別子を返します。
     *
     * @return      コントローラ識別子です。
     */
    virtual ControllerId GetControllerId() NN_NOEXCEPT = 0;

    /**
     * @brief       コントローラの管理者を返します。
     *
     * @return      コントローラの管理者です。
     */
    ControllerManager* GetControllerManager() const NN_NOEXCEPT
    {
        return m_pManager;
    }

    /**
     * @brief       コントローラアドオンリストを返します。
     *
     * @details     コントローラアドオンリストを返します。
     *              アドオンを追加したい場合はこのメソッドを用いてください。
     *
     *
     * @return      コントローラアドオンのリストです。
     */
    std::vector<ControllerAddon*>& GetControllerAddons() NN_NOEXCEPT
    {
         return m_ControllerAddons;
    }

    /**
     * @brief       コントローラアドオンを返します。
     *
     * @details     指定のコントローラアドオン識別子を持つコントローラアドオンを探索し、
     *              index 番目に見つかったものを返します。
     *              条件を満たすコントローラアドオンが見つからなかった場合は NULL を返します。
     *
     * @param[in]   id                          コントローラアドオン識別子です。
     * @param[in]   index                       添字の指定です。
     *
     * @return      コントローラアドオンです。
     */
    ControllerAddon* GetControllerAddon(
        ControllerAddonId id, int index) const NN_NOEXCEPT;

    //! @name   操作情報の取得
    //! @{

    /**
     * @brief       押下中のデジタルボタンを返します。
     *
     * @return      押下中のデジタルボタンです。
     */
    ButtonSet GetButtons() const NN_NOEXCEPT
    {
        return m_Buttons;
    }

    /**
     * @brief       最後の更新で新たに押下されたデジタルボタンを返します。
     *
     * @return      最後の更新で新たに押下されたデジタルボタンです。
     */
    ButtonSet GetButtonsDown() const NN_NOEXCEPT
    {
        return m_ButtonsDown;
    }

    /**
     * @brief       最後の更新で新たに開放されたデジタルボタンを返します。
     *
     * @return      最後の更新で新たに開放されたデジタルボタンです。
     */
    ButtonSet GetButtonsUp() const NN_NOEXCEPT
    {
        return m_ButtonsUp;
    }

    /**
     * @brief       リピートされたデジタルボタンを返します。
     *
     * @return      リピートされたデジタルボタンです。
     */
    ButtonSet GetButtonsRepeat() const NN_NOEXCEPT
    {
        return m_ButtonsRepeat;
    }

    /**
     * @brief       左スティックの座標を返します。
     *
     * @details     コントローラを正面から見た際の右と上をそれぞれ x 成分と y 成分の正方向として、左スティックの座標を返します。
     *              成分の値域は共に [-1.0f, 1.0f] です。
     *
     * @return      左スティックの座標です。
     */
    const nn::util::Float2& GetLeftStick() const NN_NOEXCEPT
    {
        return m_LeftStick;
    }

    /**
     * @brief       右スティックの座標を返します。
     *
     * @details     コントローラを正面から見た際の右と上をそれぞれ x 成分と y 成分の正方向として、右スティックの座標を返します。
     *              成分の値域は共に [-1.0f, 1.0f] です。
     *
     * @return      右スティックの座標です。
     */
    const nn::util::Float2& GetRightStick() const NN_NOEXCEPT
    {
        return m_RightStick;
    }

    /**
     * @brief       ポインティングデバイスの座標が有効範囲に入っているか否かを返します。
     *
     * @return      ポインティングデバイスの座標が有効範囲に入っているか否かを表す値です。
     */
    bool IsPointerOn() const NN_NOEXCEPT
    {
        return m_PointerPhase.Test<PointerPhase::On>();
    }

    /**
     * @brief       ポインティングデバイスの座標が最後の更新で有効範囲に入ったか否かを返します。
     *
     * @return      ポインティングデバイスの座標が最後の更新で有効範囲に入ったか否かを表す値です。
     */
    bool IsPointerBegan() const NN_NOEXCEPT
    {
        return m_PointerPhase.Test<PointerPhase::Began>();
    }

    /**
     * @brief       ポインティングデバイスの座標が最後の更新で有効範囲から出たか否かを返します。
     *
     * @return      ポインティングデバイスの座標が最後の更新で有効範囲から出たか否かを表す値です。
     */
    bool IsPointerEnded() const NN_NOEXCEPT
    {
        return m_PointerPhase.Test<PointerPhase::Ended>();
    }

    /**
     * @brief       ポインティングデバイスの無効な座標を返します。
     *
     * @return      ポインティングデバイスの無効な座標です。
     */
    static const nn::util::Float2& GetInvalidPointer() NN_NOEXCEPT
    {
        return InvalidPointer;
    }

    /**
     * @brief       ポインティングデバイスの座標を返します。
     *
     * @return      ポインティングデバイスの座標です。
     *              座標が有効範囲に入っていない場合は無効な座標を返します。
     */
    const nn::util::Float2& GetPointer() const NN_NOEXCEPT
    {
        return this->IsPointerOn() ? m_Pointer : GetInvalidPointer();
    }

    /**
     * @brief       ポインティングデバイスの座標が最後に有効範囲内にあった際の値を返します。
     *
     * @return      ポインティングデバイスの座標が最後に有効範囲内にあった際の値です。
     *              一度も有効範囲に入ったことがない場合は無効な座標を返します。
     */
    const nn::util::Float2& GetLastValidPointer() const NN_NOEXCEPT
    {
        return m_Pointer;
    }

    //! @}

    //! @name   デジタルボタンの押下判定
    //! @{

    /**
     * @brief       指定されたデジタルボタンの何れかが押下中か否かを返します。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     *
     * @return      指定されたデジタルボタンの何れかが押下中か否かを表す値です。
     */
    bool HasAnyButtons(ButtonSet buttons) const NN_NOEXCEPT
    {
        return (m_Buttons & buttons).IsAnyOn();
    }

    /**
     * @brief       指定されたデジタルボタンの何れかが最後の更新で新たに押下されたか否かを返します。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     *
     * @return      指定されたデジタルボタンの何れかが最後の更新で新たに押下されたか否かを表す値です。
     */
    bool HasAnyButtonsDown(ButtonSet buttons) const NN_NOEXCEPT
    {
        return (m_ButtonsDown & buttons).IsAnyOn();
    }

    /**
     * @brief       指定されたデジタルボタンの何れかが最後の更新で新たに開放されたか否かを返します。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     *
     * @return      指定されたデジタルボタンの何れかが最後の更新で新たに開放されたか否かを表す値です。
     */
    bool HasAnyButtonsUp(ButtonSet buttons) const NN_NOEXCEPT
    {
        return (m_ButtonsUp & buttons).IsAnyOn();
    }

    /**
     * @brief       指定されたデジタルボタンの何れかがリピートされたか否かを返します。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     *
     * @return      指定されたデジタルボタンの何れかがリピートされたか否かを表す値です。
     */
    bool HasAnyButtonsRepeat(ButtonSet buttons) const NN_NOEXCEPT
    {
        return (m_ButtonsRepeat & buttons).IsAnyOn();
    }

    /**
     * @brief       指定されたデジタルボタンの全てが押下中か否かを返します。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     *
     * @return      指定されたデジタルボタンの全てが押下中か否かを表す値です。
     */
    bool HasAllButtons(ButtonSet buttons) const NN_NOEXCEPT
    {
        return ((m_Buttons & buttons) == buttons);
    }

    //! @}

    //! @name   コントローラの設定
    //! @{

    /**
     * @brief       デジタルボタンにリピートを設定します。
     *
     * @details     intervalCount に 0 を指定するとリピートは行われなくなります。
     *
     * @param[in]   buttons                     デジタルボタンの指定です。
     * @param[in]   delayCount                  リピートを開始するまでのフレーム数です。
     * @param[in]   intervalCount               リピートの間隔となるフレーム数です。
     */
    void SetButtonsRepeatOption(
        ButtonSet buttons, uint8_t delayCount, uint8_t intervalCount
        ) NN_NOEXCEPT;

    /**
     * @brief       十字キー割り当て時の左スティックの閾値を設定します。
     *
     * @details     down と up に同じ値を設定した場合、その値は押下側として扱われます。
     *
     * @param[in]   down                        押下状態と見做す閾値です。
     * @param[in]   up                          開放状態と見做す閾値です。
     *
     * @pre
     *              - 0.0f <= down && down <= 1.0f
     *              - 0.0f <= up && up <= 1.0f
     *              - up <= down
     */
    void SetLeftStickDpadThreshold(float down, float up) NN_NOEXCEPT;

    /**
     * @brief       十字キー割り当て時の左スティックの閾値を設定します。
     *
     * @details     down と up に同じ値を設定した場合、その値は押下側として扱われます。
     *
     * @param[in]   down                        押下状態と見做す閾値です。
     * @param[in]   up                          開放状態と見做す閾値です。
     *
     * @pre
     *              - 0.0f <= down && down <= 1.0f
     *              - 0.0f <= up && up <= 1.0f
     *              - up <= down
     */
    void SetRightStickDpadThreshold(float down, float up) NN_NOEXCEPT;

    /**
     * @brief       ポインティングデバイスの座標の有効範囲を設定します。
     *
     * @param[in]   min                         有効範囲の下限座標です。
     * @param[in]   max                         有効範囲の上限座標です。
     *
     * @pre
     *              - std::numeric_limits<float>::min() < min.x
     *              - std::numeric_limits<float>::min() < min.y
     *              - min.x <= max.x
     *              - min.y <= max.y
     */
    void SetPointerBoundary(
        const nn::util::Float2& min, const nn::util::Float2& max) NN_NOEXCEPT;

    /**
     * @brief       ポインティングデバイスの有効範囲の下限座標を返します。
     *
     * @return      ポインティングデバイスの有効範囲の下限座標です。
     */
    const nn::util::Float2& GetPointerBoundaryMin() const NN_NOEXCEPT
    {
        return m_PointerBoundaryMin;
    }

    /**
     * @brief       ポインティングデバイスの有効範囲の上限座標を返します。
     *
     * @return      ポインティングデバイスの有効範囲の上限座標です。
     */
    const nn::util::Float2& GetPointerBoundaryMax() const NN_NOEXCEPT
    {
        return m_PointerBoundaryMax;
    }

    //! @}

    //! @name   コントローラの状態の取得
    //! @{

    /**
     * @brief       コントローラが接続状態にあるか否かを返します。
     *
     * @details     接続状態をコントローラが持たない場合は常に true を返します。
     *
     * @return      コントローラが接続状態にあるか否かを表す値です。
     */
    virtual bool IsConnected() const NN_NOEXCEPT
    {
        return true;
    }

    /**
     * @brief       コントローラがアイドル状態にあるか否かを返します。
     *
     * @details     以下の条件を全て満たす場合のみ、アイドル状態と判定されます。
     *              - GetButtons() が 0 を返す。
     *              - IsPointerOn() が true を返す。
     *              - GetLeftStick() の返す座標が (0.0f, 0.0f) を示す。
     *              - GetRightStick() の返す座標が (0.0f, 0.0f) を示す。
     *
     * @return      コントローラがアイドル状態にあるか否かを表す値です。
     */
    virtual bool IsIdle() const NN_NOEXCEPT
    {
        return this->GetButtons().IsAllOff() &&
              !this->IsPointerOn() &&
               this->GetLeftStick().x == 0.0f &&
               this->GetLeftStick().y == 0.0f &&
               this->GetRightStick().x == 0.0f &&
               this->GetRightStick().y == 0.0f;
    }

    //! @}

protected:
    static const uint8_t DefaultRepeatDelayCount = 25;      //!< リピートを開始するまでのフレーム数のデフォルト値です。
    static const uint8_t DefaultRepeatIntervalCount = 8;    //!< リピートの間隔となるフレーム数のデフォルト値です。

    static const float DefaultStickDpadDownThreshold;   //!< 十字キー割り当て時にスティックを押下状態と見做す閾値のデフォルト値です。
    static const float DefaultStickDpadUpThreshold;     //!< 十字キー割り当て時にスティックを開放状態と見做す閾値のデフォルト値です。
    static const nn::util::Float2 InvalidPointer;       //!< ポインティングデバイスの無効な座標です。

    /**
     * @brief   ポインティングデバイスのフェーズ定義です。
     */
    struct PointerPhase
    {
        typedef nn::util::BitFlagSet<32, PointerPhase>::Flag<0> On;             //!< 座標が有効範囲に入っています。
        typedef nn::util::BitFlagSet<32, PointerPhase>::Flag<1> Began;          //!< 最後の更新で座標が有効範囲に入りました。
        typedef nn::util::BitFlagSet<32, PointerPhase>::Flag<2> Ended;          //!< 最後の更新で座標が有効範囲から出ました。
        typedef nn::util::BitFlagSet<32, PointerPhase>::Flag<3> Reconfigured;   //!< 有効範囲が再設定されました。
        typedef nn::util::BitFlagSet<32, PointerPhase>::Flag<4> Canceled;       //!< 操作がキャンセルされました。
    };

    /**
     * @brief   ポインティングデバイスのフェーズの集合を扱う型です。
     */
    typedef nn::util::BitFlagSet<32, PointerPhase> PointerPhaseSet;

    /**
     * @brief      十字キー内の各方向のボタン割り当て位置です。
     */
    enum Dpad
    {
        Dpad_Up,    //!< 十字キー上のボタン割り当て位置です。
        Dpad_Down,  //!< 十字キー下のボタン割り当て位置です。
        Dpad_Left,  //!< 十字キー左のボタン割り当て位置です。
        Dpad_Right  //!< 十字キー右のボタン割り当て位置です。
    };

    /**
     * @brief       コントローラの基底の状態を更新します。
     *
     * @details     Update() 内で呼び出されます。
     *
     * @param[in]   buttons                     押下中のデジタルボタンです。
     * @param[in]   leftStick                   左スティックの座標です。
     * @param[in]   rightStick                  右スティックの座標です。
     * @param[in]   isTouched                   ポインティングデバイスのデジタルボタンが押下中か否かを表す値です。
     * @param[in]   pointer                     ポインティングデバイスの座標です。
     *
     * @pre
     *              - -1.0f <= leftStick.x && leftStick.x <= 1.0f
     *              - -1.0f <= leftStick.y && leftStick.y <= 1.0f
     *              - -1.0f <= rightStick.x && rightStick.x <= 1.0f
     *              - -1.0f <= rightStick.y && rightStick.y <= 1.0f
     */
    virtual void UpdateBase(
        ButtonSet buttons,
        const nn::util::Float2& leftStick, const nn::util::Float2& rightStick,
        bool isTouched, const nn::util::Float2& pointer) NN_NOEXCEPT;

private:
    ControllerManager* m_pManager;  //!< コントローラの管理者です。

    ButtonSet m_Buttons;            //!< 押下中のデジタルボタンです。
    ButtonSet m_ButtonsDown;        //!< 最後の更新で新たに押下されたデジタルボタンです。
    ButtonSet m_ButtonsUp;          //!< 最後の更新で新たに開放されたデジタルボタンです。
    ButtonSet m_ButtonsRepeat;      //!< リピートされたデジタルボタンです。
    nn::util::Float2 m_LeftStick;   //!< 左スティックの座標です。
    nn::util::Float2 m_RightStick;  //!< 右スティックの座標です。
    PointerPhaseSet m_PointerPhase; //!< ポインティングデバイスのフェーズです。
    nn::util::Float2 m_Pointer;     //!< ポインティングデバイスの座標です。

    uint32_t m_ButtonsDurationCounts[ButtonCountMax];   //!< デジタルボタンが押下中になってから経過したフレーム数です。

    uint8_t m_RepeatDelayCounts[ButtonCountMax];        //!< リピートを開始するまでのフレーム数です。
    uint8_t m_RepeatIntervalCounts[ButtonCountMax];     //!< リピートの間隔となるフレーム数です。

    float m_LeftStickDpadDownThreshold;     //!< 十字キー割り当て時に左スティックを押下状態と見做す閾値です。
    float m_RightStickDpadDownThreshold;    //!< 十字キー割り当て時に右スティックを押下状態と見做す閾値です。
    float m_LeftStickDpadUpThreshold;       //!< 十字キー割り当て時に左スティックを開放状態と見做す閾値です。
    float m_RightStickDpadUpThreshold;      //!< 十字キー割り当て時に右スティックを開放状態と見做す閾値です。

    bool m_IsPointerBoundaryDefined;        //!< ポインティングデバイスの有効範囲が定義されているか否かを表す値です。
    nn::util::Float2 m_PointerBoundaryMin;  //!< ポインティングデバイスの有効範囲の下限座標です。
    nn::util::Float2 m_PointerBoundaryMax;  //!< ポインティングデバイスの有効範囲の上限座標です。

    /**
     * @brief   Controller が管理対象とするアドオンです。
     */
    std::vector<ControllerAddon*> m_ControllerAddons;

    /**
     * @brief       スティックを十字キーへ割り当てた結果をデジタルボタンの集合で返します。
     *
     * @param[in]   stick                       スティックの座標です。
     * @param[in]   down                        押下状態と見做す閾値です。
     * @param[in]   up                          開放状態と見做す閾値です。
     * @param[in]   prevButtons                 前回の更新時に押下中だったデジタルボタンです。
     * @param[in]   stickDpadPosition           十字キー割り当て開始位置です。
     *
     * @return      十字キーへ割り当てた結果を表すデジタルボタンの集合です。
     */
    ButtonSet GetStickDpad(
        const nn::util::Float2& stick, float down, float up,
        ButtonSet prevButtons,
        int dpadPosition) const NN_NOEXCEPT;

    /**
     * @brief       指定位置の十字キーのビットを全てオンにしたデジタルボタンの集合を返します。
     *
     * @param[in]   stickDpadPosition           十字キー割り当て開始位置です。割り当てを行わない場合は -1 を指定します。
     *
     * @return      指定位置の十字キーのビットを全てオンにしたデジタルボタンの集合です。
     */
    ButtonSet GetDpadMask(int dpadPosition) const NN_NOEXCEPT;

    /**
     * @brief       指定されたデジタルボタンにリピートが必要か否かを返します。
     *
     * @param[in]   position                    指定されたデジタルボタンの位置です。
     *
     * @return      指定されたデジタルボタンにリピートが必要か否かを表す値です。
     *
     * @pre
     *              - position < ButtonCountMax
     */
    bool NeedsRepeat(int position) const NN_NOEXCEPT;

    /**
     * @brief       指定された座標がポインティングデバイスの有効範囲に入っているか否かを返します。
     *
     * @param[in]   pointer                     指定された座標です。
     *
     * @return      指定された座標がポインティングデバイスの有効範囲に入っているか否かを表す値です。
     */
    bool IsInsidePointerBoundary(
        const nn::util::Float2& pointer) const NN_NOEXCEPT;

    /**
     * @brief       ポインティングデバイスの状態を更新します。
     *
     * @param[in]   pointer                     ポインティングデバイスの座標です。
     * @param[in]   isInside                    ポインティングデバイスの座標が有効範囲に入っているか否かを表す値です。
     */
    void UpdatePointer(
        const nn::util::Float2& pointer, bool isInside) NN_NOEXCEPT;
};

}} // namespace nns::hid
