﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
* @file gfxLog.h
*
* ログ出力の設定を行うAPI のヘッダです。
*/
#pragma once

#include <nn/vi.h>
#include <nn/gfx/gfx_Types.h>
#include <nn/util/util_Color.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

namespace nns {
namespace gfxLog {
/**
* @brief テキスト色を設定します。
*
* @param[in]    Color        設定する色
* @details 次に描画するテキストの色を設定します。
*/
void SetTextColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;

/**
* @brief テキストの背景色を設定します。
*
* @param[in]    Color        設定する色
* @details 次に描画するテキストの背景色を設定します。
*/
void SetBackgroudColor(const nn::util::Color4u8& Color) NN_NOEXCEPT;

/**
* @brief 描画用の矩形を設定します。
*
* @param[in]    Left    左のX座標
* @param[in]    Top     上のY座標
* @param[in]    Width   幅
* @param[in]    Height  高さ
* @details   次に描画するときの使用する矩形を設定します。
*/
void SetDrawRect(const float Left, const float Top, const float Width, const float Height) NN_NOEXCEPT;

/**
* @brief SetConfiguration での設定をもとに描画コマンドを生成します。
*
* @details   事前に SetConfiguration を実行している場合にカスタムドローを行う際に
*            描画コマンドの発行を行います。
*/
void WriteCommand() NN_NOEXCEPT;

/**
* @brief ログ描画用スレッドのコアを設定します。
*
* @param[in] idealCore     優先して動作するコア番号
* @details ログ描画用スレッドのコア割り当てを設定します。
*
*          この関数を呼び出す場合は、 SetConfiguration() 以外の関数や
*          NNS_LOG() を呼ぶ前に呼び出してください。
*          他の関数や NNS_LOG() を呼んだあとで SetThreadCoreNumber() を呼び出しても
*          効果を持ちません。
*/
void SetThreadCoreNumber(int idealCore) NN_NOEXCEPT;

/**
* @brief デバイス、コマンドバッファ、 デバッグフォントライターを設定します。
*
* @param[in]    pDevice               デバイス
* @param[in]    pCommandBuffer        コマンドバッファ
* @param[in]    pDebugFontWriter      デバッグフォントライター
* @details   ライブラリユーザーが自前で描画する際、ライブラリ内部で使用するために
*            デバイス、コマンドバッファ、 デバッグフォントライターを設定します。
*
*            この関数を呼び出す場合は、他の関数や NNS_LOG() を呼ぶ前に呼び出してください。
*            他の関数や NNS_LOG() を呼んだあとで SetConfiguration() を呼び出しても
*            効果を持ちません。
*/
void SetConfiguration(
    nn::gfx::Device* pDevice,
    nn::gfx::CommandBuffer* pCommandBuffer,
    nn::gfx::util::DebugFontTextWriter* pDebugFontWriter) NN_NOEXCEPT;
} // namespace gfxLog
} // namespace nns
