﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/gfx.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>

#include <nn/mem/mem_StandardAllocator.h>

namespace nns
{
namespace gfx
{
namespace PrimitiveRenderer
{

//! @brief 前方宣言です。
class Shader;
class MemoryInfo;

//! @brief シェーダスロット未使用です。
static const int ShaderSlotNone = -1;

//! @brief 頂点属性です。
enum VertexAttribute
{
    VertexAttribute_Pos = 0,
    VertexAttribute_Color,
    VertexAttribute_Uv,
    VertexAttribute_Normal,

    VertexAttribute_CountMax
};

//! @brief 頂点フォーマットです。
enum VertexFormat
{
    VertexFormat_Pos      = 1 << VertexAttribute_Pos,
    VertexFormat_Color    = 1 << VertexAttribute_Color,
    VertexFormat_Uv       = 1 << VertexAttribute_Uv,
    VertexFormat_Normal   = 1 << VertexAttribute_Normal,

    VertexFormat_Default = VertexFormat_Pos | VertexFormat_Uv | VertexFormat_Color | VertexFormat_Normal
};

//! @brief デフォルトシェーダのサンプラ種類です。
enum SamplerType
{
    SamplerType_Texture = 0,
    SamplerType_TextureArray,
    SamplerType_TextureCubemap,

    SamplerType_End,
    SamplerType_CountMax = SamplerType_End
};

//! @brief デフォルトシェーダのバリエーションです。
enum ShaderVariation
{
    ShaderVariation_P = 0,
    ShaderVariation_PN,
    ShaderVariation_PUv,
    ShaderVariation_PNUv,
    ShaderVariation_PC,
    ShaderVariation_PUvC,
    ShaderVariation_TextureArray,
    ShaderVariation_Cubemap,

    ShaderVariation_End,
    ShaderVariation_CountMax = ShaderVariation_End
};

//! @brief デフォルトシェーダの定数バッファです。
enum ConstantBufferType
{
    ConstantBufferType_View = 0,
    ConstantBufferType_Model,
    ConstantBufferType_User,

    ConstantBufferType_End,
    ConstantBufferType_CountMax = ConstantBufferType_End
};

//! @brief シェーダです。
class Shader
{
    NN_DISALLOW_COPY(Shader);
public:
    //! @brief デフォルトシェーダの頂点フォーマットを取得します。
    //! @param[in] shaderVariation   シェーダバリエーションです。
    //! @return 頂点フォーマットです。
    static uint32_t GetDefaultVertexFormat( ShaderVariation shaderVariation );

    //! @brief シェーダを生成します。
    //! @details リソースシェーダのコード種類は、GL版はソース、NVN版はバイナリです。
    //! @param[inout]   pShader         生成するシェーダです。
    //! @param[in]      pResShaderFile  リソースシェーダファイルです。
    //! @param[in]      variationIndex  バリエーションです。
    //! @param[in]      pGfxDevice      GPU デバイスです。
    //! @param[in]      pMemoryInfo      メモリ管理構造体です。
    //! @return         生成結果です。true なら生成に成功しました。
    static bool Create( Shader* pShader,
                        nn::gfx::ResShaderFile* pResShaderFile,
                        ShaderVariation variationIndex,
                        nn::gfx::Device* pGfxDevice,
                        nn::util::BytePtr& pMemory );

    //! @brief 頂点ステートを生成します。
    //! @param[inout]   pShader         生成するシェーダです。
    //! @param[in]      shaderVariation  シェーダバリエーション
    //! @param[in]      pVertexShader        頂点シェーダです。
    //! @param[in]      pGfxDevice           GPU デバイスです。
    //! @param[in]      pMemoryInfo      メモリ管理構造体です。
    //! @return         生成結果です。trueなら生成に成功しました。
    static bool CreateVertexState( Shader*            pShader,
                                   ShaderVariation    shaderVariation,
                                   nn::gfx::Shader*   pVertexShader,
                                   nn::gfx::Device*   pGfxDevice,
                                   nn::util::BytePtr& pMemory );

    //! @brief 非インターリーブ版の頂点ステート情報を生成します。
    //! @param[out]     info             頂点ステートを初期化するためのクラスです。
    //! @param[out]     attribs          頂点属性ステートを表します。
    //! @param[out]     buffers          頂点バッファステートを表します。
    //! @param[in]      shaderVariation  シェーダバリエーション
    //! @param[in]      pVertexShader    頂点シェーダです。
    //! @return         頂点属性数を返します。o なら失敗です。
    static int InitializeVertexStateInfo( nn::gfx::VertexState::InfoType*      info,
                                          nn::gfx::VertexAttributeStateInfo*   attribs,
                                          nn::gfx::VertexBufferStateInfo*      buffers,
                                          ShaderVariation                      shaderVariation,
                                          nn::gfx::Shader*                     pVertexShader );

    //! @brief インターリーブ版の頂点ステート情報を生成します。
    //! @param[out]     info             頂点ステートを初期化するためのクラスです。
    //! @param[out]     attribs          頂点属性ステートを表します。
    //! @param[out]     buffers          頂点バッファステートを表します。
    //! @param[in]      shaderVariation  シェーダバリエーション
    //! @param[in]      pVertexShader    頂点シェーダです。
    //! @return         頂点属性数を返します。o なら失敗です。
    static int InitializeInterleavedVertexStateInfo( nn::gfx::VertexState::InfoType*      info,
                                                     nn::gfx::VertexAttributeStateInfo*   attribs,
                                                     nn::gfx::VertexBufferStateInfo*      buffers,
                                                     ShaderVariation                      shaderVariation,
                                                     nn::gfx::Shader*                     pVertexShader );



public:
    //! @brief 定数です。
    enum Constants
    {
        Constants_UserSlotCountMax =  2, // 0: Vertex, 1: Pixel
        Constants_ModelSlotCountMax = 2, // 0: Vertex, 1: Pixel
        Constants_SamplerSlotCountMax
                = SamplerType_CountMax, // 0: texture, 1: textureArray, 2: cubemap
    };

public:
    //! @brief コンストラクタです。
    Shader();

    //! @brief デストラクタです。
    ~Shader();

    //! @brief 初期設定にします。
    void SetDefault();

    //! @brief 破棄します。
    void Cleanup( nn::gfx::Device* pGfxDevice );

    //! @brief 頂点シェーダを設定します。
    //! @param[in] pGfxShader   GFXシェーダを指定します。
    void SetVertexShader( nn::gfx::Shader* pGfxShader );

    //! @brief ピクセルシェーダを設定します。
    //! @param[in] pGfxShader   GFXシェーダを指定します。
    void SetPixelShader( nn::gfx::Shader* pGfxShader );

    //! @brief 頂点シェーダを取得します。
    //! @return シェーダです。
    nn::gfx::Shader* GetVertexShader();

    //! @brief ピクセルシェーダを取得します。
    //! @return シェーダです。
    nn::gfx::Shader* GetPixelShader();

    //! @brief 非インターリブ版の頂点ステートを取得します。
    //! @return 頂点ステートです。
    const nn::gfx::VertexState* GetVertexState() const;

    //! @brief インターリーブ版の頂点ステートを取得します。
    //! @return 頂点ステートです。
    const nn::gfx::VertexState* GetInterleavedVertexState() const;

    //! @brief 引数で指定した頂点属性を入力として受け付けるかを返します。
    //! @param[in] attribute  頂点属性です。
    //! @return 判定結果です。true なら有効です。
    bool HasVertexAttribute( VertexAttribute attribute ) const;

    //! @brief サンプラスロットを取得します。
    //! @param[in] name         サンプラ名です。シェーダで使用している名前です。
    //! @param[in] shaderStage  シェーダステージです。
    //! @return サンプラスロットインデックスです。
    int GetSamplerSlot( const char* name, const nn::gfx::ShaderStage shaderStage );

    //! @brief システムスロット情報を更新します。
    void UpdateSystemSlots();

public:
    uint32_t vertexFormat;
    int shaderVariationIndex;
    int shaderSlotCount;

    // システムで予約されているスロットです。
    int slotView;
    int slotModel[ Constants_ModelSlotCountMax ]; // 0: Vertex, 1: Pixel

    // ユーザスロットです。
    int slotUser[ Constants_UserSlotCountMax ];

    // システムで予約されているサンプラスロットです。
    int slotSamplerPs[ Constants_SamplerSlotCountMax ];  // 0: texture, 1: textureArray, 2: cubemap

private:
    nn::gfx::Shader*        m_pGfxShader[ nn::gfx::ShaderStage_End ];
    nn::gfx::VertexState    m_VertexState;                              //!< 頂点ステートです。シェーダと頂点属性がリンクするので、このオブジェクトで保持します。
    nn::gfx::VertexState    m_VertexStateInterleaved;                   //!< インターリーブされた頂点ステートです。
};


} // namespace PrimitiveRenderer
} // namespace gfx
} // namespace nns
