﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/gfx.h>
#include <nn/nn_SdkLog.h>

namespace nns
{
namespace gfx
{

//------------------------------------------------------------------------------
//! @brief 公開APIです。
//------------------------------------------------------------------------------

//! @brief GPUバッファを管理します GpuBuffer
class GpuBuffer
{
public:
    //! @brief Initialize() の引数オブジェクトです。
    class InitializeArg
    {
    public:
        //! @brief コンストラクタです。
        InitializeArg()
            : m_GpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer)
            , m_BufferSize(0)
            , m_BufferCount(1)
        {}

        //! @brief このバッファがどのような用途のバッファとして使用されるかを指定します。
        //!
        //! @details
        //! この情報に基づいて nn::gfx::Buffer を作成します。
        //!
        //! @param[in] gpuAccess  バッファの種別を nn::gfx::GpuAccess で指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        InitializeArg& SetGpuAccessFlag(int gpuAccess)
        {
            m_GpuAccessFlag = gpuAccess;
            return *this;
        }

        //! @brief  使用するバッファのサイズを指定します。
        //!
        //! @details
        //! このメソッドに指定されたサイズのメモリが SetBufferCount で指定された数だけメモリプール上の領域を使用します。
        //!
        //! @param[in]  bufferSize  バッファのサイズをバイト単位で指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        //! @sa SetBufferCount
        //!
        InitializeArg& SetBufferSize(size_t bufferSize)
        {
            m_BufferSize = bufferSize;
            return *this;
        }

        //! @brief  バッファのマルチバッファリングの数をしてします。
        //!
        //! @details
        //! ここで設定した数だけ内部で nn::gfx::Buffer が作成され、切り替えて使用することが出来ます。
        //! たとえばダブルバッファリングしたい場合は 2 を指定、トリプルバッファリングしたい場合は 3 を指定します。
        //!
        //! @param[in] bufferCount  マルチバッファリングする数を指定します。
        //!
        //! @return オブジェクト自身への参照を返します。
        //!
        InitializeArg& SetBufferCount(int bufferCount)
        {
            m_BufferCount = bufferCount;
            return *this;
        }
    private:
        int                   m_GpuAccessFlag;
        size_t                m_BufferSize;
        uint32_t              m_BufferCount;

        friend class GpuBuffer;
    };
    //----------------------------------------
    //! @name コンストラクタ／デストラクタ
    //@{

    //! @brief コンストラクタです。
    //!
    GpuBuffer()
    : m_Buffer()
    , m_Size(0)
    , m_Alignment(0)
    , m_Count(0)
    , m_AllBufferSize(0)
    , m_MappedBufferIndex(-1)
    , m_pMappedPointer(NULL)
    , m_isInitialized(false)
    {
    }

    //! @brief  デストラクタ
    //!
    ~GpuBuffer()
    {
    }

    //@}

    //! @brief  初期化処理です。
    //!
    //! @details
    //! メモリプールからメモリを確保しバッファを作成します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //! @param[in]  pMemoryPool 作成するバッファのメモリを確保するメモリプールです。
    //! @param[in]  memoryPoolOffset    メモリを確保する領域のメモリプール先頭からのオフセットです。
    //!
    //! @return バッファの作成成功可否です。
    //!
    bool Initialize(
        nn::gfx::Device* pDevice,
        const InitializeArg&  arg,
        nn::gfx::MemoryPool* pMemoryPool,
        size_t memoryPoolOffset
        );

    //! @brief   この関数は非推奨です。初期化処理です。
    //! @deprecated  初期化時の引数が InitializeArg にまとめられた Initialize メソッドが用意されました。そちらの利用に切り替えてください。
    //!
    //! @details
    //! 本関数は互換性の目的で残されている関数です。渡されたメモリプールからメモリを確保しバッファを作成します。
    //! メモリプールからメモリを確保しバッファを作成します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  gpuAccess   Gpu からのアクセス方法です。
    //! @param[in]  bufferSize  作成するバッファのサイズです。
    //! @param[in]  bufferCount 作成するバッファのマルチバッファリングのカウントです。
    //! @param[in]  pMemoryPool 作成するバッファのメモリを確保するメモリプールです。
    //! @param[in]  memoryPoolOffset    メモリを確保する領域のメモリプール先頭からのオフセットです。
    //!
    //! @return バッファの作成成功可否です。
    //!
    NN_DEPRECATED bool Initialize(
        nn::gfx::Device* pDevice,
        nn::gfx::GpuAccess gpuAccess,
        size_t bufferSize,
        int bufferCount,
        nn::gfx::MemoryPool* pMemoryPool,
        size_t memoryPoolOffset
        );

    //! @brief  終了処理です。
    //!
    //! @param[in]  pDevice デバイスです。
    //!
    void Finalize(
        nn::gfx::Device* pDevice);

    //! @brief 初期化済みかどうかフラグを取得します。
    //!
    //! @return 初期化済みならば true が返ります。
    bool IsInitialized() const
    {
        return m_isInitialized;
    }

    //! @brief バッファを CPU で更新するための事前処理を行います。
    //!
    //! @details
    //! バッファの値を CPU で書き換える前に呼び出してください。
    //!
    //! @param[in]  bufferIndex 事前準備を行うバッファのインデックスを指定します。
    //!
    void Map(int bufferIndex);

    //! @brief バッファを CPU で書き換えた後の終了処理を行います。
    //!
    //! @details
    //! バッファの値を CPU で書き換えた後に呼び出してください。
    //!
    void Unmap();

    //! @brief バッファを Map したポインタを取得します。
    //!
    //! @return バッファを Map したポインタです。
    //!
    //! @sa Map
    //!
    void* GetMappedPointer()
    {
        NN_SDK_ASSERT_NOT_NULL(m_pMappedPointer);

        return m_pMappedPointer;
    }

    //! @brief バッファに指定されたサイズ分の領域を確保します。
    //!
    //! @return 確保された領域の先頭ポインタ。失敗した場合は、nullptrを返す。
    //!
    //! @sa GetMappedPointer
    //! @sa GetGpuAddress
    //!
    void* Allocate(size_t size)
    {
        size_t offset = m_BufferAllocatedSize.m_Raw + m_MappedBufferIndex*m_Size;
        size_t sizeRaw = nn::util::align_up(m_BufferAllocatedSize.m_Raw + size, m_Alignment );

        // バッファサイズを超えていたら、nullptrを返す。
        if ( sizeRaw > m_Size )
        {
            NN_SDK_LOG("[GpuBuffer] Warning!! m_BufferAllocatedSize.m_Raw > m_Size\n");
            return nullptr;
        }

        m_BufferAllocatedSize.m_Raw = sizeRaw;
        return nn::util::BytePtr( m_pMappedPointer ).Advance( offset ).Get();
    }

    //----------------------------------------
    //! @name 設定/取得
    //@{

    //! @brief バッファのサイズを取得します。
    //!
    //! @return バッファのサイズです。
    //!
    size_t GetBufferSize() const
    {
        return m_Size;
    }

    //! @brief バッファの数を取得します。
    //!
    //! @return バッファの数です。
    //!
    int GetBufferCount() const
    {
        return m_Count;
    }

    //! @brief バッファのアライメントを取得します。
    //!
    //! @return バッファのアライメント値です。
    //!
    size_t GetBufferAlignment() const
    {
        return m_Alignment;
    }

    //! @brief 確保済みのメモリのサイズを取得します。
    //!
    //! @return 確保済みの領域のサイズです。
    //!
    size_t GetAllocatedSize()
    {
        return m_BufferAllocatedSize.m_Raw;
    }

    // バッファの GpuAddr を取得します。

    //! @brief バッファのポインタからGetGpuAddressを取得します。
    //!
    //! @param[out] pGpuAddress  取得したGpuAdress
    //! @param[in]  address      GpuAddressを取得するバッファのポインタ
    //!
    void GetGpuAddress(nn::gfx::GpuAddress* pGpuAddress, void* address) const
    {
        NN_SDK_ASSERT(m_isInitialized);
        NN_SDK_ASSERT(nn::util::BytePtr( m_pMappedPointer ).Distance( address ) >= 0);
        NN_SDK_ASSERT(static_cast<size_t>(nn::util::BytePtr( m_pMappedPointer ).Distance( address )) <= m_AllBufferSize);

        m_Buffer.GetGpuAddress(pGpuAddress);
        pGpuAddress->Offset( nn::util::BytePtr( m_pMappedPointer ).Distance( address ) );
    }

    //! @brief  初期化パラメータから、バッファのアライメントを計算します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //!
    //! @return バッファのアライメント
    //!
    static size_t GetGpuBufferAlignement(
        nn::gfx::Device* pDevice,
        const InitializeArg& arg);

    //! @brief  初期化パラメータから、必要なメモリプールの容量を計算します。
    //!
    //! @param[in]  pDevice デバイスです。
    //! @param[in]  arg     初期化パラメータです。
    //!
    //! @return 必要なメモリプールの容量
    //!
    static size_t CalculateGpuBufferPoolSize(
        nn::gfx::Device* pDevice,
        const InitializeArg& arg);

    //@}
private:
    // コピー演算子を禁止します。
    const GpuBuffer& operator=( const GpuBuffer& );

    // Gpu がアクセスするバッファ
    nn::gfx::Buffer     m_Buffer;
    // バッファのサイズ
    size_t              m_Size;
    // バッファのアライメント
    size_t              m_Alignment;
    // バッファのバッファリングカウント
    uint32_t            m_Count;
    // 全バッファのサイズ(nn::gfx::Bufferであるm_Bufferの実際のサイズ)
    size_t              m_AllBufferSize;
    //  マップしたバッファのインデックス
    int32_t m_MappedBufferIndex;
    //  マップした際のポインタ
    void*   m_pMappedPointer;
    //  初期化したかどうか
    bool    m_isInitialized;
    //  マップ後に確保された領域のサイズ
    union
    {
        size_t                  m_Raw;      // 同期処理不要の時に使用される変数
    }m_BufferAllocatedSize;
};

} // namespace gfx
} // namespace nns
