﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/gfx/util/gfx_MemoryPoolAllocator.h>
#include <nn/gfx.h>

namespace nns { namespace g3d {

#define NNS_G3D_WARNING(fmt, ...) NNS_G3D_WARNING_IMPL("[nns::g3d] WARNING : " fmt " (file %s line %d)\n", ##__VA_ARGS__, __FILE__, __LINE__)
#define NNS_G3D_WARNING_IMPL(...)  \
    do                             \
    {                              \
        if (nns::g3d::IsVerbose()) \
        {                          \
            NN_LOG(__VA_ARGS__);   \
        }                          \
    } while (NN_STATIC_CONDITION(0));

//! @brief メモリー確保時に使用されるデフォルトのアライメントです。
static const size_t Default_Alignment = 8;

//! @brief nns::g3d 内で使用されるメモリーアロケーターを取得します。
//!
//! @return nn::mem::StandardAllocator のポインター。
//!
nn::mem::StandardAllocator* GetMemoryAllocator() NN_NOEXCEPT;

//! @brief nns::g3d 内で使用されるメモリープールアロケーターを取得します。
//!
//! @return nn::gfx::util::MemoryPoolAllocator のポインター。
//!
nn::gfx::util::MemoryPoolAllocator* GetMemoryPoolAllocator() NN_NOEXCEPT;

//! @briefprivate nns::g3d 内で使用されるメモリープールを取得します。
//!
//! @return nn::gfx::MemoryPool のポインター。
//!
nn::gfx::MemoryPool* GetMemoryPool() NN_NOEXCEPT;

//! @briefprivate 指定した型および指定したアライメントでメモリーを確保します。
//!
//! @tparam T 確保するメモリーの型。
//!
//! @param[in] size 確保するメモリーのサイズ。
//! @param[in] alignment 確保するメモリーのアライメント。
//!
//! @return 確保したメモリーのT型のポインター。
//!
template<typename T = void>
NN_FORCEINLINE T* Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    void* ptr = GetMemoryAllocator()->Allocate(size, alignment);
    NN_ASSERT(ptr != NULL, "Can't allocate the memory");

    return reinterpret_cast<T*>(ptr);
}

//! @briefprivate 指定した型およびデフォルトのアライメントでメモリーを確保します。
//!
//! @tparam T 確保するメモリーの型。
//!
//! @param[in] size 確保するメモリーのサイズ。
//!
//! @return 確保したメモリーのT型のポインター。
//!
template<typename T = void>
NN_FORCEINLINE T* Allocate(size_t size) NN_NOEXCEPT
{
    return Allocate<T>(size, Default_Alignment);
}

//! @briefprivate メモリーを解放します。
//!
//! @param[in] ptr 解放するメモリーのポインター。
//!
NN_FORCEINLINE void Free(void* ptr) NN_NOEXCEPT
{
    GetMemoryAllocator()->Free(ptr);
}

//! @briefprivate メモリープールを確保します。
//!
//! @param[in] size 確保するメモリープールのサイズ。
//! @param[in] alignment 確保するメモリープールのアライメント。
//!
//! @return 確保したメモリープールのオフセット。
//!
NN_FORCEINLINE ptrdiff_t AllocateMemoryPool(size_t size, size_t alignment) NN_NOEXCEPT
{
    ptrdiff_t offset = GetMemoryPoolAllocator()->Allocate(size, alignment);
    NN_ASSERT(offset != -1, "Can't allocate the memory pool");

    return offset;
}

//! @briefprivate デフォルトのアライメントでメモリープールを確保します。
//!
//! @param[in] size 確保するメモリープールのメモリーサイズ。
//!
//! @return 確保したメモリープールのオフセット。
//!
NN_FORCEINLINE ptrdiff_t AllocateMemoryPool(size_t size) NN_NOEXCEPT
{
    return AllocateMemoryPool(size, Default_Alignment);
}

//! @briefprivate デフォルトのアライメントでメモリープールを確保します。
//!
//! @param[in] offset 解放するメモリープールのオフセット。
//!
NN_FORCEINLINE void FreeMemoryPool(ptrdiff_t offset) NN_NOEXCEPT
{
    GetMemoryPoolAllocator()->Free(offset);
}

//! @brief nns::g3d を初期化します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//! @param[in] pMemory nns::g3d に渡すメモリー領域へのポインター。
//! @param[in] memorySize nns::g3d に渡すメモリー領域のサイズ。
//! @param[in] pMemoryPoolMemory nns::g3d に渡すメモリープール用メモリー領域へのポインター。
//! @param[in] memoryPoolMemorySize nns::g3d に渡すメモリープール用メモリー領域のサイズ。
//!
//! @details
//! pMemoryPoolMemory は GetMemoryPoolAlignment() で取得できるアライメントでアラインされている必要があります。
//!
void Initialize(nn::gfx::Device* pDevice,
                void* pMemory, size_t memorySize,
                void* pMemoryPoolMemory, size_t memoryPoolMemorySize) NN_NOEXCEPT;

//! @brief nns::g3d を終了します。
//!
//! @param[in] pDevice nn::gfx::Device へのポインター。
//!
void Finalize(nn::gfx::Device* pDevice) NN_NOEXCEPT;

//! @brief メモリープールに必要なアライメントを返します。
//!
//! @return メモリープールのアライメント。
//!
size_t GetMemoryPoolAlignment(nn::gfx::Device* pDevice) NN_NOEXCEPT;

//! @brief デバッグメッセージを有効にします。
//!
//! @param[in] enabled デバッグメッセージを有効にする場合は true、無効にする場合は false を指定します。
//!
//! @details
//! デフォルトではデバッグメッセージは無効になっています。
//!
void SetVerbose(bool enabled) NN_NOEXCEPT;

//! @brief デバッグメッセージの有効、無効を取得します。
//!
//! @return デバッグメッセージが有効の場合は true、無効の場合は false を返します。
//!
bool IsVerbose() NN_NOEXCEPT;

}}
//--------------------------------------------------------------------------------------------------
