﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include <nns/afx/afx_SimpleChorusTypes.h>

namespace nns { namespace afx {

/**
 * @brief       SimpleChorus に必要なメモリサイズを返します。
 * @param[in]   c 初期化パラメータ
 * @return      メモリサイズを返します。
 * @details
 * 初期化パラメータとして有効な値については SimpleChorusConstant を参照してください。
 */
size_t GetRequiredMemorySizeForSimpleChorus(const SimpleChorusConstant* c);

/**
 * @brief       SimpleChorus を初期化します。
 * @param[out]  pOutEffect SimpleChorus の管理構造体
 * @param[in]   buffer SimpleChorus が使用する作業用メモリ
 * @param[in]   size SimpleChorus が使用する作業用メモリのサイズ
 * @param[in]   c 初期化パラメータ
 * @details
 * 初期化パラメータとして有効な値については SimpleChorusConstant を参照してください。
 */
void InitializeSimpleChorus(SimpleChorusType* pOutEffect, void* buffer, size_t size, const SimpleChorusConstant* c);

/**
 * @brief       SimpleChorus の終了処理を行います。
 * @param[in]   pEffect SimpleChorus の管理構造体
 * @return      SimpleChorus が使用していた作業用メモリを返します。
 */
void* FinalizeSimpleChorus(SimpleChorusType* pEffect);

/**
 * @brief       SimpleChorus の初期化パラメータを取得します。
 * @param[in]   pEffect SimpleChorus の管理構造体
 * @return      初期化パラメータを返します。
 */
SimpleChorusConstant GetSimpleChorusConstant(const SimpleChorusType* pEffect);

/**
 * @brief       SimpleChorus の可変パラメータを取得します。
 * @param[in]   pEffect SimpleChorus の管理構造体
 * @return      可変パラメータを返します。
 */
SimpleChorusParameter GetSimpleChorusParameter(const SimpleChorusType* pEffect);

/**
 * @brief       SimpleChorus の可変パラメータを設定します。
 * @param[in]   pEffect SimpleChorus の管理構造体
 * @param[in]   p 可変パラメータ
 * @details
 * 可変パラメータとして有効な値については SimpleChorusParameter を参照してください。
 */
void SetSimpleChorusParameter(SimpleChorusType* pEffect, const SimpleChorusParameter* p);

/**
 * @brief       SimpleChorus を適用します。
 * @param[in]   pEffect SimpleChorus の管理構造体
 * @param[out]  outputs 出力バッファ
 * @param[in]   inputs 入力バッファ
 * @param[in]   channelCount outputs/inputs のチャンネル数
 * @param[in]   sampleCount outputs/inputs のサンプル数
 * @details
 * inputs および outputs は、サイズが channelCount の int32_t* 型の配列で、それぞれの要素が指すバッファのサンプル数が sampleCount である必要があります。
 * inputs および outputs には同じ値を指定することができます。
 *
 * channelCount は InitializeSimpleChorus() 時に指定した初期化パラメータ SimpleChorusConstant::channelCount 以下である必要があります。
 */
void ApplySimpleChorus(SimpleChorusType* pEffect, int32_t* outputs[], const int32_t* const inputs[], int channelCount, int sampleCount);

}}  // namespace nns::afx
