﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NW_CONSOLE_ENABLE // Release 版でも NW_LOG を有効にするため

#include <nw/snd.h>
#include <nw/demo.h>

#if defined( NW_PLATFORM_CAFE )
#include <cafe.h>
#elif defined( NW_USE_NINTENDO_SDK )
#else
#include <winext/cafe.h>
#include <Windows.h>
#endif

//#define CPU_RENDERING

namespace
{
const s32 WAVE_AMP_MAX = 8192;

void* s_pMemoryForSoundSystem;

class MyFinalMixCallback : public nw::snd::FinalMixCallback
{
public:
    void OnFinalMix(nw::snd::OutputDevice device, const nw::snd::FinalMixData* data)
    {
        NW_ASSERT_NOT_NULL(data);
        if (device == nw::snd::OUTPUT_DEVICE_MAIN)
        {
            OnMainOut(data);
        }
        else
        {
            // 現時点ではDRC2台に対応していません。
            // 今後対応された場合には、2台ともこの処理を行うようになるため、
            // どちらのDRCからも矩形波の音が再生されるようになります。
            OnDrcOut(data);
        }
    }

private:
    // TV出力に対して、1 サウンドフレームを 1 周期とした約 333 Hzのサイン波を合成します。
    void OnMainOut(const nw::snd::FinalMixData* data)
    {
        s32 sampleCount = data->sampleCount;
        u32 step = static_cast<u32>(0x100000000 / sampleCount);
        for (s32 i = 0; i < sampleCount; ++i)
        {
            f32 factor = nw::math::SinIdx(i * step);

            if (data->frontLeft != NULL)
            {
                data->frontLeft[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
            }
            if (data->frontRight != NULL)
            {
                data->frontRight[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
            }
        }
    }

    // DRC出力に対して、1 サウンドフレームを1 周期とした約 333 Hzの矩形波を合成します。
    void OnDrcOut(const nw::snd::FinalMixData* data)
    {
        s32 sampleCount = data->sampleCount;
        for (s32 i = 0; i < sampleCount; ++i)
        {
            f32 factor = 0.0f;
            if (i > sampleCount * 0.5f)
            {
                factor = 0.5f;
            }

            if (data->frontLeft != NULL)
            {
                data->frontLeft[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
            }
            if (data->frontRight != NULL)
            {
                data->frontRight[i] += static_cast<s32>(WAVE_AMP_MAX * factor);
            }
        }
    }
};

} // namespace

int
NwDemoMain(int argc, char **argv)
{
#if defined( NW_PLATFORM_CAFE )
    AXInit();

#ifdef CPU_RENDERING
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_PPC );
#else
    AXSetDefaultMixerSelect( AX_PB_MIXER_SELECT_DSP );
#endif

#else
    NW_UNUSED_VARIABLE(argc);
    NW_UNUSED_VARIABLE(argv);
#endif

    // デモシステムの初期化
    nw::demo::DefaultAllocator allocator;
    nw::demo::DemoSystem::CreateArg demoSystemArg;
    demoSystemArg.allocator = &allocator;
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    demoSystemArg.waitVBlank = 0; // DemoSystem の SwapBuffer は呼ばないため、VBlank 待ちはしない。
#endif
    nw::demo::DemoSystem* pDemo =
        new( allocator.Alloc( sizeof( nw::demo::DemoSystem ) ) )
        nw::demo::DemoSystem( demoSystemArg );
    pDemo->Initialize();
    pDemo->InitializeGraphicsSystem();

    // サウンドシステムの初期化
    nw::snd::SoundSystem::SoundSystemParam param;
    size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
    s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

    nw::snd::SoundSystem::Initialize(
            param,
            reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
            workMemSize );

    MyFinalMixCallback finalMixCallback;
    bool isRegistered = false;

    NW_LOG("----------------------------------------\n");
    NW_LOG("NW4F Demo/snd/finalMix\n");
    NW_LOG("----------------------------------------\n");
    NW_LOG("[A]     Register/Unregister FinalMixCallback\n");
#if defined( NW_PLATFORM_CAFE )
    NW_LOG("[HOME]  Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    NW_LOG("[S]     Exit Application\n");
#endif
    NW_LOG("---------------------------------------\n");

    while ( !pDemo->IsExiting() )
    {
#if defined( NW_PLATFORM_CAFE )
        GX2WaitForVsync();
#else
        pDemo->WaitForVBlank();
#endif

        pDemo->UpdatePad();
        nw::demo::Pad* pad = pDemo->GetPad();

        // Register/Unregister FinalMixCallback
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            if (!isRegistered)
            {
                NW_LOG("Register FinalMixCallback\n");
                nw::snd::SoundSystem::AppendFinalMixCallback(&finalMixCallback);
                isRegistered = true;
            }
            else
            {
                NW_LOG("Unregister FinalMixCallback\n");
                nw::snd::SoundSystem::EraseFinalMixCallback(&finalMixCallback);
                isRegistered = false;
            }
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) ) {
            break;
        }
    }

    if (isRegistered)
    {
        nw::snd::SoundSystem::EraseFinalMixCallback(&finalMixCallback);
    }

    nw::snd::SoundSystem::Finalize();
    allocator.Free( s_pMemoryForSoundSystem );

#if defined( NW_PLATFORM_CAFE )
    AXQuit();
#endif

    pDemo->FinalizeGraphicsSystem();
    pDemo->Finalize();
    allocator.Free( pDemo );

    return 0;
}
