﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "remote";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    f32 s_Volume = 1.0f;
    f32 s_Pitch = 1.0f;
    const f32 PARAM_DELTA = 0.2f;
    const f32 PARAM_MIN = 0.2f;
    const f32 PARAM_MAX = 4.0f;

    const int REMOTE_OUTPUT_LINE_FLAG[ WPAD_MAX_CONTROLLERS ] =
    {
        nw::snd::OUTPUT_LINE_REMOTE0,
        nw::snd::OUTPUT_LINE_REMOTE1,
        nw::snd::OUTPUT_LINE_REMOTE2,
        nw::snd::OUTPUT_LINE_REMOTE3
    };

    void SpeakerSetupCallback( s32 channel, s32 result )
    {
        NW_LOG( "SpeakerSetupCallback    [channel] %d, [result] %d \n", channel, result );
    }

    void SpeakerShutdownCallback( s32 channel, s32 result )
    {
        NW_LOG( "SpeakerShutdownCallback [channel] %d, [result] %d \n", channel, result );
    }

    void ConnectCallback( s32 channel, s32 reason )
    {
        if ( reason == WPAD_ERR_NONE )
        {
            nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Initialize( SpeakerSetupCallback );
        }
        else if ( reason == WPAD_ERR_NO_CONTROLLER )
        {
            nw::snd::SoundSystem::GetRemoteSpeaker( channel ).Finalize( SpeakerShutdownCallback );
        }
    }

    void StartSound( u32 soundId, int remoteIndex )
    {
        s_SoundHandle.Stop( 0 );
        bool result = s_SoundArchivePlayer.StartSound( &s_SoundHandle, soundId ).IsSuccess();
        if ( result )
        {
            s_SoundHandle.SetOutputLine( REMOTE_OUTPUT_LINE_FLAG[ remoteIndex ] );
            s_SoundHandle.SetRemoteOutVolume( remoteIndex, s_Volume );
            s_SoundHandle.SetPitch( s_Pitch );
            NW_LOG("StartSound(%08x) remoteIndex(%d) volume(%.2f) pitch(%.2f)\n",
                soundId, remoteIndex, s_Volume, s_Pitch );
        }
        else
        {
            NW_LOG("StartSound(%08x) remoteIndex(%d) failed.\n", soundId, remoteIndex );
        }
    }

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        if ( ! s_SoundDataManager.LoadData( WSD_HIHAT_CLOSE, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(WSD_HIHAT_CLOSE) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A] Start WSD  (WSD_HIHAT_CLOSE)\n");
        NW_LOG("[+] Start WSD  (WSD_SNARE)\n");
        NW_LOG("[-] Start WSD  (WSD_HIHAT_CLOSE_REMOTE_FILTER)\n");
        NW_LOG("[B] Stop Sound\n");
        NW_LOG("[LEFT/RIGHT] Change RemoteOutVolume (0.2-4.0)\n");
        NW_LOG("[UP/DOWN]    Change Pitch (0.2-4.0)\n");
        NW_LOG("[1/2]  RemoteSpeaker ON/OFF\n");
        NW_LOG("[HOME] Exit Application\n");
        NW_LOG("---------------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        NW_UNUSED_VARIABLE(pDemo);

        static KPADStatus kpads[WPAD_MAX_CONTROLLERS];

        int readResult[WPAD_MAX_CONTROLLERS];

        for (int i = 0; i < WPAD_MAX_CONTROLLERS; i++)
        {
            KPADReadEx(i, &kpads[i], 1, &readResult[i]);
            KPADStatus& kpad = kpads[i];

            // サウンド再生・停止
            if ( kpad.trig & KPAD_BUTTON_A )
            {
                StartSound( WSD_HIHAT_CLOSE, i );
            }
            if ( kpad.trig & KPAD_BUTTON_PLUS )
            {
                StartSound( WSD_SNARE, i );
            }
            if ( kpad.trig & KPAD_BUTTON_MINUS )
            {
                StartSound( WSD_HIHAT_CLOSE_REMOTE_FILTER, i );
            }
            if ( kpad.trig & KPAD_BUTTON_B )
            {
                s_SoundHandle.Stop( 0 );
            }

            // パラメータ変更
            if ( kpad.trig & KPAD_BUTTON_LEFT )
            {
                s_Volume -= PARAM_DELTA;
                if ( s_Volume < PARAM_MIN ) { s_Volume = PARAM_MIN; }
                s_SoundHandle.SetRemoteOutVolume( i, s_Volume );
                NW_LOG("RemoteOutVolume(%.2f)\n", s_Volume);
            }
            if ( kpad.trig & KPAD_BUTTON_RIGHT )
            {
                s_Volume += PARAM_DELTA;
                if ( s_Volume > PARAM_MAX ) { s_Volume = PARAM_MAX; }
                s_SoundHandle.SetRemoteOutVolume( i, s_Volume );
                NW_LOG("RemoteOutVolume(%.2f)\n", s_Volume);
            }
            if ( kpad.trig & KPAD_BUTTON_UP )
            {
                s_Pitch += PARAM_DELTA;
                if ( s_Pitch > PARAM_MAX ) { s_Pitch = PARAM_MAX; }
                s_SoundHandle.SetPitch( s_Pitch );
                NW_LOG("Pitch(%.2f)\n", s_Pitch);
            }
            if ( kpad.trig & KPAD_BUTTON_DOWN )
            {
                s_Pitch -= PARAM_DELTA;
                if ( s_Pitch < PARAM_MIN ) { s_Pitch = PARAM_MIN; }
                s_SoundHandle.SetPitch( s_Pitch );
                NW_LOG("Pitch(%.2f)\n", s_Pitch);
            }

            if ( kpad.trig & KPAD_BUTTON_1 )
            {
                nw::snd::SoundSystem::GetRemoteSpeaker( i ).Initialize( SpeakerSetupCallback );
            }
            if ( kpad.trig & KPAD_BUTTON_2 )
            {
                nw::snd::SoundSystem::GetRemoteSpeaker( i ).Finalize( SpeakerShutdownCallback );
            }

            if ( kpad.trig & KPAD_BUTTON_HOME )
            {
                s_SoundHandle.Stop( 0 );
                return false;
            }
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void RemoteDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // KPADの初期化
        KPADInit();
        for ( int i = 0; i < WPAD_MAX_CONTROLLERS; i++ )
        {
            KPADSetConnectCallback( i, ConnectCallback );
        }

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            //pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();

        // KPAD の終了処理
        for ( int i = 0; i < WPAD_MAX_CONTROLLERS; i++ )
        {
            KPADSetConnectCallback( i, NULL );
        }
        KPADShutdown();
    }

}
