﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "externalFile.fsid"

#include <nw/demo.h>
#include <nw/snd/util/sndutil_FileReader.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "externalFile";
    const char SOUND_ARC_PATH[] = "snd/externalFile/externalFile.bfsar";
    const char EXTERNAL_BFWAR_PATH[]  = "snd/externalFile/WARC_TEST.bfwar";
    const char EXTERNAL_BFWAV_PATH[] = "snd/externalFile/yoshiPianoLoop.44.dspadpcm.bfwav";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    void* s_pMemoryForExternalBfwarFile;
    s32   s_LengthForExternalBfwarFile;

    void* s_pMemoryForExternalBfwavFile;
    s32   s_LengthForExternalBfwavFile;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(SOUND_ARC_PATH, soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void LoadExternalFiles(nw::ut::IAllocator& allocator)
    {
        nw::snd::util::FileReader fileReader;
        nw::snd::util::FileReader::Result result = fileReader.Initialize(
#if defined(NW_PLATFORM_CAFE)
            snddemo::GetFsClientPointer()
#endif
        );
        NW_ASSERT(result == nw::snd::util::FileReader::RESULT_SUCCESS);

        // ウェーブサウンドデータのみ、サウンドヒープにロード
        if ( ! s_SoundDataManager.LoadData( WSD_YOSHI, &s_SoundHeap, nw::snd::SoundDataManager::LOAD_WSD ) )
        {
            NW_ASSERTMSG( false, "LoadData(WSD_YOSHI) failed." );
        }

        // 波形アーカイブファイルをメモリにロード
        {
            fileReader.Open( EXTERNAL_BFWAR_PATH );
            s_LengthForExternalBfwarFile = fileReader.GetSize();
            s_pMemoryForExternalBfwarFile = allocator.Alloc( s_LengthForExternalBfwarFile, FS_IO_BUFFER_ALIGN );

            int readSize = 0;
            fileReader.Read( s_pMemoryForExternalBfwarFile, s_LengthForExternalBfwarFile, &readSize );
            fileReader.Close();
#if defined( NW_PLATFORM_CAFE )
            DCFlushRange( s_pMemoryForExternalBfwarFile, s_LengthForExternalBfwarFile );
#endif
        }

        // ロードした波形アーカイブファイルを SoundDataManager に登録
        {
            u32 fileId = s_SoundArchive.GetItemFileId( WARC_TEST );
            s_SoundDataManager.SetFileAddress( fileId, s_pMemoryForExternalBfwarFile );
        }

        // 外部波形ファイルをメモリにロード
        {
            fileReader.Open( EXTERNAL_BFWAV_PATH );
            s_LengthForExternalBfwavFile = fileReader.GetSize();
            s_pMemoryForExternalBfwavFile = allocator.Alloc( s_LengthForExternalBfwavFile, FS_IO_BUFFER_ALIGN );

            int readSize = 0;
            fileReader.Read( s_pMemoryForExternalBfwavFile, s_LengthForExternalBfwavFile, &readSize );
            fileReader.Close();
#if defined( NW_PLATFORM_CAFE )
            DCFlushRange( s_pMemoryForExternalBfwavFile, s_LengthForExternalBfwavFile );
#endif
        }

        fileReader.Finalize();
    }

    void UnloadExternalFiles(nw::ut::IAllocator& allocator )
    {
        allocator.Free( s_pMemoryForExternalBfwavFile );

        // SoundDataManager の登録を解除し、波形アーカイブファイルをメモリから解放
        s_SoundDataManager.InvalidateSoundData(s_pMemoryForExternalBfwarFile, s_LengthForExternalBfwarFile );
        allocator.Free( s_pMemoryForExternalBfwarFile );
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]         Play WSD_YOSHI \n");
        NW_LOG("[R + A]     Play External WSD \n");
        NW_LOG("[X]         Play WSD_WIHAHO \n");
        NW_LOG("[Y]         Play STRM_PIANO16\n");
        NW_LOG("[L + R + Y] Play External Same Track and Channel STRM \n");
        NW_LOG("[L + Y]     Play External Different Channel STRM \n");
        NW_LOG("[R + Y]     Play External Different Track STRM \n");
        NW_LOG("[B]         Stop Sound\n");
#if defined( NW_PLATFORM_CAFE )
        NW_LOG("[HOME]      Exit Application\n");
#elif defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        NW_LOG("[S]         Exit Application\n");
#endif
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );

            if ( pad->IsHold( nw::demo::Pad::MASK_R ) )
            {
                nw::snd::SoundStartable::StartInfo info;
                info.enableFlag |= nw::snd::SoundStartable::StartInfo::ENABLE_WAVE_SOUND_INFO;
                info.waveSoundInfo.waveAddress = s_pMemoryForExternalBfwavFile;
                bool result = s_SoundArchivePlayer
                  .StartSound( &s_SoundHandle, WSD_YOSHI, &info ).IsSuccess();
                NW_LOG("[WSD] StartSound(WSD_YOSHI) ... (%d)\n", result);
            }
            else
            {
                bool result = s_SoundArchivePlayer
                    .StartSound( &s_SoundHandle, WSD_YOSHI ).IsSuccess();
                NW_LOG("[WSD] StartSound(WSD_YOSHI) ... (%d)\n", result);
            }
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, WSD_WIHAHO ).IsSuccess();
            NW_LOG("[WSD] StartSound(WSD_WIHAHO) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );

            if ( pad->IsHold( nw::demo::Pad::MASK_L ) && pad->IsHold( nw::demo::Pad::MASK_R ) )
            {
                // 同チャンネル数、同トラック数の別のサウンドに差し替える
                nw::snd::SoundStartable::StartInfo info;
                info.enableFlag |= nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_INFO;
#if defined(NW_USE_NINTENDO_SDK)
                info.streamSoundInfo.externalPath = "content:snd/externalFile/stream/BASS16.dspadpcm.bfstm";
#else
                info.streamSoundInfo.externalPath = "snd/externalFile/stream/BASS16.dspadpcm.bfstm";
#endif

                bool result = s_SoundArchivePlayer
                    .StartSound( &s_SoundHandle, STRM_PIANO16, &info ).IsSuccess();
                NW_LOG("[STRM] StartSound(STRM_PIANO16) ... (%d)\n", result);
            }
            else if ( pad->IsHold( nw::demo::Pad::MASK_L ) )
            {
                // チャンネル数が異なる別のサウンドに差し替える
                nw::snd::SoundStartable::StartInfo info;
                info.enableFlag |=
                    nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_INFO |
                    nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_META_INFO;
#if defined(NW_USE_NINTENDO_SDK)
                info.streamSoundInfo.externalPath = "content:snd/externalFile/stream/kart_title.32.dspadpcm.bfstm";
#else
                info.streamSoundInfo.externalPath = "snd/externalFile/stream/kart_title.32.dspadpcm.bfstm";
#endif

                // メタ情報は既存のパラメータを流用
                s_SoundArchive.ReadStreamSoundInfo(
                    STRM_PIANO16, &info.streamSoundMetaInfo );

                // 必要なパラメータのみを変更
                info.streamSoundMetaInfo.trackInfo[0].channelCount = 2;
                info.streamSoundMetaInfo.Setup();

                bool result = s_SoundArchivePlayer
                    .StartSound( &s_SoundHandle, STRM_PIANO16, &info ).IsSuccess();
                NW_LOG("[STRM] StartSound(STRM_PIANO16) ... (%d)\n", result);
            }
            else if ( pad->IsHold( nw::demo::Pad::MASK_R ) )
            {
                // トラック数が異なる別のサウンドに差し替える
                nw::snd::SoundStartable::StartInfo info;
                info.enableFlag |=
                    nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_INFO |
                    nw::snd::SoundStartable::StartInfo::ENABLE_STRM_SOUND_META_INFO;
#if defined(NW_USE_NINTENDO_SDK)
                info.streamSoundInfo.externalPath = "content:snd/externalFile/stream/STRM_MULTITRACK.bfstm";
#else
                info.streamSoundInfo.externalPath = "snd/externalFile/stream/STRM_MULTITRACK.bfstm";
#endif

                // メタ情報は既存のパラメータを流用
                s_SoundArchive.ReadStreamSoundInfo(
                    STRM_PIANO16, &info.streamSoundMetaInfo );

                // 必要なパラメータのみを変更
                info.streamSoundMetaInfo.allocateTrackFlags = 0x7;

                // 元々のトラック情報を流用
                info.streamSoundMetaInfo.trackInfo[1] = info.streamSoundMetaInfo.trackInfo[0];
                info.streamSoundMetaInfo.trackInfo[2] = info.streamSoundMetaInfo.trackInfo[0];

                // チャンネル数を一致させる
                info.streamSoundMetaInfo.trackInfo[0].channelCount = 1;
                info.streamSoundMetaInfo.trackInfo[1].channelCount = 1;
                info.streamSoundMetaInfo.trackInfo[2].channelCount = 1;

                info.streamSoundMetaInfo.Setup();

                bool result = s_SoundArchivePlayer
                    .StartSound( &s_SoundHandle, STRM_PIANO16, &info ).IsSuccess();
                NW_LOG("[STRM] StartSound(STRM_PIANO16) ... (%d)\n", result);
            }
            else
            {
                bool result = s_SoundArchivePlayer
                    .StartSound( &s_SoundHandle, STRM_PIANO16 ).IsSuccess();
                NW_LOG("[STRM] StartSound(STRM_PIANO16) ... (%d)\n", result);
            }
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void ExternalFileDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        LoadExternalFiles(allocator);

        PrintUsage();

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo))
            {
                break;
            }
        }

        UnloadExternalFiles(allocator);

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
