﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "common.fsid"

#include <nw/demo.h>

#include "main.h"
#include "common/SampleUtility.h"
#include "common/NwSoundSetupUtility.h"

//#define CPU_RENDERING

namespace
{

    const char DEMO_TITLE[] = "effect";
    const s32 SOUND_HEAP_SIZE = 4 * 1024 * 1024;

#if defined( CPU_RENDERING )
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_PPC;
#else
    const u32 RENDERER_SELECT = AX_PB_MIXER_SELECT_DSP;
#endif

    nw::snd::FsSoundArchive     s_SoundArchive;
    nw::snd::SoundArchivePlayer s_SoundArchivePlayer;
    nw::snd::SoundDataManager   s_SoundDataManager;
    nw::snd::SoundHeap          s_SoundHeap;

    nw::snd::SoundHandle s_SoundHandle;

    enum EffectType
    {
        EFFECT_TYPE_NONE,
        EFFECT_TYPE_REVERB_STD,
        EFFECT_TYPE_REVERB_HI,
        EFFECT_TYPE_DELAY,
        EFFECT_TYPE_CHORUS,
        EFFECT_TYPE_REVERB_STD_DRC,
        EFFECT_TYPE_REVERB_HI_DRC,
        EFFECT_TYPE_DELAY_DRC,
        EFFECT_TYPE_CHORUS_DRC,
        EFFECT_TYPE_NUM,
        EFFECT_TYPE_MAX = EFFECT_TYPE_CHORUS_DRC
    };

    nw::snd::FxReverbStd                 s_FxReverbStd;
    nw::snd::FxReverbHi                  s_FxReverbHi;
    nw::snd::FxDelay                     s_FxDelay;
    nw::snd::FxChorus                    s_FxChorus;
    nw::snd::FxReverbStd::ReverbStdParam s_FxReverbStdParam;
    nw::snd::FxReverbHi::ReverbHiParam   s_FxReverbHiParam;
    nw::snd::FxDelay::DelayParam         s_FxDelayParam;
    nw::snd::FxChorus::ChorusParam       s_FxChorusParam;

    void* s_pMemoryForFxReverbStd;
    void* s_pMemoryForFxReverbHi;
    void* s_pMemoryForFxDelay;
    void* s_pMemoryForFxChorus;

    int s_EffectType, s_PreEffectType;

    void InitializeNwSound(nw::ut::IAllocator& allocator)
    {
        char soundArchivePath[512];
        snddemo::ConvertToPlatformDependentPath(snddemo::GetCommmonSoundArchivePath(), soundArchivePath);

        snddemo::InitializeSoundSystem(allocator);
        snddemo::InitializeFsSoundArchive(s_SoundArchive, soundArchivePath, allocator);
        snddemo::InitializeSoundDataManager(s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundArchivePlayer(s_SoundArchivePlayer, s_SoundDataManager, s_SoundArchive, allocator);
        snddemo::InitializeSoundHeap(s_SoundHeap, SOUND_HEAP_SIZE, allocator);

        // データロード
        if ( ! s_SoundDataManager.LoadData( SEQ_MARIOKART, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SEQ_MARIOKART) failed." );
        }
        if ( ! s_SoundDataManager.LoadData( SE_YOSHI, &s_SoundHeap ) )
        {
            NW_ASSERTMSG( false, "LoadData(SE_YOSHI) failed." );
        }
    }

    void FinalizeNwSound(nw::ut::IAllocator& allocator)
    {
        snddemo::FinalizeSoundArchivePlayer(s_SoundArchivePlayer, allocator);
        snddemo::FinalizeSoundDataManager(s_SoundDataManager, allocator);
        snddemo::FinalizeFsSoundArchive(s_SoundArchive, allocator);
        snddemo::FinalizeSoundHeap(s_SoundHeap, allocator);
        snddemo::FinalizeSoundSystem(allocator);
    }

    void InitializeEffect(int effectType, nw::ut::IAllocator& allocator)
    {
        switch ( effectType )
        {
            case EFFECT_TYPE_REVERB_STD:
                {
                    s_FxReverbStdParam.fusedGain = 0.4f;
                    s_FxReverbStd.SetParam( s_FxReverbStdParam );
                    size_t reverbBufferSize = s_FxReverbStd.GetRequiredMemSize();
                    s_pMemoryForFxReverbStd = allocator.Alloc( reverbBufferSize );
                    s_FxReverbStd.AssignWorkBuffer( s_pMemoryForFxReverbStd, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxReverbStd );
                    NW_LOG( "REVREB_STD %d\n", result );
                }
                break;
            case EFFECT_TYPE_REVERB_HI:
                {
                    s_FxReverbHiParam.fusedGain = 0.4f;
                    s_FxReverbHi.SetParam( s_FxReverbHiParam );
                    size_t reverbBufferSize = s_FxReverbHi.GetRequiredMemSize();
                    s_pMemoryForFxReverbHi = allocator.Alloc( reverbBufferSize );
                    s_FxReverbHi.AssignWorkBuffer( s_pMemoryForFxReverbHi, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxReverbHi );
                    NW_LOG( "REVREB_HI %d\n", result );
                }
                break;
            case EFFECT_TYPE_DELAY:
                {
                    s_FxDelayParam.delay = 200.f;
                    s_FxDelay.SetParam( s_FxDelayParam );
                    size_t reverbBufferSize = s_FxDelay.GetRequiredMemSize();
                    s_pMemoryForFxDelay = allocator.Alloc( reverbBufferSize );
                    s_FxDelay.AssignWorkBuffer( s_pMemoryForFxDelay, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxDelay );
                    NW_LOG( "DELAY %d\n", result );
                }
                break;
            case EFFECT_TYPE_CHORUS:
                {
                    s_FxChorus.SetParam( s_FxChorusParam );
                    size_t reverbBufferSize = s_FxChorus.GetRequiredMemSize();
                    s_pMemoryForFxChorus = allocator.Alloc( reverbBufferSize );
                    s_FxChorus.AssignWorkBuffer( s_pMemoryForFxChorus, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxChorus );
                    NW_LOG( "CHORUS %d\n", result );
                }
                break;
            case EFFECT_TYPE_REVERB_STD_DRC:
                {
                    s_FxReverbStdParam.fusedGain = 0.4f;
                    s_FxReverbStd.SetParam( s_FxReverbStdParam );
                    size_t reverbBufferSize = s_FxReverbStd.GetRequiredMemSize();
                    s_pMemoryForFxReverbStd = allocator.Alloc( reverbBufferSize );
                    s_FxReverbStd.AssignWorkBuffer( s_pMemoryForFxReverbStd, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxReverbStd, nw::snd::OUTPUT_DEVICE_DRC );
                    NW_LOG( "REVREB_STD (DRC)%d\n", result );
                }
                break;
            case EFFECT_TYPE_REVERB_HI_DRC:
                {
                    s_FxReverbHiParam.fusedGain = 0.4f;
                    s_FxReverbHi.SetParam( s_FxReverbHiParam );
                    size_t reverbBufferSize = s_FxReverbHi.GetRequiredMemSize();
                    s_pMemoryForFxReverbHi = allocator.Alloc( reverbBufferSize );
                    s_FxReverbHi.AssignWorkBuffer( s_pMemoryForFxReverbHi, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxReverbHi, nw::snd::OUTPUT_DEVICE_DRC );
                    NW_LOG( "REVREB_HI (DRC)%d\n", result );
                }
                break;
            case EFFECT_TYPE_DELAY_DRC:
                {
                    s_FxDelayParam.delay = 200.f;
                    s_FxDelay.SetParam( s_FxDelayParam );
                    size_t reverbBufferSize = s_FxDelay.GetRequiredMemSize();
                    s_pMemoryForFxDelay = allocator.Alloc( reverbBufferSize );
                    s_FxDelay.AssignWorkBuffer( s_pMemoryForFxDelay, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxDelay, nw::snd::OUTPUT_DEVICE_DRC );
                    NW_LOG( "DELAY (DRC)%d\n", result );
                }
                break;
            case EFFECT_TYPE_CHORUS_DRC:
                {
                    s_FxChorus.SetParam( s_FxChorusParam );
                    size_t reverbBufferSize = s_FxChorus.GetRequiredMemSize();
                    s_pMemoryForFxChorus = allocator.Alloc( reverbBufferSize );
                    s_FxChorus.AssignWorkBuffer( s_pMemoryForFxChorus, reverbBufferSize );
                    bool result = nw::snd::SoundSystem::AppendEffect(
                            nw::snd::AUX_BUS_A, &s_FxChorus, nw::snd::OUTPUT_DEVICE_DRC );
                    NW_LOG( "CHORUS (DRC)%d\n", result );
                }
                break;
            default:
                {
                    NW_LOG( "NONE\n");
                }
                break;
        }
    }

    void FinalizeEffect(int effectType, nw::ut::IAllocator& allocator)
    {
        switch ( effectType )
        {
            case EFFECT_TYPE_REVERB_STD:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
                s_FxReverbStd.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxReverbStd );
                break;
            case EFFECT_TYPE_REVERB_HI:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
                s_FxReverbHi.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxReverbHi );
                break;
            case EFFECT_TYPE_DELAY:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
                s_FxDelay.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxDelay );
                break;
            case EFFECT_TYPE_CHORUS:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0 );
                s_FxChorus.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxChorus );
                break;
            case EFFECT_TYPE_REVERB_STD_DRC:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
                s_FxReverbStd.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxReverbStd );
                break;
            case EFFECT_TYPE_REVERB_HI_DRC:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
                s_FxReverbHi.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxReverbHi );
                break;
            case EFFECT_TYPE_DELAY_DRC:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
                s_FxDelay.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxDelay );
                break;
            case EFFECT_TYPE_CHORUS_DRC:
                nw::snd::SoundSystem::ClearEffect( nw::snd::AUX_BUS_A, 0, nw::snd::OUTPUT_DEVICE_DRC );
                s_FxChorus.ReleaseWorkBuffer();
                allocator.Free( s_pMemoryForFxChorus );
                break;
            default:
                break;
        }
    }

    void PrintUsage()
    {
        NW_LOG("----------------------------------------\n");
        NW_LOG("NintendoWare %s Sample\n", DEMO_TITLE);
        NW_LOG("----------------------------------------\n");
        NW_LOG("[A]          Start SEQ  (SEQ_MARIOKART)\n");
        NW_LOG("[X]          Start WSD  (SE_YOSHI)\n");
        NW_LOG("[Y]          Start STRM (STRM_MARIOKART)\n");
        NW_LOG("[B]          Stop Sound\n");
        NW_LOG("[LEFT/RIGHT] Change Effect\n");
        NW_LOG("[HOME]       Exit Application\n");
        NW_LOG("---------------------------------------\n");
    }

    bool Process(nw::demo::DemoSystem* pDemo, nw::ut::IAllocator& allocator)
    {
        nw::demo::Pad* pad = pDemo->GetPad();

        // StartSound / Stop
        if ( pad->IsTrig( nw::demo::Pad::MASK_A ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SEQ_MARIOKART ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[SEQ] StartSound(SEQ_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_X ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, SE_YOSHI ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[WSD] StartSound(SE_YOSHI) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_Y ) )
        {
            s_SoundHandle.Stop( 0 );
            bool result = s_SoundArchivePlayer
                .StartSound( &s_SoundHandle, STRM_MARIOKART ).IsSuccess();
            s_SoundHandle.SetFxSend( nw::snd::AUX_BUS_A, 0.5f );
            s_SoundHandle.SetOutputLine( nw::snd::OUTPUT_LINE_MAIN | nw::snd::OUTPUT_LINE_DRC0 );
            NW_LOG("[STRM] StartSound(STRM_MARIOKART) ... (%d)\n", result);
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_B ) )
        {
            s_SoundHandle.Stop( 0 );
        }

        // エフェクトの適用
        bool isChangeEffect = false;
        if ( pad->IsTrig( nw::demo::Pad::MASK_LEFT ) )
        {
            s_PreEffectType = s_EffectType;
            s_EffectType -= 1;
            if ( s_EffectType < 0 ) {
                s_EffectType = EFFECT_TYPE_MAX;
            }
            isChangeEffect = true;
        }
        if ( pad->IsTrig( nw::demo::Pad::MASK_RIGHT ) )
        {
            s_PreEffectType = s_EffectType;
            s_EffectType += 1;
            if ( s_EffectType > EFFECT_TYPE_MAX ) {
                s_EffectType = 0;
            }
            isChangeEffect = true;
        }
        if ( isChangeEffect )
        {
            FinalizeEffect( s_PreEffectType, allocator );
            InitializeEffect( s_EffectType, allocator );
        }

        // Exit
        if ( pad->IsTrig( nw::demo::Pad::MASK_START ) )
        {
            FinalizeEffect( s_EffectType, allocator );

            return false;
        }

        s_SoundArchivePlayer.Update();

        return true;
    }
}

namespace snddemo
{

    void EffectDemo(nw::demo::DemoSystem* pDemo)
    {
        nw::demo::DefaultAllocator allocator;

        // SDK 層のサウンドの初期化
        snddemo::InitializeSdkSound(RENDERER_SELECT);

        // NW 層のサウンドの初期化
        InitializeNwSound(allocator);

        PrintUsage();

        s_EffectType = s_PreEffectType = 0;
        InitializeEffect( s_EffectType, allocator );

        // メインループ
        while ( !pDemo->IsExiting() )
        {
            snddemo::WaitForVBlank(pDemo);

            pDemo->UpdatePad();
            if (!Process(pDemo, allocator))
            {
                break;
            }
        }

        // NW 層のサウンドの終了処理
        FinalizeNwSound(allocator);

        // SDK 層のサウンドの終了処理
        snddemo::FinalizeSdkSound();
    }

}
