﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/demo.h>

#include "common/NwSoundSetupUtility.h"
#include "common/SampleUtility.h"

#if defined( NW_USE_NINTENDO_SDK )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#elif defined( NW_PLATFORM_WIN32 )
#include <Windows.h>
#endif

namespace snddemo
{
    static void* s_pMemoryForSoundSystem;
    static void* s_pMemoryForFsSoundArchive;
    static void* s_pMemoryForInfoBlock;
    static void* s_pMemoryForSoundDataManager;
    static void* s_pMemoryForSoundArchivePlayer;
    static void* s_pMemoryForStreamBuffer;
    static void* s_pMemoryForSoundHeap;

    const char COMMON_SOUND_ARCHIVE_PATH[] = "snd/common/common.bfsar";

    void InitializeSoundSystem(nw::ut::IAllocator& allocator)
    {
        nw::snd::SoundSystem::SoundSystemParam param;
        size_t workMemSize = nw::snd::SoundSystem::GetRequiredMemSize( param );
        s_pMemoryForSoundSystem = allocator.Alloc( workMemSize, 4 );

        nw::snd::SoundSystem::Initialize(
                param,
                reinterpret_cast<uptr>( s_pMemoryForSoundSystem ),
                workMemSize );
    }

    void InitializeFsSoundArchive(
        nw::snd::FsSoundArchive& soundArchive,
        const char* soundArchivePath,
        nw::ut::IAllocator& allocator)
    {
        // サウンドアーカイブの初期化
        {
#if defined(NW_PLATFORM_CAFE)
            size_t size = soundArchive.GetRequiredMemSize();
            s_pMemoryForFsSoundArchive = allocator.Alloc(size);
            if ( ! soundArchive.Open(
                snddemo::GetFsClientPointer(), soundArchivePath, s_pMemoryForFsSoundArchive, size) )
            {
                NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", soundArchivePath );
            }
#else
            if ( ! soundArchive.Open(soundArchivePath) )
            {
                NW_ASSERTMSG( 0, "cannot open bfsar(%s)\n", soundArchivePath );
            }
#endif
        }

        // INFO ブロックのロード
        {
            size_t infoBlockSize = soundArchive.GetHeaderSize();
            s_pMemoryForInfoBlock = allocator.Alloc( infoBlockSize, nw::snd::FsSoundArchive::BUFFER_ALIGN_SIZE );
            if ( ! soundArchive.LoadHeader( s_pMemoryForInfoBlock, infoBlockSize ) )
            {
                NW_ASSERTMSG( 0, "cannot load infoBlock" );
            }
        }

    }

    void InitializeSoundDataManager(
        nw::snd::SoundDataManager& soundDataManager,
        nw::snd::SoundArchive& soundArchive,
        nw::ut::IAllocator& allocator)
    {
        size_t setupSize = soundDataManager.GetRequiredMemSize( &soundArchive );
        s_pMemoryForSoundDataManager = allocator.Alloc( setupSize, 4 );
        soundDataManager.Initialize( &soundArchive, s_pMemoryForSoundDataManager, setupSize );
    }

    void InitializeSoundArchivePlayer(
        nw::snd::SoundArchivePlayer& soundArchivePlayer,
        nw::snd::SoundDataManager& soundDataManager,
        nw::snd::SoundArchive& soundArchive,
        nw::ut::IAllocator& allocator)
    {
        size_t setupSize = soundArchivePlayer.GetRequiredMemSize( &soundArchive );
        s_pMemoryForSoundArchivePlayer = allocator.Alloc( setupSize, 32 );
        size_t setupStrmBufferSize =
            soundArchivePlayer.GetRequiredStreamBufferSize( &soundArchive );
        s_pMemoryForStreamBuffer = allocator.Alloc( setupStrmBufferSize, 8 );

        bool result = soundArchivePlayer.Initialize(
                &soundArchive,
                &soundDataManager,
                s_pMemoryForSoundArchivePlayer, setupSize,
                s_pMemoryForStreamBuffer, setupStrmBufferSize );
        NW_ASSERT( result );
    }

    void InitializeSoundHeap(
        nw::snd::SoundHeap& soundHeap,
        int soundHeapSize,
        nw::ut::IAllocator& allocator)
    {
        s_pMemoryForSoundHeap = allocator.Alloc( soundHeapSize );
        bool result = soundHeap.Create( s_pMemoryForSoundHeap, soundHeapSize );
        NW_ASSERT( result );
    }

    void FinalizeSoundSystem(nw::ut::IAllocator& allocator)
    {
        nw::snd::SoundSystem::Finalize();

        allocator.Free( s_pMemoryForSoundSystem );
    }

    void FinalizeFsSoundArchive(
        nw::snd::FsSoundArchive& soundArchive,
        nw::ut::IAllocator& allocator)
    {
        soundArchive.Close();

        allocator.Free( s_pMemoryForInfoBlock );
#if defined( NW_PLATFOMR_CAFE )
        allocator.Free( s_pMemoryForFsSoundArchive );
#endif
    }

    void FinalizeSoundDataManager(
        nw::snd::SoundDataManager& soundDataManager,
        nw::ut::IAllocator& allocator)
    {
        soundDataManager.Finalize();

        allocator.Free( s_pMemoryForSoundDataManager );
    }

    void FinalizeSoundArchivePlayer(
        nw::snd::SoundArchivePlayer& soundArchivePlayer,
        nw::ut::IAllocator& allocator)
    {
        soundArchivePlayer.Finalize();

        allocator.Free( s_pMemoryForStreamBuffer );
        allocator.Free( s_pMemoryForSoundArchivePlayer );
    }

    void FinalizeSoundHeap(
        nw::snd::SoundHeap& soundHeap,
        nw::ut::IAllocator& allocator)
    {
        soundHeap.Destroy();

        allocator.Free( s_pMemoryForSoundHeap );
    }

    const char* GetCommmonSoundArchivePath()
    {
        return COMMON_SOUND_ARCHIVE_PATH;
    }
}
