﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d.h>
#include <g3ddemo_DemoUtility.h>
#include <g3ddemo_GfxUtility.h>
#include "Simple.h"
#include <nw/g3d/fnd/g3d_GLUtility.h>

namespace g3ddemo = nw::g3d::demo;

namespace {

enum UniformBlockType
{
    MATERIAL_BLOCK,
    SHAPE_BLOCK,
    SKELETON_BLOCK,
    VIEW_BLOCK
};

enum
{
    // シェーダで使用する各項目の最大数を設定します。
    // 必要数を事前に決定しておくことでロケーションを記録するメモリを節約することも可能です。
    NUM_SHADER_ATTRIB   = 8,
    NUM_SHADER_SAMPLER  = 8,
    NUM_SHADER_BLOCK    = 4,

    // モデルに使用するテクスチャのアライメントの最大値を設定します。
    // リソースファイルごとのアライメントを事前に取得しておくことで、
    // 必要なメモリのアライメントを最小限に抑えてメモリを節約することも可能です。
    TEXTURE_ALIGNMENT   = 8192
};

#define MODEL_PATH  ""
#define SHADER_PATH ""

const char* BFRES_PATH = MODEL_PATH "human.bfres";

// s_ModelName が NULL の場合はこの s_ModelIndex 番目のモデルを使用します。
const char* s_ModelName = "human";
const int s_ModelIndex = 0;

const char* s_SkeletalAnimName = "human_walk";
const int s_SkeletalAnimIndex = 0;

g3ddemo::SimpleShader::Path s_ShaderPath = {
    NULL,
    SHADER_PATH "simple_vertex.glsl",
    SHADER_PATH "simple_fragment.glsl",
};

// g3d ではシェーダに依存しない頂点属性やサンプラーの識別文字列を名前と呼びます。
// 同様にシェーダに依存するサンプラーや Uniform の識別文字列を ID と呼びます。

// 使用する頂点属性の名前を列挙します。
// ここでは配列中のインデックスを explicit attribute location と一致させています。
const char* s_AttribName[] = { "_p0", "_n0", "_u0", "_i0", "_w0" };

// 使用するサンプラーの名前と ID を列挙します。
// 事前にシェーダを関連付けたマテリアルは ResShaderAssign に名前と ID の関連付けを持っています。
const char* s_SamplerName[] = { "_a0" };
const char* s_SamplerId[] = { "albedoTex" };

// Uniform Block の ID と Uniform の ID を列挙します。
const char* s_BlockId[] = { "Material", "Shape", "Skeleton", "View" };
const char* s_UniformId[] = { "offset" };

const int s_NumAttrib = sizeof(s_AttribName) / sizeof(char*);
const int s_NumSampler = sizeof(s_SamplerName) / sizeof(char*);
const int s_NumBlock = sizeof(s_BlockId) / sizeof(char*);
const int s_NumUniform = sizeof(s_UniformId) / sizeof(char*);

//--------------------------------------------------------------------------------------------------

struct SamplerTable
{
    s8 index[NUM_SHADER_SAMPLER];
};

struct ViewBlock
{
    nw::g3d::Mtx44 projMtx;
    nw::g3d::Mtx34 cameraMtx;
};

// SimpleShader を継承して、サンプラーと Uniform Block のロケーションキャッシュ機能を追加します。
class SimpleShaderEx : public g3ddemo::SimpleShader
{
public:
    struct Location
    {
        s8 vertex;
        s8 geometry;
        s8 fragment;
        u8 reserved;
    };

    SimpleShaderEx() : SimpleShader(), m_NumSampler(0), m_NumBlock(0) {}

    void InitSamplerLocations(int numSampler, const char* name[]);
    void InitBlockLocations(int numBlock, const char* name[]);

    const Location& GetSamplerLocation(int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_NumSampler);
        return m_SamplerLocation[index];
    }

    const Location& GetBlockLocation(int index) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS(index, m_NumBlock);
        return m_BlockLocation[index];
    }

private:
    u8 m_NumSampler;
    u8 m_NumBlock;
    u16 reserved;
    Location m_SamplerLocation[NUM_SHADER_SAMPLER];
    Location m_BlockLocation[NUM_SHADER_BLOCK];
};

void SimpleShaderEx::InitSamplerLocations(int numSampler, const char* name[])
{
    this->m_NumSampler = static_cast<u8>(numSampler);
    memset(m_SamplerLocation, -1, sizeof(m_SamplerLocation));
    for (int idxSampler = 0; idxSampler < numSampler; ++idxSampler)
    {
        const char* samplerName = name[idxSampler];
        Location& location = m_SamplerLocation[idxSampler];
        location.vertex = GetVertexSamplerLocation(samplerName);
        location.geometry = GetGeometrySamplerLocation(samplerName);
        location.fragment = GetFragmentSamplerLocation(samplerName);
    }
}

void SimpleShaderEx::InitBlockLocations(int numBlock, const char* name[])
{
    this->m_NumBlock = static_cast<u8>(numBlock);
    memset(m_BlockLocation, -1, sizeof(m_BlockLocation));
    for (int idxBlock = 0; idxBlock < numBlock; ++idxBlock)
    {
        const char* blockName = name[idxBlock];
        Location& location = m_BlockLocation[idxBlock];
        location.vertex = GetVertexBlockLocation(blockName);
        location.geometry = GetGeometryBlockLocation(blockName);
        location.fragment = GetFragmentBlockLocation(blockName);
    }
}

u32 counter = 0;

nw::g3d::ModelObj modelObj;
nw::g3d::SkeletalAnimObj skeletalAnimObj;
nw::g3d::GfxBuffer viewBlock;
g3ddemo::FrameBuffer frameBufferTV;
nw::g3d::GfxFetchShader* pFetchShaderArray = NULL;
SimpleShaderEx shader;
SamplerTable* pSamplerTableArray = NULL;
void* pFetchShaderBuffer = NULL;
nw::g3d::ResModel* pResModel = NULL;
nw::g3d::ResFile* pResFile = NULL;
bool initialized = false;

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

void SimpleCalc()
{
    if (initialized == false)
    {
        return;
    }

    // 投影行列。
    nw::g3d::Mtx44 projMtx;
    projMtx.Perspective(nw::g3d::Math::DegToRad(45.0f), 16.0f / 9.0f, 0.01f, 1000.0f);

    // カメラ行列。
    nw::g3d::Mtx34 cameraMtx;
    cameraMtx.LookAt(
        nw::g3d::Vec3::Make(0.0f, 2.5f, 7.5f),
        nw::g3d::Vec3::Make(0.0f, 1.0f, 0.0f),
        nw::g3d::Vec3::Make(0.0f, 2.5f, 0.0f));

    // スケルタルアニメーション。
    if (skeletalAnimObj.GetResource())
    {
        skeletalAnimObj.Calc();
        skeletalAnimObj.ApplyTo(&modelObj);
        skeletalAnimObj.GetFrameCtrl().UpdateFrame();
    }

    // ベース行列。
    nw::g3d::Mtx34 baseMtx;
    baseMtx.SetR(nw::g3d::Vec3::Make(0.0f, counter++ * 3.14f * 2 / 256, 0.0f));
    baseMtx.SetT(nw::g3d::Vec3::Make(0.0f, 0.0f, 0.0f));
    modelObj.CalcWorld(baseMtx);

    // 残っているコマンドの完了を待ちます。
    // CPU と GPU を直列実行している場合、ここでは GPU は完了しているはず。
    //nw::g3d::GfxManage::DrawDone();

    // GPU 待ちが必要な計算
    modelObj.CalcMtxBlock();
    modelObj.CalcMaterial();
    modelObj.CalcShape();
    modelObj.CalcView(0, cameraMtx);

    // モデルのマテリアルに混載した領域にプログラムから値を差し込む場合、
    // 所定のエンディアンで書き込んだ上で CPU キャッシュを吐き出します。
    for (int idxMat = 0, numMat = modelObj.GetMaterialCount(); idxMat < numMat; ++idxMat)
    {
        nw::g3d::MaterialObj* pMaterialObj = modelObj.GetMaterial(idxMat);
        nw::g3d::GfxBuffer& matBlock = pMaterialObj->GetMatBlock();
        float colorOffset[] = { 0.0f, 0.0f, 0.0f };
        nw::g3d::Copy32<true>(matBlock.GetData(), colorOffset, sizeof(colorOffset) >> 2);
        matBlock.DCFlush();
    }

    // ビュー用の Uniform Block の更新。
    ViewBlock* pViewBlockBuffer = static_cast<ViewBlock*>(viewBlock.GetData());
    nw::g3d::Copy32<true>(&pViewBlockBuffer->projMtx, &projMtx, sizeof(projMtx) >> 2);
    nw::g3d::Copy32<true>(&pViewBlockBuffer->cameraMtx, &cameraMtx, sizeof(cameraMtx) >> 2);
    viewBlock.DCFlush();

    nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。
    // GPU のリードキャッシュはここで一括して Invalidate します。
    nw::g3d::GPUCache::InvalidateAll();
}

void SimpleDraw()
{
    if (initialized == false)
    {
        return;
    }

    nw::g3d::GfxColorBuffer& colorBufferTV =
        frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;

    nw::g3d::GfxDepthBuffer& depthBufferTV = frameBufferTV.GetDepthBufferTexture()->renderBuffer;

    // TV 描画。
    nw::g3d::ClearBuffers(&colorBufferTV, &depthBufferTV,
        0.0f, 0.0f, 0.0f, 1.0f, GX2_CLEAR_BOTH);

    frameBufferTV.Load();

    int viewIndex = 0; // ビューの番号です。

    // g3d は Uniform Block を前提で設計されていますが、
    // ModelObj の DisableBlockSwapAll() を呼ぶことによって
    // レジスタモードでは不要なエンディアンスワップを無効化することもできます。
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);

    const nw::g3d::SkeletonObj* pSkeletonObj = modelObj.GetSkeleton();

    // ここではソートやカリング、同一の描画ステートの省略などは行っていません。
    // アプリケーションの仕様にとって最適な方法で描画処理を行うことを推奨します。
    for (int idxShape = 0, numShape = modelObj.GetShapeCount();
        idxShape < numShape; ++idxShape)
    {
        const nw::g3d::ShapeObj* pShapeObj = modelObj.GetShape(idxShape);

        if (!modelObj.IsShapeVisible(idxShape)) // ビジビリティの制御を行います。
        {
            continue;
        }

        int idxMaterial = pShapeObj->GetMaterialIndex();
        const nw::g3d::MaterialObj* pMaterialObj = modelObj.GetMaterial(idxMaterial);
        int idxVertex = pShapeObj->GetVertexIndex();
        const nw::g3d::GfxFetchShader* pFetchShader = &pFetchShaderArray[idxVertex];

        // 描画ステートのロード。
        const nw::g3d::ResRenderState* pRenderState = pMaterialObj->GetResRenderState();
        pRenderState->Load();

        // フェッチシェーダと頂点バッファのロード。
        pFetchShader->Load();
        for (int idxVtxBuf = 0, numVtxBuffer = pShapeObj->GetVertexBufferCount();
                idxVtxBuf < numVtxBuffer; ++idxVtxBuf)
        {
            const nw::g3d::GfxBuffer& vtxBuf = pShapeObj->GetVertexBuffer(idxVtxBuf);
            vtxBuf.LoadVertices(idxVtxBuf);
        }

        // シェーダのロード。
        shader.Load();

        // マテリアルの Uniform Block をロード。
        // ここではジオメトリシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& matBlock = pMaterialObj->GetMatBlock();
        if (matBlock.GetData())
        {
            const SimpleShaderEx::Location& location =
                shader.GetBlockLocation(MATERIAL_BLOCK);
            if (location.vertex != -1)
            {
                matBlock.LoadVertexUniforms(location.vertex);
            }
            if (location.fragment != -1)
            {
                matBlock.LoadFragmentUniforms(location.fragment);
            }
        }
        // シェイプの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& shpBlock = pShapeObj->GetShpBlock(viewIndex);
        NW_G3D_ASSERT_NOT_NULL(shpBlock.GetData());
        {
            const SimpleShaderEx::Location& location =
                    shader.GetBlockLocation(SHAPE_BLOCK);
            if (location.vertex != -1)
            {
                shpBlock.LoadVertexUniforms(location.vertex);
            }
        }
        // スケルトンの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        const nw::g3d::GfxBuffer& mtxBlock = pSkeletonObj->GetMtxBlock();
        if (mtxBlock.GetData())
        {
            const SimpleShaderEx::Location& location =
                shader.GetBlockLocation(SKELETON_BLOCK);
            if (location.vertex != -1)
            {
                mtxBlock.LoadVertexUniforms(location.vertex);
            }
        }
        // ビューの Uniform Block をロード。
        // ここではジオメトリシェーダとフラグメントシェーダの処理を省略しています。
        {
            const SimpleShaderEx::Location& location =
                shader.GetBlockLocation(VIEW_BLOCK);
            if (location.vertex != -1)
            {
                viewBlock.LoadVertexUniforms(location.vertex);
            }
        }

        // サンプラーのロード。
        // ここでは頂点シェーダとジオメトリシェーダの処理を省略しています。
        SamplerTable* pSamplerTable = &pSamplerTableArray[idxMaterial];
        for (int idxTex = 0; idxTex < s_NumSampler; ++idxTex)
        {
            const SimpleShaderEx::Location& location = shader.GetSamplerLocation(idxTex);
            int idxSampler = pSamplerTable->index[idxTex];
            if (location.fragment != -1 && idxSampler != -1)
            {
                const nw::g3d::ResSampler* pSampler = pMaterialObj->GetResSampler(idxSampler);
                const nw::g3d::ResTexture* pTexture = pMaterialObj->GetResTexture(idxSampler);
                pSampler->GetGfxSampler()->LoadFragmentSampler(location.fragment);
                pTexture->GetGfxTexture()->LoadFragmentTexture(location.fragment);
            }
        }

        // 描画コール。
        pShapeObj->GetResMesh()->Draw();
    }
}

void SimpleCopyOut(int screenWidth, int screenHeight)
{
    nw::g3d::GfxColorBuffer& colorBufferTV =
    frameBufferTV.GetColorBufferTexture(GX2_RENDER_TARGET_0)->renderBuffer;

    g3ddemo::CopyOut(&colorBufferTV, screenWidth, screenHeight);
    nw::g3d::GfxManage::DrawDone();
}

void SimpleInit(const char* resPath, const char* vshPath, const char* fshPath)
{
    if (initialized == true)
    {
        return;
    }

    initialized = true;
    // SystemContextの初期化を行います。
    nw::g3d::fnd::detail::SystemContext &system = nw::g3d::fnd::detail::GetSystemContext();
    system.Setup();
    s_ShaderPath.vertexShader = vshPath;
    s_ShaderPath.fragmentShader = fshPath;
    BFRES_PATH = resPath;

    g3ddemo::InitDisplayArg initDisplayArg;
    initDisplayArg.modeDRC = GX2_DRC_NONE;
    //initDisplayArg.modeTV = GX2_TV_RENDER_480_WIDE;
    initDisplayArg.modeTV = GX2_TV_RENDER_1080;
    g3ddemo::InitDisplay(initDisplayArg);

    // フレームバッファの初期化処理。
    {
        g3ddemo::FrameBuffer::InitArg initArg(854, 480);
        initArg.colorBufferFTV = true;
        size_t bufferSize = g3ddemo::FrameBuffer::CalcSize(initArg);
        frameBufferTV.Init(initArg, g3ddemo::AllocMem2(bufferSize), bufferSize);
        frameBufferTV.Setup();
        frameBufferTV.Alloc(g3ddemo::AllocMem1);
    }

    // シェーダの初期化。
    // シェーダはモデルに依存しないため、モデル間で共有することができます。
    {
        shader.Setup(s_ShaderPath);
        // バリエーションごとにロケーションが変わる可能性があるため、
        // ロケーションを事前にテーブル化してキャッシュしておきます。
        shader.InitSamplerLocations(s_NumSampler, s_SamplerId);
        shader.InitBlockLocations(s_NumBlock, s_BlockId);
    }

    // モデルリソースの初期化。
    {
        // ResFile のアライメントは内部に持っているテクスチャの最大アライメントに一致します。
        void* pFile = g3ddemo::LoadFile(BFRES_PATH, NULL, TEXTURE_ALIGNMENT);
        NW_G3D_ASSERT(nw::g3d::ResFile::IsValid(pFile));
        pResFile = nw::g3d::ResFile::ResCast(pFile);
        pResFile->Setup();
        // ResFile 内のテクスチャや頂点を CPU で書き込んだ場合は
        // CPU のキャッシュを吐き出す必要があります。
    }

    if (s_ModelName)
    {
        pResModel = pResFile->GetModel(s_ModelName);
    }
    else
    {
        pResModel = pResFile->GetModel(s_ModelIndex);
    }
    NW_G3D_ASSERT_NOT_NULL(pResModel);

    // フェッチシェーダの構築。
    // フェッチシェーダはモデルインスタンスに依存しないので、モデルリソース毎に構築します。
    {
        // フェッチシェーダは ResVertex と1対1で対応します。フェッチシェーダの数を減らすため、
        // 複数の ResShape が ResVertex を共有している場合があります。
        int numVertex = pResModel->GetVertexCount();
        size_t fetchShaderArraySize = sizeof(nw::g3d::GfxFetchShader) * numVertex;
        pFetchShaderArray = g3ddemo::AllocMem2<nw::g3d::GfxFetchShader>(fetchShaderArraySize);
        NW_G3D_ASSERT_NOT_NULL(pFetchShaderArray);

        // 単純化のため、s_NumAttrib 分のバッファをアライメントで切り上げて確保します。
        // 個々の ResVertex が使用する頂点属性数から正確にバッファサイズを計算することで
        // 必要なメモリを節約することもできます。
        nw::g3d::GfxFetchShader fetchShader;
        fetchShader.SetAttribCount(s_NumAttrib);
        fetchShader.CalcSize();
        // フェッチシェーダのバッファは GPU から参照されるため、所定のアライメントが必要です。
        size_t shaderSize = nw::g3d::Align(fetchShader.GetShaderSize(), GX2_SHADER_ALIGNMENT);
        void* shaderPtr = g3ddemo::AllocMem2(shaderSize * numVertex, GX2_SHADER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(shaderPtr);
        pFetchShaderBuffer = shaderPtr;

        for (int idxVertex = 0; idxVertex < numVertex; ++idxVertex)
        {
            // 存在する頂点属性の数を調べます。正しい位置に終端インストラクションを書き込むため、
            // ここでは正確な頂点属性数を指定する必要があります。
            nw::g3d::ResVertex* pResVertex = pResModel->GetVertex(idxVertex);
            nw::g3d::ResVtxAttrib* pResVtxAttrib[NUM_SHADER_ATTRIB];
            memset(pResVtxAttrib, 0, sizeof(pResVtxAttrib));
            int numAttribValid = 0;
            for (int idxAttrib = 0; idxAttrib < s_NumAttrib; ++idxAttrib)
            {
                const char* name = s_AttribName[idxAttrib];
                pResVtxAttrib[idxAttrib] = pResVertex->GetVtxAttrib(name);
                if (pResVtxAttrib[idxAttrib])
                {
                    ++numAttribValid;
                }
            }

            // シェーダバッファの設定。
            nw::g3d::GfxFetchShader* pFetchShader = &pFetchShaderArray[idxVertex];
            new(pFetchShader) nw::g3d::GfxFetchShader();
            pFetchShader->Setup();
            pFetchShader->SetAttribCount(numAttribValid);
            pFetchShader->CalcSize();
            pFetchShader->SetShaderPtr(shaderPtr);
            pFetchShader->SetDefault();
            shaderPtr = nw::g3d::AddOffset(shaderPtr, shaderSize);

            // 個別の頂点属性の設定。
            numAttribValid = 0;
            for (int idxAttrib = 0; idxAttrib < s_NumAttrib; ++idxAttrib)
            {
                nw::g3d::ResVtxAttrib* pAttrib = pResVtxAttrib[idxAttrib];
                if (pAttrib)
                {
                    int index = numAttribValid++;
                    int bufferIndex = pAttrib->GetBufferIndex();
                    // ここでは単純化のため、 bufferIndex が GX2_MAX_ATTRIB_BUFFERS より
                    // 小さいという制約を課します。GX2_MAX_ATTRIB_BUFFERS を超える場合は
                    // 空いているインデックスに詰める処理を実装する必要があります。
                    NW_G3D_ASSERT(bufferIndex < GX2_MAX_ATTRIB_BUFFERS);
                    nw::g3d::ResBuffer* pResBuffer = pResVertex->GetVtxBuffer(bufferIndex);

                    pFetchShader->SetLocation(index, idxAttrib);
                    pFetchShader->SetBufferSlot(index, bufferIndex);
                    pFetchShader->SetFormat(index, pAttrib->GetFormat());
                    pFetchShader->SetOffset(index, pAttrib->GetOffset());
                    pFetchShader->SetVertexBuffer(index, pResBuffer->GetGfxBuffer());
                }
            }
            pFetchShader->DCFlush(); // GPU が参照するメモリの CPU キャッシュを吐き出します。
        }
    }

    // マテリアルにシェーダを関連付けます。
    // フェッチシェーダと同じくモデルインスタンスに依存しないので、モデルリソース毎に構築します。
    {
        // マテリアル毎にサンプラーのインデックスをキャッシュします。
        int numMat = pResModel->GetMaterialCount();
        size_t samplerTableArraySize = sizeof(SamplerTable) * numMat;
        pSamplerTableArray = g3ddemo::AllocMem2<SamplerTable>(samplerTableArraySize);
        NW_G3D_ASSERT_NOT_NULL(pSamplerTableArray);

        for (int idxMat = 0, numMat = pResModel->GetMaterialCount(); idxMat < numMat; ++idxMat)
        {
            nw::g3d::ResMaterial* pResMaterial = pResModel->GetMaterial(idxMat);

            // SamplerTable の初期化します。
            SamplerTable* pSamplerTable = &pSamplerTableArray[idxMat];
            memset(pSamplerTable->index, -1, sizeof(pSamplerTable->index));
            for (int idxTex = 0; idxTex < s_NumSampler; ++idxTex)
            {
                const char* name = s_SamplerName[idxTex];
                pSamplerTable->index[idxTex] = static_cast<s8>(pResMaterial->GetSamplerIndex(name));
            }

            // Uniform Block のサイズを指定します。
            u32 blockSize = shader.GetFragmentBlockSize(s_BlockId[MATERIAL_BLOCK]);
            pResMaterial->SetRawParamSize(blockSize);

            // Uniform ごとに UniformBlock 内のオフセットを指定します。
            for (int idxParam = 0; idxParam < s_NumUniform; ++idxParam)
            {
                const char* id = s_UniformId[idxParam];
                nw::g3d::ResShaderParam* pResShaderParam = pResMaterial->GetShaderParam(id);
                if (pResShaderParam)
                {
                    s32 offset = shader.GetFragmentUniformOffset(id);
                    if (offset >= 0)
                    {
                        pResShaderParam->SetOffset(offset);
                    }
                }
            }
        }
    }

    // モデルインスタンスの構築。
    {
        nw::g3d::ModelObj::InitArg arg(pResModel);
        size_t bufferSize = nw::g3d::ModelObj::CalcBufferSize(arg);
        void* pBuffer = g3ddemo::AllocMem2(bufferSize, nw::g3d::ModelObj::BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        bool success = modelObj.Init(arg, pBuffer, bufferSize);
        NW_G3D_ASSERT(success);

        // Uniform Block のバッファは所定のアライメントが必要です。
        size_t blockBufferSize = modelObj.CalcBlockBufferSize();
        void* pBlockBuffer = g3ddemo::AllocMem2(blockBufferSize,
            nw::g3d::ModelObj::BLOCK_BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(pBlockBuffer);
        success = modelObj.SetupBlockBuffer(pBlockBuffer, blockBufferSize);
        NW_G3D_ASSERT(success);
    }

    // スケルタルアニメの構築。
    nw::g3d::ResSkeletalAnim* pResSkeletalAnim = NULL;
    if (s_SkeletalAnimName)
    {
        pResSkeletalAnim = pResFile->GetSkeletalAnim(s_SkeletalAnimName);
    }
    else
    {
        pResSkeletalAnim = pResFile->GetSkeletalAnim(s_SkeletalAnimIndex);
    }

    if (pResSkeletalAnim)
    {
        nw::g3d::SkeletalAnimObj::InitArg arg;
        arg.Reserve(pResModel);
        arg.Reserve(pResSkeletalAnim);
        size_t bufferSize = nw::g3d::SkeletalAnimObj::CalcBufferSize(arg);
        void* pBuffer = g3ddemo::AllocMem2(bufferSize, nw::g3d::SkeletalAnimObj::BUFFER_ALIGNMENT);
        NW_G3D_ASSERT_NOT_NULL(pBuffer);
        bool success = skeletalAnimObj.Init(arg, pBuffer, bufferSize);
        (void)success;
        NW_G3D_ASSERT(success);

        // リソースの設定を行います。再設定可能です。
        skeletalAnimObj.SetResource(pResSkeletalAnim);
        // モデルへの関連付けを行います。
        skeletalAnimObj.Bind(&modelObj);
    }

    // ビューの Uniform Block
    // モデルに依存しないので通常はビューごとに1つ使用します。
    {
        void* pViewBlockBuffer = g3ddemo::AllocMem2(
            sizeof(ViewBlock), GX2_UNIFORM_BLOCK_ALIGNMENT);
        viewBlock.SetData(pViewBlockBuffer, sizeof(ViewBlock));
        viewBlock.Setup();
    }
}

void SimpleDestroy(void)
{
    if (initialized == false)
    {
        return;
    }

    initialized = false;

    if (void* pViewBlockBuffer = viewBlock.GetData())
    {
        viewBlock.Cleanup();
        g3ddemo::FreeMem2(pViewBlockBuffer);
    }

    // スケルタルアニメの破棄。
    if (void* pBuffer = skeletalAnimObj.GetBufferPtr())
    {
        g3ddemo::FreeMem2(pBuffer);
    }

    // モデルインスタンスの破棄。
    if (void* pBlockBuffer = modelObj.GetBlockBufferPtr())
    {
        modelObj.CleanupBlockBuffer();
        g3ddemo::FreeMem2(pBlockBuffer);
    }
    if (void* pBuffer = modelObj.GetBufferPtr())
    {
        g3ddemo::FreeMem2(pBuffer);
    }

    // サンプラーのインデック情報を破棄。
    if (pSamplerTableArray)
    {
        g3ddemo::FreeMem2(pSamplerTableArray);
        pSamplerTableArray = NULL;
    }

    // フェッチシェーダの破棄。
    if (pFetchShaderArray)
    {
        for (int idxVertex = 0, numVertex = pResModel->GetVertexCount();
            idxVertex < numVertex; ++idxVertex)
        {
            nw::g3d::GfxFetchShader* pFetchShader = &pFetchShaderArray[idxVertex];
            pFetchShader->Cleanup();
        }
        g3ddemo::FreeMem2(pFetchShaderArray);
        pFetchShaderArray = NULL;
    }
    if (pFetchShaderBuffer)
    {
        g3ddemo::FreeMem2(pFetchShaderBuffer);
        pFetchShaderBuffer = NULL;
    }

    // モデルリソースの破棄。
    {
        pResFile->Cleanup();
        g3ddemo::FreeMem2(pResFile);
        pResFile = NULL;
    }

    // シェーダの破棄。
    shader.Cleanup();

    // フレームバッファの破棄
    frameBufferTV.Free(g3ddemo::FreeMem1);
    frameBufferTV.Cleanup();
    g3ddemo::FreeMem2(frameBufferTV.GetBufferPtr());

    return;
}
